<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use GuzzleHttp\Promise;
use GuzzleHttp\Promise\Create;
use GuzzleHttp\Promise\Utils;
use Illuminate\Support\Str;
use GuzzleHttp\Exception\RequestException;

class BusinessCentralService
{
    public function getAccessToken()
    {
        $client = new Client();
        $response = $client->post("https://login.microsoftonline.com/".env('AZURE_TENANT_ID')."/oauth2/v2.0/token", [
            'form_params' => [
                'grant_type' => 'client_credentials',
                'client_id' => env('AZURE_CLIENT_ID'),
                'client_secret' => env('AZURE_CLIENT_SECRET'),
                'scope' => 'https://api.businesscentral.dynamics.com/.default'
            ]
        ]);
        $body = json_decode((string)$response->getBody(), true);

        return $body['access_token'];
    }

    public function getCompanyId($token)
    {
        $client = new Client();
        $response = $client->get("https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/api/v2.0/companies", [
            'headers' => [
                'Authorization' => "Bearer {$token}",
                'Accept'        => 'application/json'
            ]
        ]);
        $companies = json_decode((string)$response->getBody(), true);
        $role = session('user')['role'] ?? null; 
        $companyId1 = $companies['value'][0]['id'];
        $companyId2 = $companies['value'][1]['id'];

        return [
            'companyId1' => $companyId1,
            'companyId2' => $companyId2
        ];
    }

    public function __construct()
    {
     $this->token = $this->getAccessToken();
     $companyIds = $this->getCompanyId($this->token);
     $this->companyRegent = $companyIds['companyId1'];
     $this->companyHIN    = $companyIds['companyId2'];
     
     $activeCompany = session('current_company_name') ?? session('user')['role'];
    if ($activeCompany === 'Regent' or $activeCompany === 'SUPER') {
        $this->companyId = $this->companyRegent;
    } elseif ($activeCompany === 'HIN' ) {
        $this->companyId = $this->companyHIN;
    }
     $this->client = new Client();
    }

    
    function getMetadata() {

        $customersResponse = $this->client->get("https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/\$metadata", [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer' => 'odata.maxpagesize=1000'
            ]
        ]);
        $xml = simplexml_load_string((string)$customersResponse->getBody());
        $xml->registerXPathNamespace('edmx', 'http://docs.oasis-open.org/odata/ns/edmx'); 
        $xml->registerXPathNamespace('edm', 'http://docs.oasis-open.org/odata/ns/edm');
        $schemas = $xml->xpath('//edm:Schema');
        
        $entities = [];

        foreach ($schemas as $schema) {
            foreach ($schema->EntityType as $entity) {
                $entityName = (string) $entity['Name'];
                $properties = [];

                foreach ($entity->Property as $prop) {
                    $properties[] = [
                        'name' => (string) $prop['Name'],
                        'type' => (string) $prop['Type'],
                        'nullable' => isset($prop['Nullable']) ? (string) $prop['Nullable'] : 'true',
                    ];
                }

                $entities[$entityName] = $properties;
            }
        }
        
        if (isset($entities['purchaseOrderLine'])) {
        $logData = [];

        foreach ($entities['purchaseOrderLine'] as $field) {
            $logData[] = [
                'name'     => $field['name'],
                'type'     => $field['type'],
                'nullable' => $field['nullable'],
            ];
        }

        Log::info('📦 Entity: purchaseOrderLine', $logData);
    } else {
        Log::warning('Entity purchaseOrderLine tidak ditemukan di metadata');
    }

    return $schemas;

        echo "<pre>";
        echo "ðŸ“¦ Entity: purchaseOrderLine\n\n";

        foreach ($entities['purchaseOrderLine'] as $field) {
            echo " â–¸ {$field['name']} ({$field['type']})";
            echo $field['nullable'] === 'false' ? " [NOT NULL]" : "";
            echo "\n";
        }
        echo "</pre>";
        exit;
    }

      public function getData(string $endpoint, array $queryParams = [])
{
    $url = "https://api.businesscentral.dynamics.com/v2.0/"
        . env('AZURE_TENANT_ID') . "/"
        . env('BC_ENVIRONMENT')
        . "/api/v2.0/companies({$this->companyId})/{$endpoint}";

    if (!empty($queryParams)) {
        $url .= '?' . http_build_query($queryParams);
    }

    try {
        $response = $this->client->get($url, [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept' => 'application/json',
            ]
        ]);

        $data = json_decode((string)$response->getBody(), true);
        $items = $data['value'] ?? [];

        // === Group dan sort otomatis hanya jika endpoint = 'items' ===
        if ($endpoint === 'items') {
            $groupedItems = [];
            $itemsWithoutPrefix = [];

            foreach ($items as $item) {
                $number = $item['number'] ?? '';
                if (preg_match('/^[A-Z]+/i', $number, $matches)) {
                    $prefix = strtoupper($matches[0]);
                    if (!isset($groupedItems[$prefix])) {
                        $groupedItems[$prefix] = [];
                    }
                    $groupedItems[$prefix][] = $item;
                } else {
                    $itemsWithoutPrefix[] = $item;
                }
            }

            if (!empty($itemsWithoutPrefix)) {
                $groupedItems['OTHER'] = $itemsWithoutPrefix;
            }

            // Sort setiap grup berdasarkan angka di number
            foreach ($groupedItems as $prefix => &$group) {
                usort($group, function($a, $b) {
                    $numA = (int) preg_replace('/\D/', '', $a['number'] ?? 0);
                    $numB = (int) preg_replace('/\D/', '', $b['number'] ?? 0);
                    return $numA <=> $numB;
                });
            }
            unset($group);

            // Ganti data['value'] menjadi grouped & sorted
            $data['value'] = $groupedItems;
        }

        return [
            'status' => 'success',
            'data' => $data
        ];
    } catch (\Exception $e) {
        return [
            'status' => 'error',
            'message' => $e->getMessage()
        ];
    }
}



    public function getAllPurchaseQtys()
    {
        $start = microtime(true);
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/Company('" . $this->companyId . "')/Purchase_Lines_Excel?\$filter=Outstanding_Quantity gt 0 and (Status eq 'Released' or Status eq 'Pending Approval' or Status eq 'Pending Prepayment')  and (Location_Code eq 'CI.1010') &\$select=No,Location_Code,Outstanding_Quantity";

        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json'
            ]
        ];

        $allData = [];
        do {
            $response = $this->client->get($url, $headers);
            $decoded = json_decode((string)$response->getBody(), true);
            if (isset($decoded['value'])) {
                $allData = array_merge($allData, $decoded['value']);
            }
            $url = $decoded['@odata.nextLink'] ?? null;
        } while ($url);

        $result = [];
        foreach ($allData as $line) {
            $key = $line['No'] . '_' . $line['Location_Code'];
            $result[$key] = ($result[$key] ?? 0) + ($line['Outstanding_Quantity'] ?? 0);
        }
        $elapsed = microtime(true) - $start;
        Log::info("Fetch runtime Purchaseqty: {$elapsed} seconds");
        

        return $result;
    }

    public function getAllTransferQtys()
    {
        $start = microtime(true);
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/Company('" . $this->companyId . "')/Transfer_Order_Line_Excel?\$filter=Status eq 'Released' and Quantity_Shipped lt Quantity and (Transfer_From_Code eq 'CI.1010'
            ) &\$select=Document_No,Item_No,Transfer_From_Code,OutstandingShipped";

        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer'        => 'odata.maxpagesize=1000'
            ]
        ];

        $allData = [];
        do {
            $response = $this->client->get($url, $headers);
            $decoded = json_decode((string)$response->getBody(), true);
            if (isset($decoded['value'])) {
                $allData = array_merge($allData, $decoded['value']);
            }
            $url = $decoded['@odata.nextLink'] ?? null;
        } while ($url);

        $result = [];
        foreach ($allData as $line) {
            $key = $line['Item_No'] . '_' . $line['Transfer_from_Code'];
            $result[$key] = ($result[$key] ?? 0) + ($line['OutstandingShipped'] ?? 0);
        }

        $elapsed = microtime(true) - $start;
        Log::info("Fetch runtime Transfer Qty: {$elapsed} seconds");
        return $result;
    }

    public function getTransferLinesFromBC()
    {
        $start = microtime(true);
        $baseUrl = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/Company('" . $this->companyId . "')/Transfer_Order_Line_Excel?\$filter=Status eq 'Released' and Transfer_from_Code eq 'CI.1010'";

        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer'        => 'odata.maxpagesize=5000',
            ]
        ];

        $allData = [];
        $url = $baseUrl;

        do {
            $response = $this->client->get($url, $headers);
            $decoded = json_decode((string)$response->getBody(), true);

            if (isset($decoded['value'])) {
                $allData = array_merge($allData, $decoded['value']);
            }

            $url = $decoded['@odata.nextLink'] ?? null;
        } while ($url);
        $elapsed = microtime(true) - $start;
        Log::info("Fetch runtime Transfer Qty: {$elapsed} seconds");
        return $allData;
    }


    function getPurchaseLineFromBC() {
    $url = "https://api.businesscentral.dynamics.com/v2.0/"
         . env('AZURE_TENANT_ID') . "/"
         . env('BC_ENVIRONMENT')
         . "/api/v2.0/companies(" . $this->companyId . ")/purchaseOrders(83ea2b56-d008-ee11-8f70-00224857ecae)/purchaseOrderLines";

    try {
        $resp = $this->client->get($url, [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer'        => 'odata.maxpagesize=1000',
            ],
            // Optional: let 4xx/5xx come back without throwing
            // 'http_errors' => false,
        ]);

        return [
            'ok'      => true,
            'status'  => $resp->getStatusCode(),
            'reason'  => $resp->getReasonPhrase(),
            'headers' => $resp->getHeaders(),
            'json'    => json_decode((string)$resp->getBody(), true),
            'raw'     => (string)$resp->getBody(),
        ];

    } catch (RequestException $e) {
        $res     = $e->getResponse();
        $status  = $res ? $res->getStatusCode() : null;
        $reason  = $res ? $res->getReasonPhrase() : null;
        $headers = $res ? $res->getHeaders() : [];
        $raw     = $res ? (string)$res->getBody() : $e->getMessage();
        $json    = json_decode($raw, true);

        // Log everything (no truncation in logs)
        Log::error('BC GET purchaseOrderLines failed', [
            'url'      => $url,
            'status'   => $status,
            'reason'   => $reason,
            'headers'  => $headers,
            'raw'      => $raw,
            'exception'=> $e->getMessage(),
        ]);

        // Return a rich error object to caller
        return [
            'ok'       => false,
            'status'   => $status,
            'reason'   => $reason,
            'headers'  => $headers,
            'errorRaw' => $raw,
            'error'    => $json ?? $raw,  // parsed JSON if possible
            'message'  => $e->getMessage(),
        ];
    } catch (\Throwable $e) {
        Log::error('Unexpected error calling BC', [
            'url'      => $url,
            'exception'=> $e->getMessage(),
        ]);

        return [
            'ok'      => false,
            'status'  => 500,
            'reason'  => 'Unexpected Error',
            'error'   => $e->getMessage(),
        ];
    }
}

public function getAPIStockkeeping()
{
    // Endpoint tanpa filter
    $url = "https://api.businesscentral.dynamics.com/v2.0/" 
        . env('AZURE_TENANT_ID') . "/" 
        . env('BC_ENVIRONMENT')
        . "/ODataV4/Company('" . $this->companyId . "')/APIStockkeeping";

    $headers = [
        'headers' => [
            'Authorization' => "Bearer {$this->token}",
            'Accept'        => 'application/json',
            'Prefer'        => 'odata.maxpagesize=1000'
        ]
    ];

    $allData = [];
    do {
        $response = $this->client->get($url, $headers);
        $decoded = json_decode((string) $response->getBody(), true);

        if (isset($decoded['value'])) {
            $allData = array_merge($allData, $decoded['value']);
        }

        // Handle pagination jika ada nextLink
        $url = $decoded['@odata.nextLink'] ?? null;
    } while ($url);

    return ['value' => $allData];
}


    function getStockkeepingFromBC()
    {
        $filter = urlencode("(Location_Code eq 'CI.1010')");
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/Company('" . $this->companyId . "')/StockkeepingUnit?\$filter={$filter}";
        
        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer' => 'odata.maxpagesize=1000'
            ]
        ];
        $allData = [];
        do {
            $response = $this->client->get($url, $headers);
            $decoded = json_decode((string)$response->getBody(), true);

            if (isset($decoded['value'])) {
                $allData = array_merge($allData, $decoded['value']);
            }

            $url = $decoded['@odata.nextLink'] ?? null;
        } while ($url);
        return ['value' => $allData];
    }

    function itemUoM()
    {
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/Company('" . $this->companyId . "')/Item_UOM";
        
        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer' => 'odata.maxpagesize=1000'
            ]
        ];
        $allData = [];
        do {
            $response = $this->client->get($url, $headers);
            $decoded = json_decode((string)$response->getBody(), true);

            if (isset($decoded['value'])) {
                $allData = array_merge($allData, $decoded['value']);
            }

            $url = $decoded['@odata.nextLink'] ?? null;
        } while ($url);

        dd($allData);
        return ['value' => $allData];
    }


    public function getAllVendorsMap()
    {
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/api/v2.0/companies({$this->companyId})/vendors?\$select=number,displayName";
        $vendors = [];
        do {
            $response = $this->client->get($url, [
                'headers' => [
                    'Authorization' => "Bearer {$this->token}",
                    'Accept' => 'application/json',
                    'Prefer' => 'odata.maxpagesize=1000'
                ]
            ]);
            $data = json_decode($response->getBody(), true);

            foreach ($data['value'] as $vendor) {
                $vendors[$vendor['number']] = $vendor['displayName'];
            }

            $url = $data['@odata.nextLink'] ?? null; 

        } while ($url);

        return $vendors;
    }


    public function getVendor() {
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/api/v2.0/companies({$this->companyId})/vendors";

        $response = $this->client->get($url, [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept' => 'application/json'
            ]
        ]);
        return json_decode($response->getBody(), true);
    }

    
public function createItem(array $itemData)
{
    $url = "https://api.businesscentral.dynamics.com/v2.0/"
        . env('AZURE_TENANT_ID') . "/"
        . env('BC_ENVIRONMENT')
        . "/api/v2.0/companies({$this->companyId})/items";

    // Ambil unitPrice
    $unitPrice = 0;
    if (isset($itemData['unitPrice'])) {
        $unitPrice = $itemData['unitPrice'];
    } elseif (!empty($itemData['vendorQuotes']) && is_array($itemData['vendorQuotes']) && isset($itemData['vendorQuotes'][0]['price'])) {
        $unitPrice = $itemData['vendorQuotes'][0]['price'];
    }
    $unitPrice = is_numeric($unitPrice) ? (float) $unitPrice : 0.0;

    // Siapkan payload
    $payload = [
        'number' => $itemData['number'] ?? null,
        'displayName' => $itemData['displayName'] ?? null,
        'type' => $itemData['type'] ?? 'Inventory',
        'baseUnitOfMeasureCode' => $itemData['baseUnitOfMeasureCode'] ?? 'PTN',
        'generalProductPostingGroupCode' => $itemData['generalProductPostingGroupCode'] ?? 'GP.FOD075',
        'inventoryPostingGroupCode' => $itemData['inventoryPostingGroupCode'] ?? 'INV_FOOD',
        'unitPrice' => $unitPrice,
    ];

    // 🧩 Client Guzzle dengan timeout panjang
    $client = new \GuzzleHttp\Client([
        'timeout' => 180,          // Maksimal 3 menit per request
        'connect_timeout' => 20,   // Maksimal waktu membuka koneksi
    ]);

    try {
        $response = $client->post($url, [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept' => 'application/json',
            ],
            'json' => $payload,
        ]);

        return [
            'status' => 'success',
            'data' => json_decode((string) $response->getBody(), true)
        ];
    } catch (\GuzzleHttp\Exception\RequestException $e) {
        $responseBody = $e->hasResponse()
            ? (string) $e->getResponse()->getBody()
            : null;

        \Log::error('BC createItem RequestException: ' . $e->getMessage(), [
            'payload' => $payload,
            'response' => $responseBody
        ]);

        return [
            'status' => 'error',
            'message' => $e->getMessage(),
            'response' => $responseBody
        ];
    } catch (\Exception $e) {
        \Log::error('BC createItem general error: ' . $e->getMessage(), [
            'payload' => $payload
        ]);

        return [
            'status' => 'error',
            'message' => $e->getMessage()
        ];
    }
}


public function createVendor(array $vendorData): array
{
    // =====================
    // 1️⃣ Buat Vendor
    // =====================
    $url = "https://api.businesscentral.dynamics.com/v2.0/"
        . env('AZURE_TENANT_ID') . "/"
        . env('BC_ENVIRONMENT') . "/api/v2.0/companies({$this->companyId})/vendors";

    $payload = [];
    if (!empty($vendorData['number'])) $payload['number'] = $vendorData['number'];
    if (!empty($vendorData['displayName'])) $payload['displayName'] = $vendorData['displayName'];
    if (!empty($vendorData['addressLine1'])) $payload['addressLine1'] = $vendorData['addressLine1'];
    if (!empty($vendorData['addressLine2'])) $payload['addressLine2'] = $vendorData['addressLine2'];
    if (!empty($vendorData['city'])) $payload['city'] = $vendorData['city'];
    if (!empty($vendorData['state'])) $payload['state'] = $vendorData['state'];
    if (!empty($vendorData['country'])) $payload['country'] = $vendorData['country'];
    if (!empty($vendorData['postalCode'])) $payload['postalCode'] = $vendorData['postalCode'];
    if (!empty($vendorData['phoneNumber'])) $payload['phoneNumber'] = $vendorData['phoneNumber'];
    if (!empty($vendorData['email'])) $payload['email'] = $vendorData['email'];
    if (!empty($vendorData['website'])) $payload['website'] = $vendorData['website'];
    if (!empty($vendorData['taxRegistrationNumber'])) $payload['taxRegistrationNumber'] = $vendorData['taxRegistrationNumber'];
    if (!empty($vendorData['currencyId']) && $vendorData['currencyId'] !== '00000000-0000-0000-0000-000000000000') $payload['currencyId'] = $vendorData['currencyId'];
    if (!empty($vendorData['paymentTermsId']) && $vendorData['paymentTermsId'] !== '00000000-0000-0000-0000-000000000000') $payload['paymentTermsId'] = $vendorData['paymentTermsId'];
    if (!empty($vendorData['paymentMethodId']) && $vendorData['paymentMethodId'] !== '00000000-0000-0000-0000-000000000000') $payload['paymentMethodId'] = $vendorData['paymentMethodId'];
    if (isset($vendorData['taxLiable'])) $payload['taxLiable'] = (bool)$vendorData['taxLiable'];
    if (isset($vendorData['blocked'])) $payload['blocked'] = $vendorData['blocked'];

    $client = new \GuzzleHttp\Client([
        'timeout' => 180,
        'connect_timeout' => 20,
    ]);

    try {
        $response = $client->post($url, [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept' => 'application/json',
            ],
            'json' => $payload,
        ]);

        $vendorCreated = json_decode((string)$response->getBody(), true);
    } catch (\Exception $e) {
        \Log::error('BC createVendor error: ' . $e->getMessage(), ['payload' => $payload]);
        return [
            'status' => 'error',
            'message' => $e->getMessage(),
        ];
    }

    // =====================
    // 2️⃣ Buat PriceListHeaderCustom
    // =====================
    $base = "https://api.businesscentral.dynamics.com/v2.0/"
        . env('AZURE_TENANT_ID') . "/"
        . env('BC_ENVIRONMENT');
$root = $base . "/api/citbi/sku/v1.0";
$priceUrl = "{$root}/companies({$this->companyId})/priceListHeaderCustom";

$client = new \GuzzleHttp\Client([
    'timeout' => 180,
    'connect_timeout' => 20,
]);
$etag = $priceCreated['@odata.etag'] ?? null;
$priceCode = $vendorCreated['number'] ?? $vendorData['number'];
// ===================
// Tahap 1: Create PriceListHeaderCustom (priceCode + Description)
// ===================
$createPayload = [
    'priceCode' => $vendorCreated['number'] ?? $vendorData['number'],
    'Description' => $vendorCreated['displayName'] ?? $vendorData['displayName'],
];

try {
    $response = $client->post($priceUrl, [
        'headers' => [
            'Authorization' => "Bearer {$this->token}",
            'Accept' => 'application/json',
            'If-Match' => $etag ?? '*',
        ],
        'json' => $createPayload,
    ]);

    $priceCreated = json_decode((string)$response->getBody(), true);
    $priceCode = $priceCreated['priceCode'] ?? $createPayload['priceCode'];
} catch (\Exception $e) {
    \Log::error('BC createVendorPrice (create) error: ' . $e->getMessage(), [
        'payload' => $createPayload
    ]);
    $priceCreated = null;
}

// ===================
// Tahap 2: Patch sourceType
// ===================
if ($priceCode) {
    try {
        $client->patch("{$priceUrl}('{$priceCode}')", [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept' => 'application/json',
                'If-Match' => $etag ?? '*',
            ],
            'json' => [
                'sourceType' => 'Vendor'
            ]
        ]);
    } catch (\Exception $e) {
        \Log::error('BC createVendorPrice (patch sourceType) error: ' . $e->getMessage());
    }
}

// ===================
// Tahap 3: Patch sourceNo
// ===================
if ($priceCode) {
    try {
        $client->patch("{$priceUrl}('{$priceCode}')", [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept' => 'application/json',
                'If-Match' => $etag ?? '*',
            ],
            'json' => [
                'sourceNo' => $vendorCreated['number'] ?? $vendorData['number']
            ]
        ]);
    } catch (\Exception $e) {
        \Log::error('BC createVendorPrice (patch sourceNo) error: ' . $e->getMessage());
    }
}

// ===================
// Tahap 4: Patch status
// ===================
if ($priceCode) {
    try {
        $client->patch("{$priceUrl}('{$priceCode}')", [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept' => 'application/json',
                'If-Match' => $etag ?? '*',
            ],
            'json' => [
                'status' => 'Active'
            ]
        ]);
    } catch (\Exception $e) {
        \Log::error('BC createVendorPrice (patch status) error: ' . $e->getMessage());
    }
}

    return [
        'status' => 'success',
        'data' => [
            'vendor' => $vendorCreated,
            'priceListHeaderCustom' => $priceCreated
        ]
    ];
}


    public function getNoSeries()
{
    $url = "https://api.businesscentral.dynamics.com/v2.0/"
        . env('AZURE_TENANT_ID') . "/"
        . env('BC_ENVIRONMENT')
        . "/ODataV4/Company('PT.CI%20LIVE')/NoSeries";

    try {
        $response = $this->client->get($url, [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept' => 'application/json',
            ]
        ]);
        

        $data = json_decode((string)$response->getBody(), true);
        return [
            'status' => 'success',
            'data' => $data
        ];
    } catch (\Exception $e) {
        \Log::error('BC getNoSeries error: ' . $e->getMessage(), [
            'url' => $url,
            'exception' => $e
        ]);

        return [
            'status' => 'error',
            'message' => $e->getMessage()
        ];
    }
}
        



public function getallitemdata($userName)
{
    $start = microtime(true);
    $priceRaw = $vendorRaw = [];
    $baseOdata = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/ODataV4/Company('" . $this->companyId . "')";
    $baseApi = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/api/v2.0/companies({$this->companyId})";

    $headers = [
        'Authorization' => "Bearer {$this->token}",
        'Accept'        => 'application/json',
        'Prefer'        => 'odata.maxpagesize=50000'
    ];

    // -------------------------------
    // Ambil Price List dan Vendor
    // -------------------------------
    $promises = [
        'priceLists' => $this->client->getAsync(
            "$baseOdata/Price_List_Lines?\$select=Asset_No,Product_No,Unit_of_Measure_Code,DirectUnitCost,SourceNo,StartingDate,EndingDate,Status",
            ['headers' => $headers]
        ),
        'vendors' => $this->client->getAsync("$baseApi/vendors?\$select=number,displayName", ['headers' => $headers])
    ];

    $responses = Utils::settle($promises)->wait();

    $priceRaw  = $responses['priceLists']['state'] === 'fulfilled' ? json_decode($responses['priceLists']['value']->getBody()->getContents(), true)['value'] ?? [] : [];
    $vendorRaw = $responses['vendors']['state'] === 'fulfilled' ? json_decode($responses['vendors']['value']->getBody()->getContents(), true)['value'] ?? [] : [];

    // -------------------------------
    // Price processing
    // -------------------------------
    $priceMap = [];
    $unitCostsByItem = [];
    $priceDateMap = [];
    $statusPriceMap = [];

    foreach ($priceRaw as $line) {
        $itemNo   = $line['Asset_No'] ?? null;
        $vendorNo = $line['SourceNo'] ?? null;
        $uomCode  = $line['Unit_of_Measure_Code'] ?? null;
        $unitCost = $line['DirectUnitCost'] ?? null;
        $startDate = isset($line['StartingDate']) ? strtotime($line['StartingDate']) : 0;
        $endDate   = isset($line['EndingDate']) ? strtotime($line['EndingDate']) : 0;
        if (!$itemNo || !$vendorNo) continue;

        $priceMap[$itemNo][$vendorNo][] = [
            'unit_cost' => $unitCost,
            'uom'       => $uomCode,
            'start'     => $startDate,
            'end'       => $endDate,
            'starting_date' => $line['StartingDate'] ?? null,
            'ending_date'   => $line['EndingDate'] ?? null
        ];

        $statusPriceMap[$itemNo][$vendorNo] = $line['Status'] ?? null;
    }

    foreach ($priceMap as $itemNo => $vendors) {
        foreach ($vendors as $vendorNo => $prices) {
            usort($prices, function($a,$b){
                if ($a['end'] !== $b['end']) return $b['end'] - $a['end'];
                return $b['start'] - $a['start'];
            });
            $selected = $prices[0];
            $unitCostsByItem[$itemNo][$vendorNo] = $selected['unit_cost'];
            $priceDateMap[$itemNo][$vendorNo] = [
                'starting_date' => $selected['starting_date'],
                'ending_date'   => $selected['ending_date']
            ];
            $priceMap[$itemNo][$vendorNo] = true;
        }
    }
    foreach ($priceMap as $itemNo => $vendorList) {
        $priceMap[$itemNo] = array_keys($vendorList);
    }

    // -------------------------------
    // Vendor map
    // -------------------------------
    $vendorMap = [];
    foreach ($vendorRaw as $v) {
        $vendorMap[$v['number']] = $v['displayName'];
    }

    // -------------------------------
    // Build results hanya dari Price List + Vendor
    // -------------------------------
    $results = [];
    $seenKeys = [];

    foreach ($priceMap as $itemNo => $vendorNos) {
        foreach ($vendorNos as $vendorNo) {
            $uniqueKey = $itemNo.'_'.$vendorNo;
            if (isset($seenKeys[$uniqueKey])) continue;
            $seenKeys[$uniqueKey] = true;

            $dates = $priceDateMap[$itemNo][$vendorNo] ?? ['starting_date'=>null,'ending_date'=>null];

            $results[] = [
                'item_no' => $itemNo,
                'vendor_name' => $vendorMap[$vendorNo] ?? null,
                'vendor_no' => $vendorNo,
                'unit_cost' => $unitCostsByItem[$itemNo][$vendorNo] ?? 0,
                'starting_date' => $dates['starting_date'] ? date('d/m/Y', strtotime($dates['starting_date'])) : null,
                'ending_date'   => $dates['ending_date'] ? date('d/m/Y', strtotime($dates['ending_date'])) : null,
                'status_price' => $statusPriceMap[$itemNo][$vendorNo] ?? null
            ];
        }
    }

    return [
        'items' => $results,
        'vendors' => $vendorMap
    ];
}




    public function getItemIdByNo(string $itemNo)
    {
        $filter = urlencode("number eq '$itemNo'");
        $response = $this->client->get("https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
       "/api/v2.0/companies({$this->companyId})/items?\$filter=$filter", [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Content-Type'  => 'application/json',
            ]
        ]);

        $items = json_decode($response->getBody(), true);
        return $items['value'][0]['id'] ?? null;

        //$url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/api/citbi/sku/v1.0/companies(" . rawurlencode($this->companyId) . ")/GeneralJournalLines";
    }


    public function createPurchaseOrderForVendor($vendorNo, $userName, $location)
    {
        if ($location == 'CI.1010') {
            $purchaser = 'TN01';
        } 
        else if ($location == 'CI.1020'){
            $purchaser = 'UT01';
        } else if ($location == 'HIN.1200' || $location == 'HIN.1300') {
            $purchaser = 'PE01';
        } else if ($location == 'HIN.1000') {
            $purchaser = 'PW01';
        } else if ($location == 'HIN.3000') {
            $purchaser = 'FB01';
        }

        $payload = [
            'vendorNumber' => $vendorNo,
            'orderDate' => now()->toDateString(),
            'purchaser' => $purchaser,
            'shortcutDimension2Code' => "100000",
            'requestedReceiptDate' => today()->toDateString()
        ];

        $response = $this->client->post("https://api.businesscentral.dynamics.com/v2.0/".env('AZURE_TENANT_ID')."/".env('BC_ENVIRONMENT')."/api/v2.0/companies({$this->companyId})/purchaseOrders", [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Content-Type'  => 'application/json',
            ],
            'body' => json_encode($payload)
        ]);
        
        $body = json_decode($response->getBody(), true);
        return json_decode($response->getBody(), true);
    }

    public function getLocationIdByCode($locationCode)
    {
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/api/v2.0/companies({$this->companyId})/locations?\$filter=code eq '{$locationCode}'";

        $response = $this->client->get($url, [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept' => 'application/json'
            ]
        ]);

        $data = json_decode((string)$response->getBody(), true);
        
        return $data['value'][0]['id'] ?? null;
    }
    
    
    public function getAsyncAddPurchaseLinePromise($poId, $itemNo, $quantity, $userName)
    {   
        $start = microtime(true);
        $itemId     = $this->getItemIdByNo($itemNo);
        $locationId = $this->getLocationIdByCode(in_array($userName, [
                'titania@citbi.onmicrosoft.com',
                'adminbc@citbi.onmicrosoft.com'
            ]) ? 'CI.1010' : 'CI.1020');

        $payload = [
            'itemId'     => $itemId,
            'quantity'   => (float) $quantity,
            'locationId' => $locationId,
        ];
        $elapsed = microtime(true) - $start;
        Log::info("Create POLine runtime: {$elapsed} seconds");
        return $this->client->postAsync(
            "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/api/v2.0/companies({$this->companyId})/purchaseOrders({$poId})/purchaseOrderLines",
            [
                'headers' => [
                    'Authorization' => "Bearer {$this->token}",
                    'Accept'        => 'application/json',
                    'Content-Type'  => 'application/json',
                ],
                'body' => json_encode($payload),
            ]
        );
    }

    public function addPurchaseLineToPO(string $poNo, string $itemNo, float $quantity, $cost, string $locationCode = '', $userName, string $poId)
    {
        $base = "https://api.businesscentral.dynamics.com/v2.0/"
          . env('AZURE_TENANT_ID') . "/"
          . env('BC_ENVIRONMENT');

        $root = $base . "/api/citbi/sku/v1.0";
        $linesUrl = "{$root}/companies({$this->companyId})/purchaseOrderLinesCustom"; // <-- exact casing!

        $headersJson = [
            'Authorization'    => "Bearer {$this->token}",
            'Accept'           => 'application/json',
            'Content-Type'     => 'application/json',
        ];

        try {
            $flt = urlencode("documentType eq 'Order' and documentNo eq '{$poNo}'");
            $lastUrl = $linesUrl . "?\$filter={$flt}&\$orderby=lineNo desc&\$top=1&\$select=lineNo";
            $lastResp = $this->client->get($lastUrl, ['headers' => $headersJson]);
            $lastJson = json_decode((string)$lastResp->getBody(), true);
            $lastLineNo = (int)($lastJson['value'][0]['lineNo'] ?? 0);
            $nextLineNo = max(10000, $lastLineNo + 10000);

            $payload = [
                'documentType' => 'Order',
                'documentNo'   => (string)$poNo,
                'lineNo'       => $nextLineNo,
                'itemNo'       => $itemNo,
                'type'         => 'Item',
                'quantity'     => (float)$quantity
            ];
            if ($locationCode !== '') {
                $payload['locationCode'] = $locationCode;
            }
            $resp = $this->client->post($linesUrl, [ 'headers' => $headersJson, 'body' => json_encode($payload, JSON_UNESCAPED_SLASHES), ]); 
            $data = json_decode((string)$resp->getBody(), true); 
            $etag  = $data['@odata.etag'] ?? '*';
            $idUrl = $data['@odata.id']   ?? null;
            \Log::info('API line created (doc keys)', [ 'url' => $linesUrl, 'payload' => $payload, 'response' => $data ]);

            return $data;

        } catch (\GuzzleHttp\Exception\RequestException $e) {
            $res = $e->getResponse();
            $raw = $res ? (string)$res->getBody() : $e->getMessage();
            \Log::error('Custom API purchase line POST failed', [
                'url'       => $linesUrl,
                'status'    => $res?->getStatusCode(),
                'reason'    => $res?->getReasonPhrase(),
                'headers'   => $res?->getHeaders(),
                'payload'   => $payload ?? null,
                'error_raw' => $raw,
                'requestId' => $res?->getHeader('request-id')[0] ?? null,
            ]);
            throw $e;
        }
    }

    public function getAllPurchasePrices()
    {
        $start = microtime(true);
        $today = now()->toDateString(); // e.g. "2025-06-17"
        $filter = "SourceNo ne null and StartingDate le $today and EndingDate ge $today";
        $encodedFilter = urlencode($filter);

        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/Company('" . $this->companyId . "')/Price_List_Lines?\$filter={$encodedFilter}";

        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer'        => 'odata.maxpagesize=1000'
            ]
        ];

        $allData = [];

        try {
            do {
                $response = $this->client->get($url, $headers);
                $decoded = json_decode((string)$response->getBody(), true);

                if (isset($decoded['value'])) {
                    $allData = array_merge($allData, $decoded['value']);
                }

                $url = $decoded['@odata.nextLink'] ?? null;
            } while ($url);

            return collect($allData)->groupBy('Asset_No');
        } catch (\Exception $e) {
            Log::error("Failed to fetch Price List Lines: " . $e->getMessage());
            return collect(); 
        }
        $elapsed = microtime(true) - $start;
        Log::info("Fetch runtime Priccss: {$elapsed} seconds");
        
    }

    public function getStockOpnameItems($locationCode)
    {
        $baseUrl = "https://api.businesscentral.dynamics.com/v2.0/"
            . env('AZURE_TENANT_ID') . "/"
            . env('BC_ENVIRONMENT')
            . "/ODataV4/Company('{$this->companyId}')/";

        try {
            // ===============================
            // 1️⃣ CALL 3 ENDPOINT
            // ===============================
            $endpoints = [
                'base' => "StockOpnameItem?\$filter=LocationCode eq '{$locationCode}'",
                'inc'  => "StockOpnameInc?\$filter=LocationCode eq '{$locationCode}'",
                'dec'  => "StockOpnameDec?\$filter=LocationCode eq '{$locationCode}'",
            ];

            $responses = [];

            foreach ($endpoints as $key => $endpoint) {
                $res = $this->client->get($baseUrl . $endpoint, [
                    'headers' => [
                        'Authorization' => "Bearer {$this->token}",
                        'Accept'        => "application/json;odata.metadata=none"
                    ]
                ]);

                $json = json_decode($res->getBody(), true);
                $responses[$key] = $json['value'] ?? [];
            }

            // ===============================
            // 2️⃣ PREFIX FILTER
            // ===============================
            $allowedPrefix = ['FO', 'BV', 'HO', 'MT'];

            // ===============================
            // 3️⃣ INDEX BASE DATA
            // ===============================
            $items = [];

            foreach ($responses['base'] as $row) {
                $itemNo = $row['ItemNo'] ?? null;
                if (!$itemNo) continue;

                $prefix = strtoupper(strtok($itemNo, '.'));
                if (!in_array($prefix, $allowedPrefix, true)) continue;

                $items[$itemNo] = [
                    'LocationCode' => $row['LocationCode'] ?? $locationCode,
                    'ItemNo'       => $itemNo,
                    'Description'  => $row['Description'] ?? null,
                    'UoMCode'      => $row['UoMCode'] ?? null,
                    'UnitCost'     => $row['UnitCost'] ?? 0,
                    'Qty'          => $row['Qty'] ?? 0,        // Remaining / base
                    'QtyIncrease'  => 0,
                    'QtyDecrease'  => 0,
                ];
            }

            // ===============================
            // 4️⃣ MERGE INCREASE
            // ===============================
            foreach ($responses['inc'] as $row) {
                $itemNo = $row['ItemNo'] ?? null;
                if (!$itemNo || !isset($items[$itemNo])) continue;

                $items[$itemNo]['QtyIncrease'] = $row['QtyIncrease'] ?? 0;
            }

            // ===============================
            // 5️⃣ MERGE DECREASE
            // ===============================
            foreach ($responses['dec'] as $row) {
                $itemNo = $row['ItemNo'] ?? null;
                if (!$itemNo || !isset($items[$itemNo])) continue;

                $items[$itemNo]['QtyDecrease'] = $row['QtyDecrease'] ?? 0;
            }

            // ===============================
            // 6️⃣ RETURN FLAT ARRAY
            // ===============================
            return array_values($items);

        } catch (\Throwable $e) {
            return ['error' => $e->getMessage()];
        }
    }

public function getStockOpnameItemsv2($locationCode)
{
    $url = "https://api.businesscentral.dynamics.com/v2.0/"
        . env('AZURE_TENANT_ID') . "/"
        . env('BC_ENVIRONMENT')
        . "/api/citbi/sku/v1.0/companies({$this->companyId})/stockOpnameItems?\$filter=locationCode eq '{$locationCode}'";

    try {
        $response = $this->client->get($url, [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => "application/json;odata.metadata=none",
            ],
            // 🛠️ ⬇️ tambahkan ini
            'timeout'         => 90,  // maksimal request 15 detik
            'connect_timeout' => 90,   // maksimal koneksi 5 detik
        ]);

        $data = json_decode($response->getBody(), true);

        $rows = $data['value'] ?? $data;
        if (!is_array($rows) || empty($rows)) return [];

        $allowedPrefix = ['FO', 'BV', 'HO', 'MT'];
        $unique = [];
        $filtered = [];

        foreach ($rows as $row) {
            $item = $row['itemNo'] ?? null;
            if (!$item) continue;

            $prefix = strtoupper(strtok($item, '.'));
            if (!in_array($prefix, $allowedPrefix, true)) continue;

            if (!isset($unique[$item])) {
                $unique[$item] = true;
                $filtered[] = $row;
            }
        }

        return $filtered;

    } catch (\GuzzleHttp\Exception\RequestException $e) {
        return ['error' => 'Request timeout / gagal koneksi: ' . $e->getMessage()];
    } catch (\Throwable $e) {
        return ['error' => $e->getMessage()];
    }
}

public function createInventoryReceipt(array $payload): array
{
    try {
        $url =
            "https://api.businesscentral.dynamics.com/v2.0/"
            . env('AZURE_TENANT_ID') . "/"
            . env('BC_ENVIRONMENT')
            . "/ODataV4/InventoryReciept_CreateInventoryReceipt"
            . "?Company=PT.CI%20LIVE";

        // ===============================
        // AUTO BUSINESS UNIT FROM LOCATION
        // ===============================
        $businessUnit = strtoupper(explode('.', $payload['location_code'])[0]);

        /**
         * ====================================================
         * BUILD INNER JSON (ARRAY → STRING)
         * ====================================================
         */
        $inner = [
            'NoSeriesCode'       => $payload['no_series'],
            'LocationCode'       => $payload['location_code'],
            'GenBusCode'         => $payload['gen_bus_code'],
            'BusinessUnitCode'   => $businessUnit,
            'DepartmentCode'     => $payload['department_code'],
            'PostingDate'        => $payload['posting_date'] ?? null,
            'Lines'              => array_map(fn ($l) => [
                'ItemNo'   => $l['item_no'],
                'Quantity'=> (float) $l['quantity'],
            ], $payload['lines']),
        ];

        $body = new \stdClass();
        $body->jsonPayload = json_encode($inner, JSON_UNESCAPED_SLASHES);

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->token,
            'Accept'        => 'application/json',
            'Content-Type'  => 'application/json',
        ])->post($url, $body);

        if ($response->failed()) {
            Log::error('[BC] InventoryReciept_CreateInventoryReceipt failed', [
                'status' => $response->status(),
                'body'   => $response->body(),
                'payload'=> $inner
            ]);
            throw new \Exception('BC Inventory Receipt creation failed');
        }

        return [
            'success'     => true,
            'document_no' => $response->json(),
        ];

    } catch (\Throwable $e) {
        Log::error('[BC] InventoryReciept_CreateInventoryReceipt exception', [
            'message' => $e->getMessage(),
            'payload' => $payload
        ]);
        throw $e;
    }
}

public function createInventoryShipment(array $payload): array
{
    try {
        $url =
            "https://api.businesscentral.dynamics.com/v2.0/"
            . env('AZURE_TENANT_ID') . "/"
            . env('BC_ENVIRONMENT')
            . "/ODataV4/InventoryShipment_CreateInventoryShipment"
            . "?Company=PT.CI%20LIVE";

        // ===============================
        // AUTO BUSINESS UNIT FROM LOCATION
        // ===============================
        $businessUnit = strtoupper(explode('.', $payload['location_code'])[0]);

        $inner = [
            'NoSeriesCode'       => $payload['no_series'],
            'LocationCode'       => $payload['location_code'],
            'GenBusCode'         => $payload['gen_bus_code'],
            'BusinessUnitCode'   => $businessUnit,
            'DepartmentCode'     => $payload['department_code'],
            'PostingDate'        => $payload['posting_date'] ?? null,
            'Lines'              => array_map(fn ($l) => [
                'ItemNo'   => $l['item_no'],
                'Quantity'=> (float) $l['quantity'], // ABS sudah di controller
            ], $payload['lines']),
        ];

        $body = new \stdClass();
        $body->jsonPayload = json_encode($inner, JSON_UNESCAPED_SLASHES);

        $response = Http::withHeaders([
            'Authorization' => 'Bearer ' . $this->token,
            'Accept'        => 'application/json',
            'Content-Type'  => 'application/json',
        ])->post($url, $body);

        if ($response->failed()) {
            Log::error('[BC] InventoryShipment_CreateInventoryShipment failed', [
                'status' => $response->status(),
                'body'   => $response->body(),
                'payload'=> $inner
            ]);
            throw new \Exception('BC Inventory Shipment creation failed');
        }

        return [
            'success'     => true,
            'document_no' => $response->json(),
        ];

    } catch (\Throwable $e) {
        Log::error('[BC] InventoryShipment_CreateInventoryShipment exception', [
            'message' => $e->getMessage(),
            'payload' => $payload
        ]);
        throw $e;
    }
}

public function getAssemblyServiceData()
    {
        $url = "https://api.businesscentral.dynamics.com/v2.0/"
            . env('AZURE_TENANT_ID') . "/"
            . env('BC_ENVIRONMENT')
            . "/ODataV4/InventoryReciept_CreateInventoryReceipt?Company=PT.CI%20LIVE";

        try {
            // Contoh body JSON kosong jika action tidak pakai parameter
            $body = new \stdClass(); 
            
            // $body->assemblyNo = 'SO.CI.25120005';
            // $body->qtyBase    = 1;
            // $body->type       = 'STOCK';
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $this->token,
                'Accept'        => 'application/json',
                'Content-Type'  => 'application/json',
            ])->post($url, $body);

            if ($response->failed()) {
                Log::error('[BC] AssemblyService POST failed', [
                    'url' => $url,
                    'status' => $response->status(),
                    'body' => $response->body(),
                ]);

                return response()->json([
                    'success' => false,
                    'status'  => $response->status(),
                    'message' => 'POST request to AssemblyService failed',
                    'response' => $response->body(),
                ], 500);
            }

            return response()->json([
                'success' => true,
                'data' => $response->json(),
            ]);

        } catch (\Throwable $e) {
            Log::error('[BC] Error posting AssemblyService', [
                'message' => $e->getMessage(),
            ]);

            return response()->json([
                'success' => false,
                'message' => $e->getMessage(),
            ], 500);
        }
    }



}
