<?php

namespace App\Http\Controllers;

use App\Models\{ItemRequest, ItemRequestLine};
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\{ApprovalFlowTemplate, Company, User,ItemUserInput};
use Illuminate\Support\Facades\Validator;
use App\Models\ItemRequestAttachment;
use App\Models\OpnameSchedule;
use App\Models\ItemBeforeS2;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Arr;
use App\Services\BusinessCentralService;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Log;
use Maatwebsite\Excel\Facades\Excel;

class ItemRequestController extends Controller
{
    public function create()
    {
        $user = session('user');
        $me   = $this->resolveCurrentUser();

        abort_unless($me->canCreateItemRequests(), 403, 'You are not allowed to create item requests.');

        // ===============================
        // 🔥 FLOW, COMPANY, DEFAULT
        // ===============================
        $flows = ApprovalFlowTemplate::where('object_type', 'NewItem')
            ->where('is_active', 1)
            ->orderByDesc('version')
            ->get();

        $companies = Company::orderBy('id', 'desc')->get();

        $defaultCompanyId = $me->defaultCompanyId() ?? optional($companies->first())->id;


        // ===============================
        // 🔥 LOCATION (sesuai business_unit)
        // ===============================
        $businessUnit = $me->business_unit; // contoh: RBC / HIC / CI / dll

        // Ambil HANYA code lokasi untuk form
        $locationCodes = \App\Models\Location::when($businessUnit, function($q) use ($businessUnit) {
                return $q->where('location', $businessUnit);
            })
            ->pluck('code') // hasil contoh: ["RBC.2100", "RBC.2110", ...]
            ->toArray();


        // ===============================
        // 🔥 VIEW
        // ===============================
        return view('approvals.new', [
            'flows'            => $flows,
            'companies'        => $companies,
            'defaultCompanyId' => $defaultCompanyId,
            'vendorSet'        => [],
            'allItems'         => [],
            'locationCodes'    => $locationCodes, // ⬅️ ini yg dipakai di form
        ]);
    }


    public function getItems(Request $r)
        {
            $user = session('user');
            $userName = $user['email'] ?? null;
            $service = new BusinessCentralService();
            $poData = $service->getallitemdata($userName);

            $allItems = $poData['items'];
            $vendorSet = collect($allItems)
                ->filter(fn($item) => $item['vendor_no'] && $item['vendor_name'])
                ->pluck('vendor_name', 'vendor_no')
                ->unique()
                ->sortBy(fn($name, $no) => $name)
                ->toArray();
            
            return response()->json([
                'allItems' => $allItems,
                'vendorSet' => $vendorSet,
            ]);
        }

    protected function resolveCurrentUser(): User
    {
        $sess  = session('user') ?? [];
        $email = $sess['email'] ?? (auth()->check() ? auth()->user()->email : null);
        $name  = $sess['name'] ?? ($sess['displayName'] ?? (auth()->check() ? (auth()->user()->name ?? 'MS User') : 'MS User'));

        abort_unless($email, 401, 'Not authenticated');

        return User::firstOrCreate(['email' => $email], ['name' => $name]);
    }

public function store(Request $r)
{
    $me = $this->resolveCurrentUser();

    // ============================
    // VALIDASI HEADER + PAYLOAD
    // ============================
    $data = $r->validate([
        'company_id'       => 'required|integer|exists:companies,id',
        'flow_template_id' => 'required|integer|exists:approval_flow_templates,id',
        'posting_date'     => 'nullable|date',
        'vendor_name'      => 'nullable|string|max:255',
        'location'         => 'nullable|string|max:255',
        'remarks'          => 'nullable|string',

        // ⬇️ HASIL PARSE EXCEL DARI JS
        'excel_payload'    => 'required|string',
    ]);

    // ============================
    // DECODE EXCEL JSON
    // ============================
    $rows = json_decode($data['excel_payload'], true);

    if (!is_array($rows) || empty($rows)) {
        return back()
            ->withErrors(['excel_payload' => 'Excel data is empty or invalid'])
            ->withInput();
    }

    $req = null;

    DB::transaction(function () use ($data, $rows, $me, &$req) {

        // ======================
        // SIMPAN HEADER
        // ======================
        $req = ItemRequest::create([
            'company_id'       => (int) $data['company_id'],
            'requester_id'     => $me->id,
            'flow_template_id' => $data['flow_template_id'],
            'posting_date'     => $data['posting_date'] ?? null,
            'location'         => $data['location'] ?? null,
            'vendor_name'      => $data['vendor_name'] ?? null,
            'remarks'          => $data['remarks'] ?? null,
            'status'           => 'Draft',
            'current_step_no'  => 0,
        ]);

        // ======================
        // SIMPAN item_user_inputs (BULK)
        // ======================
        $payloads = [];

        foreach ($rows as $row) {

            // skip baris kosong total
            if (
                empty($row['storage_location']) &&
                empty($row['code']) &&
                empty($row['material_description'])
            ) {
                continue;
            }

            // ======================
            // NORMALISASI NILAI
            // ======================
            $qtyMbc = is_numeric($row['qty_mbc'] ?? null)
                ? (float) $row['qty_mbc']
                : null;

            $actual = is_numeric($row['actual'] ?? null)
                ? (float) $row['actual']
                : null;

            $selisih = (!is_null($qtyMbc) && !is_null($actual))
                ? $actual - $qtyMbc
                : null;

            $payloads[] = [
                'item_request_id'      => $req->id,
                'storage_location'     => $row['storage_location'] ?? null,
                'code'                 => $row['code'] ?? null,
                'material_description' => $row['material_description'] ?? null,

                // ⬇️ FIELD BARU
                'qty_mbc'              => $qtyMbc,
                'actual'               => $actual,
                'selisih'              => $selisih,

                'base_unit_of_measure' => $row['base_unit_of_measure'] ?? null,
                'note'                 => $row['note'] ?? null,
                'created_at'           => now(),
                'updated_at'           => now(),
            ];
        }

        if (empty($payloads)) {
            throw new \Exception('No valid rows found in Excel data.');
        }

        ItemUserInput::insert($payloads);
    });

    return redirect()
        ->route('approvals.show', $req->id)
        ->with('success', 'Request created and Excel data imported successfully');
}

public function deleteatt(Request $request)
{
    $reqId = $request->input('item_request_id');
    $attId = $request->input('attachment_id');
    Log::error($request);
    $attachment = ItemRequestAttachment::where('id', $attId)
        ->where('item_request_id', $reqId)
        ->first();

    if(!$attachment){
        return response()->json(['success'=>false,'message'=>'Attachment not found']);
    }

    // hapus file fisik
    $fullPath = storage_path('app/public/' . $attachment->path);
    if(file_exists($fullPath)) unlink($fullPath);

    // hapus record DB
    $attachment->delete();

    return response()->json([
        'success' => true,
        'message' => "Attachment '{$attachment->original_name}' has been deleted."
    ]);
}

    public function itemsopname($id)
    {
        $req = ItemRequest::findOrFail($id);

        $items = ItemUserInput::where('item_request_id', $id)
            ->orderBy('storage_location')
            ->paginate(50);

        return view('approvals.items_opname', compact('req', 'items'));
    }

    public function processSingleLocationStep2($id, $loc, BusinessCentralService $bc)
    {
        try {
            $schedule = OpnameSchedule::findOrFail($id);
            $periode  = $schedule->periode ?? date('Y-m');

            // 🔹 ambil data BC (KUMULATIF sampai latest)
            $data = $bc->getStockOpnameItems($loc);

            if (isset($data['error'])) {
                throw new \Exception($data['error']);
            }

            if (!is_array($data) || empty($data)) {
                throw new \Exception("Data BC kosong untuk lokasi {$loc}");
            }

            foreach ($data as $item) {

                ItemBeforeS2::updateOrCreate(
                    [
                        'location' => $item['LocationCode'] ?? $loc,
                        'periode'  => $periode,
                        'item_no'  => $item['ItemNo'],
                    ],
                    [
                        
                        'item_description'  => $item['Description'] ?? null,
                        'uom'               => $item['UoMCode'] ?? null,
                        'unit_cost'         => $item['UnitCost'] ?? 0,

                        // ⚠️ tetap simpan sebagai snapshot terbaru
                        'opening_balance'   => $item['Qty'] ?? 0,

                        // 🔥 TOTAL KUMULATIF dari BC (untuk delta nanti)
                        'last_qty_increase' => $item['QtyIncrease'] ?? 0,
                        'last_qty_decrease' => $item['QtyDecrease'] ?? 0,

                        'remarks'           => 'STEP 2 – latest snapshot',
                    ]
                );
            }

            return response()->json([
                'status'  => 'success',
                'message' => "STEP 2 berhasil untuk lokasi {$loc}",
                'count'   => count($data),
            ]);

        } catch (\Throwable $e) {
            return response()->json([
                'status'  => 'error',
                'message' => $e->getMessage(),
            ], 500);
        }
    }

    public function rollbackOpnameStep2($id)
    {
        $schedule = OpnameSchedule::findOrFail($id);

        ItemBeforeS2::where('periode', $schedule->periode)
            ->where('location', 'like', $schedule->location . '%')
            ->delete();

        return response()->json([
            'status'  => 'success',
            'message' => 'Rollback STEP 2 selesai',
        ]);
    }

}