<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\ItemRequest;
use App\Models\ItemRequestComment;
use Illuminate\Http\Request;
use App\Support\CommentNotifier;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class ItemRequestCommentController extends Controller
{
    public function store(Request $request, $requestId)
    {

        
        // Ambil user dari session paling pertama
        $currentUser = $this->resolveCurrentUser();

        // Validasi input
        $request->validate([
            'body' => 'required|string|max:2000',
        ]);

        // Cari request utama + load relasi
        $itemRequest = ItemRequest::query()
            ->with([
                'requester:id,name,email',
                'flow:id,name,version',
                'instance:id,item_request_id,status,started_at',
            ])
            ->findOrFail($requestId);
        $requester = $itemRequest->requester;
        $requesterId = $requester->id;
        $userTo = $requester->email;
    if ($requesterId) {
    $requesterUser = User::find($requesterId);

    if ($requesterUser) {
        // Ambil email notifikasi dari User model
        $userTo = $requesterUser->routeNotificationForMail(null);

        // Log email yang didapat
        Log::info("Notification email for requester_id {$requesterId}: " . json_encode($userTo));
    } else {
        Log::warning("Requester not found with ID: {$requesterId}");
    }
}
    
        $comment = ItemRequestComment::create([
            'item_request_id' => $itemRequest->id,
            'user_id' => auth()->id(),
            'body' => $request->input('body'),
        ]);
        $comment_send = ([
        'item_request_id' => $itemRequest->id,
        'notif_email'  => $userTo[0],
        'user_id'         => $currentUser->id,
        'user_name'       => $currentUser->name,
        'body'            => $request->input('body'),
        'target_email'    => $requester->email,
        ]);
        

        // Kirim notifikasi email ke pemilik request
        CommentNotifier::send($comment_send);

        // Redirect balik ke halaman detail
        return redirect()
            ->route('approvals.show', $itemRequest->id)
            ->with('success', 'Comment added.');
    }

    protected function resolveCurrentUser(): User
    {
        $sess  = session('user') ?? [];
        $email = $sess['email'] ?? null;
        $name  = $sess['name'] ?? ($sess['displayName'] ?? 'MS User');

        abort_unless($email, 401, 'Not authenticated');

        return User::firstOrCreate(['email' => $email], ['name' => $name]);
    }

    public function getUnreadCount()
    {
        $userId = auth()->id();

        $requests = ItemRequest::where('requester_id', $userId)
            ->with(['comments' => function ($q) {
                $q->latest('created_at')->take(1);
            }])
            ->get();

        $unreadCount = $requests->reduce(function ($count, $req) {
            $latestComment = $req->comments->first();
            if ($latestComment && (
                !$req->last_comment_seen_at ||
                $latestComment->created_at > $req->last_comment_seen_at
            )) {
                return $count + 1;
            }
            return $count;
        }, 0);

        return response()->json(['unread' => $unreadCount]);
    }

    // 🧾 Ambil komentar (lazy-load, 5 per page)
public function fetchComments(Request $request)
{
    $userId = auth()->id();

    // Ambil semua komentar dari request milik user (tanpa limit)
    $comments = ItemRequestComment::with(['user', 'request'])
        ->whereHas('request', function ($q) use ($userId) {
            $q->where('requester_id', $userId);
        })
        ->latest('created_at')
        ->get();

    return response()->json(['comments' => $comments]);
}


    // ✅ Tandai semua komentar sudah dibaca
    public function markAllAsRead()
    {
        $userId = auth()->id();

        $now = now();
        ItemRequest::where('requester_id', $userId)->update(['last_comment_seen_at' => $now]);

        return response()->json(['status' => 'ok', 'timestamp' => $now]);
    }
}

