<?php

namespace App\Http\Controllers;
use App\Models\ItemUserInput;
use App\Models\ItemAfter;;
use App\Models\ItemBeforeS1;
use App\Models\ItemBeforeS2;
use App\Models\ItemRequest;
use App\Models\OpnameSchedule;
use App\Models\Location;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;
use App\Services\BusinessCentralService;
use Illuminate\Http\JsonResponse;
use App\Models\GenBus;
use App\Models\Department;
use Illuminate\Support\Facades\Log;

class ItemAfterController extends Controller
{
    public function index($itemRequestId)
    {
        $request = ItemRequest::findOrFail($itemRequestId);

        // 🔹 ambil lokasi request (contoh: RBC.2100)
        $locationCode = $request->location;

        $location = Location::where('code', $locationCode)->first();

        // ============================
        // DATA STEP 1 & STEP 2
        // ============================
        $s1 = ItemBeforeS1::where('location', $locationCode)->get()
            ->keyBy('item_no');

        $s2 = ItemBeforeS2::where('location', $locationCode)->get()
            ->keyBy('item_no');

        // ============================
        // USER INPUT (BLIND COUNT)
        // ============================
        $blind = ItemUserInput::where('item_request_id', $itemRequestId)
            ->get()
            ->keyBy('code'); // code = item_no
        $rows = [];

        foreach ($s1 as $itemNo => $rowS1) {

            $rowS2 = $s2[$itemNo] ?? null;

            $inc1 = $rowS1->last_qty_increase ?? 0;
            $dec1 = $rowS1->last_qty_decrease ?? 0;

            $inc2 = $rowS2->last_qty_increase ?? 0;
            $dec2 = $rowS2->last_qty_decrease ?? 0;

            // 🔥 DELTA LOGIC
            $increase = max(0, $inc2 - $inc1);
            $decrease = min(0, $dec2 - $dec1);

            $opening = $rowS1->opening_balance;
            $ending  = $rowS2->opening_balance ?? $opening;

            $blindQty = $blind[$itemNo]->actual ?? null;

            // ✅ Variance hanya dihitung jika blind ada nilainya
            if ($blindQty === null || $blindQty === '') {
                $variance = 0;
            } else {
                $variance = $blindQty - $ending;
            }

            $totalCost = $variance * $rowS1->unit_cost;

            $rows[] = [
                'item_no' => $itemNo,
                'location' => $locationCode,
                'item_description' => $rowS1->item_description,
                'uom' => $rowS1->uom,
                'unit_cost' => $rowS1->unit_cost,
                'opening_balance' => $opening,
                'increase' => $increase,
                'decrease' => $decrease,
                'ending_balance' => $ending,
                'blind_count' => $blindQty,
                'variance' => $variance,
                'total_cost' => $totalCost,
                'remarks' => null
            ];
        }

        return view('item_after.process', [
            'request' => $request,
            'location' => $location,
            'rows' => $rows
        ]);
    }


    public function store(Request $request, $itemRequestId)
    {
        $items = $request->input('items', []);

        if (empty($items)) {
            return redirect()
                ->route('approvals.show', $itemRequestId)
                ->with('warning', 'Tidak ada data untuk disimpan.');
        }

        DB::transaction(function () use ($items) {

            $now = now();
            $batchSize = 1000;

            // ==============================
            // NORMALISASI DATA
            // ==============================
            $payload = [];

            foreach ($items as $row) {
                $payload[] = [
                    'item_no'          => $row['item_no'],
                    'location'         => $row['location'],
                    'item_description' => $row['item_description'] ?? null,
                    'uom'              => $row['uom'] ?? null,
                    'unit_cost'        => $row['unit_cost'] ?? 0,
                    'opening_balance'  => $row['opening_balance'] ?? 0,
                    'increase'         => $row['increase'] ?? 0,
                    'decrease'         => $row['decrease'] ?? 0,
                    'ending_balance'   => $row['ending_balance'] ?? 0,
                    'blind_count'      => $row['blind_count'] ?? 0,
                    'variance'         => $row['variance'] ?? 0,
                    'total_cost'       => $row['total_cost'] ?? 0,
                    'remarks'          => $row['remarks'] ?? null,
                    'created_at'       => $now,
                    'updated_at'       => $now,
                ];
            }

            // ==============================
            // BATCH UPSERT (1000 / batch)
            // ==============================
            foreach (array_chunk($payload, $batchSize) as $chunk) {
                DB::table('item_after')->upsert(
                    $chunk,
                    // UNIQUE KEY
                    ['item_no', 'location'],
                    // UPDATE COLUMNS
                    [
                        'item_description',
                        'uom',
                        'unit_cost',
                        'opening_balance',
                        'increase',
                        'decrease',
                        'ending_balance',
                        'blind_count',
                        'variance',
                        'total_cost',
                        'remarks',
                        'updated_at',
                    ]
                );
            }
        });

        return redirect()
            ->route('approvals.show', $itemRequestId)
            ->with('success', 'Item After berhasil disimpan.');
    }

    public function ajaxStore(Request $request, $itemRequestId)
    {
        $items = $request->items ?? [];

        // 🔥 periode otomatis (YYYY-MM)
        $periode = now()->format('Y-m');

        DB::transaction(function () use ($items, $periode) {

            $now = now();

            foreach (array_chunk($items, 1000) as $chunk) {
                DB::table('item_after')->upsert(
                    collect($chunk)->map(fn($r) => [
                        ...$r,
                        'periode'     => $periode, // ⬅️ TAMBAHAN
                        'created_at'  => $now,
                        'updated_at'  => $now,
                    ])->toArray(),
                    // ⛔ WAJIB: periode ikut jadi key agar tidak overwrite periode lain
                    ['item_no', 'location', 'periode'],
                    [
                        'item_description',
                        'uom',
                        'unit_cost',
                        'opening_balance',
                        'increase',
                        'decrease',
                        'ending_balance',
                        'blind_count',
                        'variance',
                        'total_cost',
                        'remarks',
                        'updated_at'
                    ]
                );
            }
        });

        return response()->json([
            'status' => 'success',
            'count'  => count($items),
            'periode'=> $periode
        ]);
    }


    public function showReceiptShipment(int $id)
    {
        // ====================================================
        // 1. LOAD REQUEST (SOURCE OF LOCATION & DATE)
        // ====================================================
        $req = ItemRequest::findOrFail($id);

        $location = $req->location;

        // requested_at → YYYY-MM
        $periode = Carbon::parse($req->requested_at)->format('Y-m');

        // ====================================================
        // 2. LOAD ITEM AFTER (LOCATION + PERIODE)
        // ====================================================
        $items = ItemAfter::where('location', $location) // sesuai DB
            ->where('periode', $periode)
            ->orderBy('item_no')
            ->get();

        // ====================================================
        // 3. SPLIT BY VARIANCE
        // ====================================================
        $receiptItems  = $items->filter(fn ($i) => $i->variance > 0);
        $shipmentItems = $items->filter(fn ($i) => $i->variance < 0);

        // ====================================================
        // 4. LOAD GEN BUS MASTER
        // ====================================================
        $genBusList = GenBus::orderBy('code')->get();

        $departmentList = Department::orderBy('code')->get();

        return view('item_after.inventory-receipt-ship', [
            'req'           => $req,
            'location'      => $location,
            'periode'       => $periode,
            'receiptItems'  => $receiptItems,
            'shipmentItems' => $shipmentItems,
            'genBusList'    => $genBusList,
            'departmentList' => $departmentList, // ✅ NEW
        ]);
    }


public function processInventoryFromPage(int $id, Request $request): JsonResponse
{
    $data = $request->validate([
        'location_code'  => 'required|string',
        'posting_date'   => 'required|date',
        'gen_bus_code'   => 'required|string',
        'department_code'=> 'required|string', // ✅ NEW

        'receipt'                    => 'nullable|array',
        'receipt.no_series'          => 'required_with:receipt|string',
        'receipt.lines'              => 'required_with:receipt|array|min:1',
        'receipt.lines.*.item_no'    => 'required|string',
        'receipt.lines.*.quantity'   => 'required|numeric|min:0.0001',

        'shipment'                   => 'nullable|array',
        'shipment.no_series'         => 'required_with:shipment|string',
        'shipment.lines'             => 'required_with:shipment|array|min:1',
        'shipment.lines.*.item_no'   => 'required|string',
        'shipment.lines.*.quantity'  => 'required|numeric|min:0.0001',
    ]);

    $bc = new BusinessCentralService();

    $result = [
        'receipt'  => null,
        'shipment' => null,
    ];

    // ===============================
    // AUTO BUSINESS UNIT FROM LOCATION
    // ===============================
    $businessUnit = strtoupper(explode('.', $data['location_code'])[0]);

    try {

        // ===============================
        // INVENTORY RECEIPT
        // ===============================
        if (!empty($data['receipt'])) {
            $result['receipt'] = $bc->createInventoryReceipt([
                'no_series'          => $data['receipt']['no_series'],
                'location_code'     => $data['location_code'],
                'gen_bus_code'      => $data['gen_bus_code'],
                'business_unit_code'=> $businessUnit,
                'department_code'   => $data['department_code'],
                'posting_date'      => $data['posting_date'],
                'lines'             => $data['receipt']['lines'],
            ]);
        }

        // ===============================
        // INVENTORY SHIPMENT
        // ===============================
        if (!empty($data['shipment'])) {
            $result['shipment'] = $bc->createInventoryShipment([
                'no_series'          => $data['shipment']['no_series'],
                'location_code'     => $data['location_code'],
                'gen_bus_code'      => $data['gen_bus_code'],
                'business_unit_code'=> $businessUnit,
                'department_code'   => $data['department_code'],
                'posting_date'      => $data['posting_date'],
                'lines'             => $data['shipment']['lines'],
            ]);
        }

        /**
         * ====================================================
         * MARK ITEM REQUEST AS SYNCED TO BC
         * ONLY AFTER ALL PROCESS SUCCESS
         * ====================================================
         */
        ItemRequest::where('id', $id)->update([
            'up_to_bc' => 'yes',
        ]);

        return response()->json([
            'status' => 'success',
            'data'   => $result
        ]);

    } catch (\Throwable $e) {

        // ❌ JANGAN UPDATE up_to_bc JIKA GAGAL
        Log::error('[Inventory] processInventoryFromPage failed', [
            'item_request_id' => $id,
            'message'         => $e->getMessage(),
        ]);

        return response()->json([
            'status'  => 'error',
            'message' => $e->getMessage(),
        ], 500);
    }
}



}
