<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use App\Models\{
    Role, User, ItemRequest, ItemRequestLine,
    ApprovalFlowTemplate, ApprovalStepTemplate,
    ApprovalInstance, ApprovalAction, Company
};
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;

class AdminController extends Controller
{
    public function index(Request $request)
    {
        $tab = $request->query('tab', 'flows');

        $data = ['tab' => $tab];

        if ($tab === 'users') {
            $data['users'] = User::with(['companies','role'])->orderBy('name')->paginate(20);
            $data['roles'] = Role::orderBy('code')->get();
            $data['companies'] = Company::orderBy('code')->get();
        } elseif ($tab === 'instances') {
            $data['instances'] = ApprovalInstance::with(['itemRequest.requester','flow'])
                ->orderByDesc('id')->paginate(20);
        } elseif ($tab === 'requests') {
            $data['requests'] = ItemRequest::with(['requester','flow','instance'])
                ->orderByDesc('id')->paginate(20);
        }elseif ($tab === 'roles') {
            $data['roles'] = Role::orderBy('code')->paginate(20);
        }
        else { // flows
            $data['flows'] = ApprovalFlowTemplate::with([
                'steps.approvers.user',
                'steps.approvers.role'
            ])->orderBy('object_type')->orderByDesc('version')->get();
            $data['users'] = User::orderBy('name')->get();
            $data['roles'] = Role::orderBy('code')->get();
        }

        return view('admin.index', $data);
    }

    public function storeFlow(Request $r)
    {
        $payload = $r->validate([
            'name'        => ['required','string','max:200'],
            'object_type' => ['required','string','max:100'],
            'version'     => ['required','integer','min:1'],
            'is_active'   => ['nullable','boolean'],
        ]);

        $payload['is_active'] = (bool)($payload['is_active'] ?? true);

        ApprovalFlowTemplate::create($payload);

        return back()->with('success','Flow template created.')->withFragment('flows');
    }

    public function toggleFlow(ApprovalFlowTemplate $flow)
    {
        $flow->is_active = !$flow->is_active;
        $flow->save();

        return back()->with('success', 'Flow toggled.')->withFragment('flows');
    }

    public function destroyFlow(ApprovalFlowTemplate $flow)
    {
        $flow->delete();
        return back()->with('success','Flow deleted.')->withFragment('flows');
    }

    public function storeStep(Request $r, ApprovalFlowTemplate $flow)
    {
        $payload = $r->validate([
            'step_no'            => ['nullable','integer','min:1'],
            'approver_user_ids'  => ['nullable','array'],
            'approver_user_ids.*'=> ['nullable','integer','exists:users,id'],
            'approver_role_ids'  => ['nullable','array'],
            'approver_role_ids.*'=> ['nullable','integer','exists:roles,id'],
            'require_all'        => ['nullable','boolean'],
        ]);
    
        $userIds = collect($payload['approver_user_ids'] ?? [])->filter()->map(fn($v) => (int)$v)->values()->all();
        $roleIds = collect($payload['approver_role_ids'] ?? [])->filter()->map(fn($v) => (int)$v)->values()->all();
    
        if (empty($userIds) && empty($roleIds)) {
            return back()->withErrors(['approver_user_ids' => 'Pick at least one user or role'])->withFragment('flows');
        }
    
        if (empty($payload['step_no'])) {
            $max = ApprovalStepTemplate::where('flow_template_id', $flow->id)->max('step_no');
            $payload['step_no'] = ($max ?? 0) + 1;
        }
    
        // create step template (legacy columns kept null)
        $step = ApprovalStepTemplate::create([
            'flow_template_id' => $flow->id,
            'step_no' => (int)$payload['step_no'],
            'approver_user_id' => null,
            'approver_role_id' => null,
            'require_all' => (bool)($payload['require_all'] ?? false),
        ]);
    
        // persist approver records
        foreach ($userIds as $uid) {
            \App\Models\ApprovalStepApprover::create([
                'step_template_id' => $step->id,
                'approver_user_id' => $uid,
                'approver_role_id' => null,
            ]);
        }
        foreach ($roleIds as $rid) {
            \App\Models\ApprovalStepApprover::create([
                'step_template_id' => $step->id,
                'approver_user_id' => null,
                'approver_role_id' => $rid,
            ]);
        }
    
        return back()->with('success','Step added.')->withFragment('flows');
    }

    public function destroyStep(ApprovalStepTemplate $step)
    {
        $step->approvers()->delete();
        $step->delete();
        return back()->with('success','Step removed.')->withFragment('flows');
    }

    public function updateUser(Request $r, User $user)
    {
        $validated = $r->validate([
            'id'           => ['prohibited'],
            'name'         => ['required','string','max:255'],
            'notify_email' => ['nullable','string','max:255'], 
            'role_id'      => ['nullable','exists:roles,id'],
    
            'company_ids'   => ['nullable','array'],
            'company_ids.*' => ['integer','exists:companies,id'],
    
            'company_codes' => ['nullable','string'],
        ]);
    
        $user->fill(Arr::only($validated, ['name','notify_email','role_id']))->save();
    
        $ids = collect($validated['company_ids'] ?? [])->filter()->map(fn($v) => (int)$v)->all();
    
        if (!empty($validated['company_codes'])) {
            $codes = collect(preg_split('/[,\s;]+/', $validated['company_codes'], -1, PREG_SPLIT_NO_EMPTY))
                        ->map(fn($c) => Str::upper(trim($c)))->all();
            $byCode = \App\Models\Company::whereIn('code', $codes)->pluck('id')->all();
            $ids = array_values(array_unique(array_merge($ids, $byCode)));
        }
    
        if (!empty($ids)) {
            $user->companies()->sync($ids);
        } else {
            // If the intent is "no companies", uncomment this to clear them:
            // $user->companies()->sync([]);
        }

        return back()->with('success','User updated.')->withFragment('users');
    }

    public function instances()
    {
        $instances = ApprovalInstance::with(['itemRequest.requester','flow'])
            ->orderByDesc('id')->paginate(30);
        return view('admin.index', ['tab' => 'instances', 'instances' => $instances]);
    }

    public function requests()
    {
        $requests = ItemRequest::with(['requester','flow','instance'])
            ->orderByDesc('id')->paginate(30);
        return view('admin.index', ['tab' => 'requests', 'requests' => $requests]);
    }

    public function storeUser(Request $r)
    {
        $data = $r->validate([
            'name'       => ['required','string','max:255'],
            'email'      => ['required','email','max:255'],
            'role_id'    => ['nullable','exists:roles,id'],
            'notify_email'=> ['nullable','string','max:500'],
            'company_ids'   => ['nullable','array'],
            'company_ids.*' => ['integer','exists:companies,id'],
        ]);
        

        $notify = trim((string)($data['notify_email'] ?? ''));
        if ($notify !== '') {
            $parts = preg_split('/[,\s;]+/', $notify, -1, PREG_SPLIT_NO_EMPTY) ?: [];
            $parts = collect($parts)
                ->map(fn($e) => mb_strtolower(trim($e)))
                ->filter(fn($e) => filter_var($e, FILTER_VALIDATE_EMAIL))
                ->unique()
                ->values()  
                ->all();
            $notify = implode(', ', $parts);
        } else {
            $notify = null;
        }


        // Upsert by email (so re-adding won’t fail on unique email)
        $user = \App\Models\User::firstOrNew(['email' => $data['email']]);
        $user->name       = $data['name'];
        $user->role_id    = $data['role_id'] ?? null;
        $user->notify_email= $notify;  
        $user->save();
        
        $ids = collect($data['company_ids'] ?? [])->map(fn($v)=>(int)$v)->filter()->all();
        $user->companies()->sync($ids);

        return back()->with('success', 'User added/updated.')->withFragment('users');
    }

    public function destroyUser(Request $r, \App\Models\User $user)
    {
        // prevent deleting yourself
        $me = $r->attributes->get('me'); // set by AdminOnly middleware
        if ($me && $me->id === $user->id) {
            return back()->withErrors(['You cannot delete your own account.'])->withFragment('users');
        }
        $user->delete();
        return back()->with('success','User deleted.')->withFragment('users');
    }

    public function requestResetDraft($id)
    {
        DB::transaction(function () use ($id) {
            $req = ItemRequest::with('instance.actions','lines')->findOrFail($id);
            if ($req->instance) {
                $req->instance->actions()->delete();
                $req->instance()->delete();
            }
            $req->update(['status'=>'Draft','current_step_no'=>0]);
        });
        return back()->with('success','Request reset to Draft.')->withFragment('requests');
    }

    public function requestJumpStep($id, Request $r)
    {
        $target = (int)$r->validate(['step'=>'required|integer|min:1'])['step'];

        DB::transaction(function () use ($id,$target) {
            $req = ItemRequest::with('instance')->findOrFail($id);

            $max = (int) ApprovalStepTemplate::where('flow_template_id',$req->flow_template_id)->max('step_no');
            abort_if($target > $max, 422, "Target step > max ($max).");

            $ins = $req->instance ?: ApprovalInstance::create([
                'item_request_id' => $req->id,
                'flow_template_id'=> $req->flow_template_id,
                'status'          => 'InProgress',
                'current_step_no' => 1,
                'started_at'      => now(),
            ]);

            $ins->update(['status'=>'InProgress','current_step_no'=>$target,'completed_at'=>null]);
            $req->update(['status'=>'InReview','current_step_no'=>$target]);

            ApprovalAction::create([
                'approval_instance_id'=>$ins->id,
                'step_no'       => $target,
                'actor_user_id' => $r->attributes->get('me')->id,
                'action'        => 'Submit',
                'comment'       => "[ADMIN] moved to step {$target}",
                'acted_at'      => now(),
            ]);
        });

        return back()->with('success',"Moved to step {$target}.")->withFragment('requests');
    }

    public function requestForceApprove($id, Request $r)
    {
        DB::transaction(function () use ($id,$r) {
            $req = ItemRequest::with('instance')->findOrFail($id);
            $ins = $req->instance ?: ApprovalInstance::create([
                'item_request_id'=>$req->id,
                'flow_template_id'=>$req->flow_template_id,
                'status'=>'InProgress',
                'current_step_no'=> max(1,(int)$req->current_step_no),
                'started_at'=> now(),
            ]);

            ApprovalAction::create([
                'approval_instance_id'=>$ins->id,
                'step_no'       => max(1,(int)$req->current_step_no),
                'actor_user_id' => $r->attributes->get('me')->id,
                'action'        => 'Approve',
                'comment'       => trim(($r->input('comment') ?? '').' [ADMIN force-approve]'),
                'acted_at'      => now(),
            ]);

            $ins->update(['status'=>'Approved','completed_at'=>now()]);
            $req->update(['status'=>'Approved']);
        });

        return back()->with('success','Force-approved.')->withFragment('requests');
    }

    public function requestForceReject($id, Request $r)
    {
        DB::transaction(function () use ($id,$r) {
            $req = ItemRequest::with('instance')->findOrFail($id);
            $ins = $req->instance ?: ApprovalInstance::create([
                'item_request_id'=>$req->id,
                'flow_template_id'=>$req->flow_template_id,
                'status'=>'InProgress',
                'current_step_no'=> max(1,(int)$req->current_step_no),
                'started_at'=> now(),
            ]);

            ApprovalAction::create([
                'approval_instance_id'=>$ins->id,
                'step_no'       => max(1,(int)$req->current_step_no),
                'actor_user_id' => $r->attributes->get('me')->id,
                'action'        => 'Reject',
                'comment'       => trim(($r->input('comment') ?? '').' [ADMIN force-reject]'),
                'acted_at'      => now(),
            ]);

            $ins->update(['status'=>'Rejected','completed_at'=>now()]);
            $req->update(['status'=>'Rejected']);
        });

        return back()->with('success','Force-rejected.')->withFragment('requests');
    }

    public function requestDestroy($id)
    {
        DB::transaction(function () use ($id) {
            $req = ItemRequest::with('instance.actions','lines')->findOrFail($id);
            if ($req->instance) {
                $req->instance->actions()->delete();
                $req->instance()->delete();
            }
            $req->lines()->delete();
            $req->delete();
        });
        return back()->with('success','Request deleted.')->withFragment('requests');
    }
    
    public function storeRole(Request $request)
    {
        $data = $request->validate([
            'code' => 'required|string|unique:roles,code',
            'name' => 'required|string',
        ]);

        Role::create($data);

        return back()->with('success', 'Role created.');
    }

    public function updateRole(Request $request, Role $role)
    {
        $data = $request->validate([
            'code' => 'required|string|unique:roles,code,'.$role->id,
            'name' => 'required|string',
        ]);

        $role->update($data);

        return back()->with('success', 'Role updated.');
    }

    public function deleteRole(Role $role)
    {
        $role->delete();
        return back()->with('success', 'Role deleted.');
    }

    public function businessUnitPage()
    {
        $rbc = User::where('business_unit', 'RBC')->get();
        $hic = User::where('business_unit', 'HIC')->get();
        $ci  = User::where('business_unit', 'CI')->get();
        $empty = User::whereNull('business_unit')->orWhere('business_unit', '')->get();

        return view('admin.business-unit', compact('rbc','hic','ci','empty'));
    }

    public function updateBusinessUnit(Request $request)
    {
        $request->validate([
            'user_id' => 'required|integer',
            'unit'    => 'nullable|string' // null untuk remove
        ]);

        $user = User::find($request->user_id);
        if (!$user) return response()->json(['error' => 'User not found'], 404);

        $user->business_unit = $request->unit;
        $user->save();

        return response()->json(['success' => true, 'unit' => $user->business_unit]);
    }
}