@extends('layouts.hirbc')

@section('title', 'Manage Data')
@section('page-title', 'Manage Data')

@push('styles')
<style>
    .page-header {
        background: linear-gradient(135deg, #D3D3FF, #B8B8FF);
        color: #333;
        border-radius: 12px;
        padding: 32px;
        margin-bottom: 32px;
        box-shadow: 0 4px 12px rgba(211, 211, 255, 0.3);
    }

    .page-header h1 {
        font-size: 32px;
        margin-bottom: 8px;
        font-weight: 700;
    }

    .page-header p {
        font-size: 16px;
        opacity: 0.8;
    }

    .info-banner {
        background: #fff3cd;
        border-left: 4px solid #ffc107;
        padding: 16px 20px;
        border-radius: 8px;
        margin-bottom: 24px;
    }

    .info-banner strong {
        color: #856404;
        display: block;
        margin-bottom: 8px;
    }

    .info-banner p {
        color: #856404;
        font-size: 14px;
        margin: 0;
    }

    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 24px;
        margin-bottom: 32px;
    }

    .stat-card {
        background: white;
        padding: 24px;
        border-radius: 12px;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
        text-align: center;
    }

    .stat-icon {
        width: 60px;
        height: 60px;
        background: linear-gradient(135deg, #D3D3FF, #B8B8FF);
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 28px;
        color: #9494E8;
        margin: 0 auto 16px;
    }

    .stat-label {
        font-size: 14px;
        color: #666;
        margin-bottom: 8px;
    }

    .stat-value {
        font-size: 32px;
        font-weight: 700;
        color: #333;
    }

    .action-section {
        background: white;
        padding: 32px;
        border-radius: 12px;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
        margin-bottom: 32px;
        text-align: center;
    }

    .action-section h2 {
        font-size: 24px;
        margin-bottom: 16px;
        color: #333;
    }

    .action-section p {
        color: #666;
        margin-bottom: 24px;
    }

    .btn-refresh {
        background: linear-gradient(135deg, #D3D3FF, #B8B8FF);
        color: #333;
        border: none;
        padding: 16px 48px;
        border-radius: 10px;
        font-size: 18px;
        font-weight: 700;
        cursor: pointer;
        transition: all 0.3s;
        display: inline-flex;
        align-items: center;
        gap: 12px;
    }

    .btn-refresh:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 16px rgba(211, 211, 255, 0.5);
    }

    .btn-refresh:disabled {
        opacity: 0.6;
        cursor: not-allowed;
    }

    .months-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
        gap: 16px;
        margin-bottom: 32px;
    }

    .month-card {
        background: white;
        padding: 20px;
        border-radius: 10px;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
        text-align: center;
        transition: all 0.3s;
    }

    .month-card.has-data {
        background: linear-gradient(135deg, #D3D3FF, #B8B8FF);
        color: #333;
    }

    .month-card.no-data {
        background: #f8f9fa;
        color: #999;
    }

    .month-name {
        font-size: 16px;
        font-weight: 600;
        margin-bottom: 8px;
    }

    .month-count {
        font-size: 24px;
        font-weight: 700;
    }

    .data-source {
        background: white;
        padding: 24px;
        border-radius: 12px;
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
    }

    .data-source h3 {
        font-size: 18px;
        margin-bottom: 16px;
        color: #333;
    }

    .data-source-item {
        display: flex;
        justify-content: space-between;
        padding: 12px 0;
        border-bottom: 1px solid #eee;
    }

    .data-source-item:last-child {
        border-bottom: none;
    }

    .loading-overlay {
        display: none;
        position: fixed;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(0, 0, 0, 0.7);
        z-index: 9999;
        align-items: center;
        justify-content: center;
    }

    .loading-overlay.active {
        display: flex;
    }

    .loading-content {
        background: white;
        padding: 40px;
        border-radius: 12px;
        text-align: center;
        max-width: 400px;
    }

    .spinner {
        border: 4px solid #f3f3f3;
        border-top: 4px solid #D3D3FF;
        border-radius: 50%;
        width: 60px;
        height: 60px;
        animation: spin 1s linear infinite;
        margin: 0 auto 20px;
    }

    @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
    }
</style>
@endpush

@section('content')
    <!-- Page Header -->
    <div class="page-header">
        <h1><i class="fas fa-database"></i> Manage Data</h1>
        <p>Load and refresh room data from Business Central</p>
    </div>

    <!-- Info Banner -->
    <div class="info-banner">
        <strong><i class="fas fa-info-circle"></i> Important Information</strong>
        <p>Data refresh process may take 2-5 minutes depending on the amount of data. Please do not close this page during the process.</p>
    </div>

    <!-- Statistics -->
    <div class="stats-grid">
        <div class="stat-card">
            <div class="stat-icon">
                <i class="fas fa-database"></i>
            </div>
            <div class="stat-label">Total Records</div>
            <div class="stat-value">{{ number_format($stats['total_records'] ?? 0) }}</div>
        </div>

        <div class="stat-card">
            <div class="stat-icon">
                <i class="fas fa-tools"></i>
            </div>
            <div class="stat-label">OOO Records</div>
            <div class="stat-value">{{ $stats['ooo_records'] ?? 0 }}</div>
        </div>

        <div class="stat-card">
            <div class="stat-icon">
                <i class="fas fa-calendar-check"></i>
            </div>
            <div class="stat-label">Months Loaded</div>
            <div class="stat-value">{{ $stats['months_loaded'] ?? 0 }}/12</div>
        </div>
    </div>

    <!-- Action Section -->
    <div class="action-section">
        <h2><i class="fas fa-sync-alt"></i> Refresh Data from Business Central</h2>
        <p>Click the button below to fetch and update all data from Business Central API</p>
        <button class="btn-refresh" onclick="refreshData()">
            <i class="fas fa-sync-alt"></i>
            <span>Refresh Data Now</span>
        </button>
    </div>

    <!-- Monthly Data Grid -->
    <div class="months-grid">
        @foreach(['january', 'february', 'march', 'april', 'may', 'june', 'july', 'august', 'september', 'october', 'november', 'december'] as $month)
            @php
                $monthData = $stats['monthly_data'][$month] ?? ['count' => 0, 'has_data' => false];
            @endphp
            <div class="month-card {{ $monthData['has_data'] ? 'has-data' : 'no-data' }}">
                <div class="month-name">{{ ucfirst($month) }}</div>
                <div class="month-count">
                    @if($monthData['has_data'])
                        <i class="fas fa-check-circle"></i>
                    @else
                        <i class="fas fa-times-circle"></i>
                    @endif
                </div>
            </div>
        @endforeach
    </div>

    <!-- Data Source Info -->
    <div class="data-source">
        <h3><i class="fas fa-cloud"></i> Data Source Information</h3>
        <div class="data-source-item">
            <span>Business Unit:</span>
            <strong>HIRBC</strong>
        </div>
        <div class="data-source-item">
            <span>Total Tables:</span>
            <strong>13 (12 Monthly + 1 OOO)</strong>
        </div>
        <div class="data-source-item">
            <span>Room Categories:</span>
            <strong>2 (Standard & Suite)</strong>
        </div>
        <div class="data-source-item">
            <span>Total Rooms:</span>
            <strong>139 rooms</strong>
        </div>
        <div class="data-source-item">
            <span>API Source:</span>
            <strong>Microsoft Dynamics 365 Business Central</strong>
        </div>
    </div>

    <!-- Loading Overlay -->
    <div class="loading-overlay" id="loadingOverlay">
        <div class="loading-content">
            <div class="spinner"></div>
            <h3>Refreshing Data...</h3>
            <p>Please wait while we fetch data from Business Central</p>
        </div>
    </div>
@endsection

@push('scripts')
<script>
    function refreshData() {
        // Show loading overlay
        document.getElementById('loadingOverlay').classList.add('active');
        
        // Disable button
        const btn = document.querySelector('.btn-refresh');
        btn.disabled = true;
        
        // Send AJAX request
        fetch('{{ route('hirbc.manage-data.refresh') }}', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': '{{ csrf_token() }}'
            }
        })
        .then(response => response.json())
        .then(data => {
            // Hide loading overlay
            document.getElementById('loadingOverlay').classList.remove('active');
            btn.disabled = false;
            
            if (data.success) {
                alert('Success!\n\n' + data.message.replace(/<br>/g, '\n'));
                location.reload();
            } else {
                alert('Error!\n\n' + data.message.replace(/<br>/g, '\n'));
            }
        })
        .catch(error => {
            document.getElementById('loadingOverlay').classList.remove('active');
            btn.disabled = false;
            alert('Error: ' + error.message);
        });
    }
</script>
@endpush