<?php

namespace App\Services\BusinessCentral;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use Exception;

class BaseApiService
{
    protected $baseUrl;
    protected $tokenUrl;
    protected $clientId;
    protected $clientSecret;
    protected $scope;
    protected $timeout;

    public function __construct()
    {
        $this->baseUrl = config('businesscentral.base_url');
        $this->tokenUrl = config('businesscentral.oauth.token_url');
        $this->clientId = config('businesscentral.oauth.client_id');
        $this->clientSecret = config('businesscentral.oauth.client_secret');
        $this->scope = config('businesscentral.oauth.scope');
        $this->timeout = config('businesscentral.timeout', 0);

        // ✅ VALIDASI: Pastikan config ada
        if (!$this->tokenUrl) {
            throw new Exception('AZURE_TOKEN_URL is not configured in .env');
        }
        if (!$this->clientId) {
            throw new Exception('AZURE_CLIENT_ID is not configured in .env');
        }
        if (!$this->clientSecret) {
            throw new Exception('AZURE_CLIENT_SECRET is not configured in .env');
        }
    }

    /**
     * Get access token (cached for 55 minutes)
     */
    protected function getAccessToken(): string
    {
        return Cache::remember('bc_access_token', 3300, function () {
            try {
                Log::info('BC: Requesting new access token...', [
                    'token_url' => $this->tokenUrl,
                    'client_id' => $this->clientId,
                    'scope' => $this->scope
                ]);

                // ✅ FIX: Pastikan $this->tokenUrl ada sebagai parameter pertama
                $response = Http::timeout(300)
                    ->asForm()
                    ->post($this->tokenUrl, [
                        'grant_type' => 'client_credentials',
                        'client_id' => $this->clientId,
                        'client_secret' => $this->clientSecret,
                        'scope' => $this->scope,
                    ]);

                Log::info('BC: Token response received', [
                    'status' => $response->status(),
                    'successful' => $response->successful()
                ]);

                if (!$response->successful()) {
                    Log::error('BC: Token request failed', [
                        'status' => $response->status(),
                        'body' => $response->body()
                    ]);
                    throw new Exception('Failed to get access token: HTTP ' . $response->status() . ' - ' . $response->body());
                }

                $responseData = $response->json();
                $token = $responseData['access_token'] ?? null;

                if (!$token) {
                    Log::error('BC: Access token not found in response', [
                        'response' => $responseData
                    ]);
                    throw new Exception('Access token not found in response');
                }

                Log::info('BC: ✅ Access token obtained successfully');

                return $token;

            } catch (Exception $e) {
                Log::error('BC: Token request exception', [
                    'error' => $e->getMessage(),
                    'trace' => $e->getTraceAsString()
                ]);
                throw $e;
            }
        });
    }

    /**
     * Make GET request to Business Central API
     */
    protected function get(string $endpoint): array
    {
        try {
            $token = $this->getAccessToken();
            
            // ✅ FIX: Prepend base URL jika endpoint tidak dimulai dengan http
            if (!str_starts_with($endpoint, 'http://') && !str_starts_with($endpoint, 'https://')) {
                $fullUrl = rtrim($this->baseUrl, '/') . '/' . ltrim($endpoint, '/');
            } else {
                $fullUrl = $endpoint;
            }
            
            Log::info('BC API: GET Request', [
                'endpoint' => $endpoint,
                'full_url' => $fullUrl,
                'timeout' => $this->timeout
            ]);

            // ✅ PASTIKAN TIMEOUT = 0 (UNLIMITED)
            $response = Http::timeout($this->timeout ?: 0)
                ->withToken($token)
                ->withHeaders([
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ])
                ->get($fullUrl);

            Log::info('BC API: Response received', [
                'status' => $response->status(),
                'content_type' => $response->header('Content-Type'),
            ]);

            // ⚠️ CEK: Apakah response HTML atau JSON?
            $contentType = $response->header('Content-Type') ?? '';
            
            if (str_contains(strtolower($contentType), 'text/html')) {
                Log::error('BC API: Response is HTML, not JSON!', [
                    'endpoint' => $endpoint,
                    'full_url' => $fullUrl,
                    'status' => $response->status(),
                    'content_type' => $contentType,
                    'body_preview' => substr($response->body(), 0, 500)
                ]);
                throw new Exception('API returned HTML instead of JSON. Endpoint might be incorrect or authentication failed.');
            }

            if (!$response->successful()) {
                Log::error('BC API: Request failed', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                throw new Exception('API request failed: HTTP ' . $response->status() . ' - ' . $response->body());
            }

            $data = $response->json();

            // Handle OData response format
            if (isset($data['value']) && is_array($data['value'])) {
                Log::info('BC API: OData format detected', [
                    'records' => count($data['value'])
                ]);
                return $data;
            }

            Log::info('BC API: Direct array format', [
                'records' => is_array($data) ? count($data) : 1
            ]);

            return is_array($data) ? $data : [$data];

        } catch (Exception $e) {
            Log::error('BC API: Exception occurred', [
                'endpoint' => $endpoint,
                'error' => $e->getMessage(),
                'file' => $e->getFile(),
                'line' => $e->getLine()
            ]);
            throw $e;
        }
    }

    /**
     * Clear cached access token
     */
    public function clearTokenCache(): void
    {
        Cache::forget('bc_access_token');
        Log::info('BC: Access token cache cleared');
    }
}