<?php

namespace App\Http\Controllers\Regent;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;

class RoomController extends Controller
{
    /**
     * Show main menu (all room categories)
     */
    public function index()
    {
        $roomTypes = config('regent.room_types');
        return view('regent.index', compact('roomTypes'));
    }

    /**
     * ✅ FIX: Show specific room type
     * For submenu types (Suite/Corner Suite), redirect to first submenu
     * For direct types (Penthouse, etc), show room list
     */
    public function showRoomType($roomType)
    {
        $roomTypes = config('regent.room_types');
        
        // Check if room type exists
        if (!isset($roomTypes[$roomType])) {
            abort(404, 'Room type not found');
        }

        $roomTypeData = $roomTypes[$roomType];
        
        // ✅ If has submenu (Suite OR Corner Suite), redirect to first submenu
        if (isset($roomTypeData['has_submenu']) && $roomTypeData['has_submenu']) {
            $firstSubmenu = array_key_first($roomTypeData['submenu']);
            
            // Redirect based on room type
            if ($roomType === 'suite') {
                return redirect()->route('regent.rooms.suite.submenu', ['submenu' => $firstSubmenu]);
            } elseif ($roomType === 'corner_suite') {
                return redirect()->route('regent.rooms.corner.submenu', ['submenu' => $firstSubmenu]);
            }
        }

        // ✅ Direct room type (Penthouse, Beach Villa, Padi Villa)
        $rooms = $roomTypeData['rooms'];
        $roomTypeName = $roomTypeData['name'];

        return view('regent.room-type', compact('roomType', 'roomTypeName', 'rooms'));
    }

    /**
     * ✅ FIX: Show submenu room list
     * Accepts only 1 parameter: $submenu
     * Automatically detects if it's Suite or Corner Suite based on submenu prefix
     */
    public function showSubmenu($submenu)
    {
        $roomTypes = config('regent.room_types');
        
        // ✅ Auto-detect room type based on submenu prefix
        if (str_starts_with($submenu, 'suite_gw_')) {
            $roomType = 'suite';
        } elseif (str_starts_with($submenu, 'corner_gw_')) {
            $roomType = 'corner_suite';
        } else {
            abort(404, 'Invalid submenu');
        }
        
        // Validate room type
        if (!isset($roomTypes[$roomType])) {
            abort(404, 'Room type not found');
        }

        $roomTypeData = $roomTypes[$roomType];

        // Validate submenu
        if (!isset($roomTypeData['submenu'][$submenu])) {
            abort(404, 'Submenu not found');
        }

        $submenuData = $roomTypeData['submenu'][$submenu];
        $rooms = $submenuData['rooms'];
        $submenuName = $submenuData['name'];
        $roomTypeName = $roomTypeData['name'];

        return view('regent.submenu', compact('roomType', 'roomTypeName', 'submenu', 'submenuName', 'rooms'));
    }

    /**
     * ✅ Show room matrix for specific room
     */
    public function showRoomMatrix($roomType, $roomOrSubmenu, $room = null)
    {
        // Detect if this is Suite/Corner Suite (3 params) or direct Room Type (2 params)
        if ($room === null) {
            // 2 parameters: roomType & room
            $room = $roomOrSubmenu;
            $submenu = 'direct';
        } else {
            // 3 parameters: roomType, submenu & room
            $submenu = $roomOrSubmenu;
        }

        $months = [
            'january', 'february', 'march', 'april', 'may', 'june',
            'july', 'august', 'september', 'october', 'november', 'december'
        ];

        $monthlyData = [];

        foreach ($months as $month) {
            $tableName = "{$month}_regent";
            
            try {
                $data = DB::table($tableName)
                    ->where('Room', $room)
                    ->get();
                
                $monthlyData[$month] = $data;
            } catch (\Exception $e) {
                $monthlyData[$month] = collect([]);
            }
        }

        // Get room type name
        $roomTypes = config('regent.room_types');
        $roomTypeName = $roomTypes[$roomType]['name'] ?? $roomType;
        
        // Get submenu name if exists
        $submenuName = null;
        if ($submenu !== 'direct' && isset($roomTypes[$roomType]['submenu'][$submenu])) {
            $submenuName = $roomTypes[$roomType]['submenu'][$submenu]['name'];
        }

        return view('regent.room-matrix', compact('roomType', 'roomTypeName', 'submenu', 'submenuName', 'room', 'monthlyData'));
    }
}