<?php

namespace App\Http\Controllers\Regent;

use App\Http\Controllers\Controller;
use App\Services\BusinessCentral\RegentBusinessCentralService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use App\Models\Regent\OooRegent;

class ManageDataController extends Controller
{
    protected $bcService;

    public function __construct(RegentBusinessCentralService $bcService)
    {
        $this->bcService = $bcService;
    }

    public function index()
    {
        // Get data counts for all months
        $months = [
            'january', 'february', 'march', 'april', 'may', 'june',
            'july', 'august', 'september', 'october', 'november', 'december'
        ];

        $monthlyData = [];
        $totalRecords = 0;

        foreach ($months as $month) {
            $tableName = "{$month}_regent";
            $count = DB::table($tableName)->count();
            $monthlyData[$month] = [
                'count' => $count,
                'has_data' => $count > 0
            ];
            $totalRecords += $count;
        }

        // Get OOO count
        $oooCount = OooRegent::count();

        return view('regent.manage-data', compact('monthlyData', 'oooCount', 'totalRecords'));
    }

    public function refreshData(Request $request)
{
    // ✅ SET UNLIMITED EXECUTION TIME (SAMA SEPERTI HIRBND!)
    set_time_limit(0);
    ini_set('max_execution_time', 0);
    ini_set('memory_limit', '512M');

    try {
        Log::info('REGENT: ========== REFRESH DATA STARTED ==========');
        
        // Fetch and save data for all 12 months
        $months = [
            1 => 'january', 2 => 'february', 3 => 'march', 4 => 'april',
            5 => 'may', 6 => 'june', 7 => 'july', 8 => 'august',
            9 => 'september', 10 => 'october', 11 => 'november', 12 => 'december'
        ];

        $year = 2025;
        $results = [];
        $totalRecords = 0;

        foreach ($months as $monthNum => $monthName) {
            try {
                Log::info("REGENT: Processing {$monthName}...");
                
                $startTime = microtime(true);
                $data = $this->bcService->fetchRoomData($monthNum, $year);
                $duration = round(microtime(true) - $startTime, 2);
                
                Log::info("REGENT: ✅ Fetch completed", [
                    'month' => $monthName,
                    'year' => $year,
                    'records' => count($data),
                    'duration' => "{$duration}s"
                ]);

                if (count($data) > 0) {
                    $saved = $this->bcService->saveRoomData($monthName, $data);
                    $totalRecords += $saved;
                    
                    $results[$monthName] = [
                        'success' => true,
                        'records' => $saved
                    ];
                } else {
                    $results[$monthName] = [
                        'success' => true,
                        'records' => 0
                    ];
                }

            } catch (\Exception $e) {
                Log::error("REGENT: ❌ Failed to process {$monthName}", [
                    'error' => $e->getMessage()
                ]);
                
                $results[$monthName] = [
                    'success' => false,
                    'error' => $e->getMessage()
                ];
            }
        }

        // Fetch and save OOO data
        try {
            Log::info('REGENT: Fetching OOO data...');
            $oooData = $this->bcService->fetchOOOData();
            $oooSaved = $this->bcService->saveOOOData($oooData);
            
            $results['ooo'] = [
                'success' => true,
                'records' => $oooSaved
            ];

            Log::info('REGENT: ✅ OOO data processed', [
                'records' => $oooSaved
            ]);
        } catch (\Exception $e) {
            Log::error('REGENT: ❌ OOO data failed', [
                'error' => $e->getMessage()
            ]);
            
            $results['ooo'] = [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }

        Log::info('REGENT: ========== REFRESH DATA COMPLETED ==========', [
            'total_records' => $totalRecords
        ]);

        return response()->json([
            'success' => true,
            'message' => 'Data refreshed successfully',
            'results' => $results,
            'total_records' => $totalRecords
        ]);

    } catch (\Exception $e) {
        Log::error('REGENT: ❌ Refresh failed', [
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);

        return response()->json([
            'success' => false,
            'message' => 'Failed to refresh data: ' . $e->getMessage()
        ], 500);
    }
}
}