<?php

namespace App\Http\Controllers\Hirbnd;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class ReportController extends Controller
{
    /**
     * Show House Use Report
     */
    public function houseUse(Request $request)
    {
        $month = $request->get('month', 'all');
        
        try {
            if ($month === 'all') {
                $houseUseData = [];
                $months = [
                    'january', 'february', 'march', 'april', 'may', 'june',
                    'july', 'august', 'september', 'october', 'november', 'december'
                ];
                
                foreach ($months as $monthName) {
                    $tableName = $monthName . '_hirbnd';
                    
                    if (DB::getSchemaBuilder()->hasTable($tableName)) {
                        $data = DB::table($tableName)
                            ->where(function($query) {
                                $query->where('Market_Code', 'LIKE', '%House%Use%')
                                      ->orWhere('Market_Code', 'LIKE', '%HOUSE%USE%')
                                      ->orWhere('Market_Code', '=', 'House Use - H')
                                      ->orWhere('Market_Code', '=', 'HOUSE USE - H');
                            })
                            ->get();
                        
                        $houseUseData[$monthName] = $data;
                    } else {
                        $houseUseData[$monthName] = collect([]);
                    }
                }
            } else {
                $tableName = $month . '_hirbnd';
                
                if (DB::getSchemaBuilder()->hasTable($tableName)) {
                    $houseUseData = DB::table($tableName)
                        ->where(function($query) {
                            $query->where('Market_Code', 'LIKE', '%House%Use%')
                                  ->orWhere('Market_Code', 'LIKE', '%HOUSE%USE%')
                                  ->orWhere('Market_Code', '=', 'House Use - H')
                                  ->orWhere('Market_Code', '=', 'HOUSE USE - H');
                        })
                        ->get();
                } else {
                    $houseUseData = collect([]);
                }
            }
            
            return view('hirbnd.reports.house-use', compact('houseUseData', 'month'));
            
        } catch (\Exception $e) {
            Log::error('HIRBND House Use Report Error: ' . $e->getMessage());
            
            $houseUseData = $month === 'all' ? [] : collect([]);
            return view('hirbnd.reports.house-use', compact('houseUseData', 'month'));
        }
    }

    /**
 * Show House Use Room Detail Matrix (12 months x 31 days)
 */
public function houseUseRoomDetail(Request $request, $room)
{
    try {
        // Get all months
        $months = [
            'january', 'february', 'march', 'april', 'may', 'june',
            'july', 'august', 'september', 'october', 'november', 'december'
        ];
        
        // Build yearly matrix for this room
        $yearlyMatrix = [];
        $totalDays = 0;
        
        foreach ($months as $monthName) {
            $tableName = "{$monthName}_hirbnd";
            
            if (DB::getSchemaBuilder()->hasTable($tableName)) {
                $monthData = DB::table($tableName)
                    ->where('Room', $room)
                    ->where(function($query) {
                        $query->where('Market_Code', 'LIKE', '%House%Use%')
                              ->orWhere('Market_Code', 'LIKE', '%HOUSE%USE%')
                              ->orWhere('Market_Code', '=', 'House Use - H')
                              ->orWhere('Market_Code', '=', 'HOUSE USE - H');
                    })
                    ->get();
                
                // Build day matrix for this month
                $monthNumber = array_search($monthName, $months) + 1;
                $daysInMonth = cal_days_in_month(CAL_GREGORIAN, $monthNumber, 2025);
                
                $dayMatrix = [];
                for ($day = 1; $day <= 31; $day++) {
                    if ($day > $daysInMonth) {
                        $dayMatrix[$day] = null; // Invalid day
                    } else {
                        $hasData = false;
                        foreach ($monthData as $record) {
                            $postingDay = (int) date('d', strtotime($record->Posting_Date));
                            if ($postingDay === $day) {
                                $hasData = true;
                                $totalDays++;
                                break;
                            }
                        }
                        $dayMatrix[$day] = $hasData;
                    }
                }
                
                $yearlyMatrix[$monthName] = $dayMatrix;
            } else {
                // Table doesn't exist
                $yearlyMatrix[$monthName] = array_fill(1, 31, null);
            }
        }
        
        return view('hirbnd.reports.house-use-room-detail', compact('room', 'yearlyMatrix', 'totalDays'));
        
    } catch (\Exception $e) {
        Log::error('HIRBND House Use Room Detail Error: ' . $e->getMessage());
        
        return redirect()->route('hirbnd.reports.house-use')
            ->with('error', 'Error loading room detail');
    }
}

/**
 * Show Complimentary Room Detail Matrix (12 months x 31 days)
 */
public function complimentaryRoomDetail(Request $request, $room)
{
    try {
        // Get all months
        $months = [
            'january', 'february', 'march', 'april', 'may', 'june',
            'july', 'august', 'september', 'october', 'november', 'december'
        ];
        
        // Build yearly matrix for this room
        $yearlyMatrix = [];
        $totalDays = 0;
        
        foreach ($months as $monthName) {
            $tableName = "{$monthName}_hirbnd";
            
            if (DB::getSchemaBuilder()->hasTable($tableName)) {
                $monthData = DB::table($tableName)
                    ->where('Room', $room)
                    ->where(function($query) {
                        $query->where('Market_Code', 'LIKE', '%Complimentary%')
                              ->orWhere('Market_Code', 'LIKE', '%COMPLIMENTARY%')
                              ->orWhere('Market_Code', '=', 'Complimentary - N')
                              ->orWhere('Market_Code', '=', 'COMPLIMENTARY - N');
                    })
                    ->get();
                
                // Build day matrix for this month
                $monthNumber = array_search($monthName, $months) + 1;
                $daysInMonth = cal_days_in_month(CAL_GREGORIAN, $monthNumber, 2025);
                
                $dayMatrix = [];
                for ($day = 1; $day <= 31; $day++) {
                    if ($day > $daysInMonth) {
                        $dayMatrix[$day] = null; // Invalid day
                    } else {
                        $hasData = false;
                        foreach ($monthData as $record) {
                            $postingDay = (int) date('d', strtotime($record->Posting_Date));
                            if ($postingDay === $day) {
                                $hasData = true;
                                $totalDays++;
                                break;
                            }
                        }
                        $dayMatrix[$day] = $hasData;
                    }
                }
                
                $yearlyMatrix[$monthName] = $dayMatrix;
            } else {
                // Table doesn't exist
                $yearlyMatrix[$monthName] = array_fill(1, 31, null);
            }
        }
        
        return view('hirbnd.reports.complimentary-room-detail', compact('room', 'yearlyMatrix', 'totalDays'));
        
    } catch (\Exception $e) {
        Log::error('HIRBND Complimentary Room Detail Error: ' . $e->getMessage());
        
        return redirect()->route('hirbnd.reports.complimentary')
            ->with('error', 'Error loading room detail');
    }
}

    /**
     * Show Complimentary Report
     */
    public function complimentary(Request $request)
    {
        $month = $request->get('month', 'all');
        
        try {
            if ($month === 'all') {
                $complimentaryData = [];
                $months = [
                    'january', 'february', 'march', 'april', 'may', 'june',
                    'july', 'august', 'september', 'october', 'november', 'december'
                ];
                
                foreach ($months as $monthName) {
                    $tableName = $monthName . '_hirbnd';
                    
                    if (DB::getSchemaBuilder()->hasTable($tableName)) {
                        $data = DB::table($tableName)
                            ->where(function($query) {
                                $query->where('Market_Code', 'LIKE', '%Complimentary%')
                                      ->orWhere('Market_Code', 'LIKE', '%COMPLIMENTARY%')
                                      ->orWhere('Market_Code', '=', 'Complimentary - N')
                                      ->orWhere('Market_Code', '=', 'COMPLIMENTARY - N');
                            })
                            ->get();
                        
                        $complimentaryData[$monthName] = $data;
                    } else {
                        $complimentaryData[$monthName] = collect([]);
                    }
                }
            } else {
                $tableName = $month . '_hirbnd';
                
                if (DB::getSchemaBuilder()->hasTable($tableName)) {
                    $complimentaryData = DB::table($tableName)
                        ->where(function($query) {
                            $query->where('Market_Code', 'LIKE', '%Complimentary%')
                                  ->orWhere('Market_Code', 'LIKE', '%COMPLIMENTARY%')
                                  ->orWhere('Market_Code', '=', 'Complimentary - N')
                                  ->orWhere('Market_Code', '=', 'COMPLIMENTARY - N');
                        })
                        ->get();
                } else {
                    $complimentaryData = collect([]);
                }
            }
            
            return view('hirbnd.reports.complimentary', compact('complimentaryData', 'month'));
            
        } catch (\Exception $e) {
            Log::error('HIRBND Complimentary Report Error: ' . $e->getMessage());
            
            $complimentaryData = $month === 'all' ? [] : collect([]);
            return view('hirbnd.reports.complimentary', compact('complimentaryData', 'month'));
        }
    }

    /**
 * Show OOO Report - MATRIX VIEW
 */
public function ooo(Request $request)
{
    $month = $request->get('month', 'all');
    
    try {
        // Get all OOO data where not completed
        $oooData = DB::table('ooo_hirbnd')
            ->where('Completed_On', '0001-01-01')
            ->orderBy('Room_No')
            ->get();
        
        // Calculate statistics
        $stats = [
            'total' => $oooData->count(),
            'standard' => 0,
            'suite' => 0,
        ];
        
        // Suite patterns
        $suitePatterns = ['SUITE', 'Suite', 'CLUB', 'Club', 'OCEAN', 'Ocean', 'DELUXE', 'Deluxe'];
        
        foreach($oooData as $ooo) {
            $isSuite = false;
            foreach($suitePatterns as $pattern) {
                if (stripos($ooo->Room_No, $pattern) !== false) {
                    $isSuite = true;
                    break;
                }
            }
            
            if ($isSuite) {
                $stats['suite']++;
            } else {
                $stats['standard']++;
            }
        }
        
        return view('hirbnd.reports.ooo', compact('oooData', 'stats', 'month'));
        
    } catch (\Exception $e) {
        Log::error('HIRBND OOO Report Error: ' . $e->getMessage());
        
        $oooData = collect([]);
        $stats = ['total' => 0, 'standard' => 0, 'suite' => 0];
        
        return view('hirbnd.reports.ooo', compact('oooData', 'stats', 'month'));
    }
}

    /**
 * Build Monthly Matrix (Rooms x 12 Months)
 */
private function buildMonthlyMatrix($oooData, $rooms)
{
    $matrix = [];
    $months = [
        'january', 'february', 'march', 'april', 'may', 'june',
        'july', 'august', 'september', 'october', 'november', 'december'
    ];
    
    foreach ($rooms as $room) {
        $matrix[$room] = [];
        
        foreach ($months as $monthIndex => $monthName) {
            $hasOOO = false;
            
            // Define current month range
            $monthStart = Carbon::create(2025, $monthIndex + 1, 1)->startOfMonth();
            $monthEnd = Carbon::create(2025, $monthIndex + 1, 1)->endOfMonth();
            
            // Check if room has OOO in this month
            foreach ($oooData as $ooo) {
                if ($ooo->Room_No !== $room) continue;
                
                // Parse dates
                try {
                    $beginDate = Carbon::parse($ooo->Begin_Date);
                    
                    // Handle End_Date
                    if ($ooo->End_Date && $ooo->End_Date !== '0001-01-01') {
                        $endDate = Carbon::parse($ooo->End_Date);
                    } else {
                        // Jika End_Date kosong atau 0001-01-01, anggap masih ongoing
                        $endDate = Carbon::now()->endOfYear();
                    }
                    
                    // ✅ FIX: Check if OOO period OVERLAPS with this month
                    // OOO overlaps if: beginDate <= monthEnd AND endDate >= monthStart
                    if ($beginDate->lte($monthEnd) && $endDate->gte($monthStart)) {
                        $hasOOO = true;
                        break;
                    }
                    
                } catch (\Exception $e) {
                    // Skip invalid dates
                    continue;
                }
            }
            
            $matrix[$room][$monthName] = $hasOOO;
        }
    }
    
    return $matrix;
}

    /**
 * Build Daily Matrix (Rooms x Days 1-31)
 */
private function buildDailyMatrix($oooData, $rooms, $monthName)
{
    $matrix = [];
    
    // Get month number
    $monthsList = [
        'january' => 1, 'february' => 2, 'march' => 3, 'april' => 4,
        'may' => 5, 'june' => 6, 'july' => 7, 'august' => 8,
        'september' => 9, 'october' => 10, 'november' => 11, 'december' => 12
    ];
    
    $monthNumber = $monthsList[$monthName] ?? 1;
    $daysInMonth = Carbon::create(2025, $monthNumber, 1)->daysInMonth;
    
    foreach ($rooms as $room) {
        $matrix[$room] = [];
        
        for ($day = 1; $day <= $daysInMonth; $day++) {
            $hasOOO = false;
            $currentDate = Carbon::create(2025, $monthNumber, $day);
            
            // Check if room has OOO on this day
            foreach ($oooData as $ooo) {
                if ($ooo->Room_No !== $room) continue;
                
                try {
                    $beginDate = Carbon::parse($ooo->Begin_Date);
                    
                    // Handle End_Date
                    if ($ooo->End_Date && $ooo->End_Date !== '0001-01-01') {
                        $endDate = Carbon::parse($ooo->End_Date);
                    } else {
                        $endDate = Carbon::now()->endOfYear();
                    }
                    
                    // ✅ FIX: Check if currentDate is between beginDate and endDate (INCLUSIVE)
                    if ($currentDate->gte($beginDate) && $currentDate->lte($endDate)) {
                        $hasOOO = true;
                        break;
                    }
                    
                } catch (\Exception $e) {
                    continue;
                }
            }
            
            $matrix[$room][$day] = $hasOOO;
        }
    }
    
    return $matrix;
}

    /**
     * Calculate Monthly Totals for ALL MONTHS view
     */
    private function calculateMonthTotals($matrixData)
    {
        $totals = [];
        $months = [
            'january', 'february', 'march', 'april', 'may', 'june',
            'july', 'august', 'september', 'october', 'november', 'december'
        ];
        
        foreach ($months as $month) {
            $count = 0;
            foreach ($matrixData as $room => $monthsData) {
                if (isset($monthsData[$month]) && $monthsData[$month]) {
                    $count++;
                }
            }
            $totals[$month] = $count;
        }
        
        return $totals;
    }

    /**
 * Show OOO Room Detail Matrix
 */
public function oooRoomDetail(Request $request, $room)
{
    try {
        // Get OOO data for this room
        $oooData = DB::table('ooo_hirbnd')
            ->where('Room_No', $room)
            ->where('Completed_On', '0001-01-01')
            ->get();
        
        if ($oooData->isEmpty()) {
            return redirect()->route('hirbnd.reports.ooo')
                ->with('error', "No OOO data found for Room {$room}");
        }
        
        return view('hirbnd.reports.ooo-room-detail', compact('room', 'oooData'));
        
    } catch (\Exception $e) {
        Log::error('HIRBND OOO Room Detail Error: ' . $e->getMessage());
        
        return redirect()->route('hirbnd.reports.ooo')
            ->with('error', 'Error loading room detail');
    }
}

    /**
 * Build Yearly Matrix for Room Detail (12 months x 31 days)
 */
private function buildYearlyMatrix($oooData, $room)
{
    $matrix = [];
    $months = [
        'january' => 1, 'february' => 2, 'march' => 3, 'april' => 4,
        'may' => 5, 'june' => 6, 'july' => 7, 'august' => 8,
        'september' => 9, 'october' => 10, 'november' => 11, 'december' => 12
    ];
    
    foreach ($months as $monthName => $monthNumber) {
        $daysInMonth = Carbon::create(2025, $monthNumber, 1)->daysInMonth;
        $matrix[$monthName] = [];
        
        for ($day = 1; $day <= 31; $day++) {
            if ($day > $daysInMonth) {
                $matrix[$monthName][$day] = null; // Invalid date
                continue;
            }
            
            $hasOOO = false;
            $currentDate = Carbon::create(2025, $monthNumber, $day);
            
            foreach ($oooData as $ooo) {
                try {
                    $beginDate = Carbon::parse($ooo->Begin_Date);
                    
                    if ($ooo->End_Date && $ooo->End_Date !== '0001-01-01') {
                        $endDate = Carbon::parse($ooo->End_Date);
                    } else {
                        $endDate = Carbon::now()->endOfYear();
                    }
                    
                    // Check if currentDate is within OOO period
                    if ($currentDate->gte($beginDate) && $currentDate->lte($endDate)) {
                        $hasOOO = true;
                        break;
                    }
                    
                } catch (\Exception $e) {
                    continue;
                }
            }
            
            $matrix[$monthName][$day] = $hasOOO;
        }
    }
    
    return $matrix;
}

    /**
     * Show Revenue Report
     */
    public function revenue(Request $request)
    {
        $month = $request->get('month', 'all');
        
        try {
            $revenueTrxCodes = config('businesscentral.revenue_trx_codes', []);
            
            if (empty($revenueTrxCodes)) {
                $revenueTrxCodes = [
                    '1000', '1001', '1002', '1010', '1011', '1020', '1021', '1030', '1031',
                    '1040', '1041', '1050', '1051', '1060', '1061', '1070', '1071', '1072',
                    '1075', '1076', '1080', '1081', '1082', '1083', '1084', '1085', '1086',
                    '1087', '1088', '1089', '1090', '1091', '1092', '1093', '1094', '1095',
                ];
            }
            
            if ($month === 'all') {
                $revenueData = [];
                $months = [
                    'january', 'february', 'march', 'april', 'may', 'june',
                    'july', 'august', 'september', 'october', 'november', 'december'
                ];
                
                foreach ($months as $monthName) {
                    $tableName = $monthName . '_hirbnd';
                    
                    if (DB::getSchemaBuilder()->hasTable($tableName)) {
                        $data = DB::table($tableName)
                            ->whereIn('TRX_Code', $revenueTrxCodes)
                            ->whereNotIn('Market_Code', ['House Use - H', 'Complimentary - N'])
                            ->where('Amount_1', '>', 0)
                            ->get();
                        
                        $revenueData[$monthName] = $data;
                    } else {
                        $revenueData[$monthName] = collect([]);
                    }
                }
                
            } else {
                $tableName = $month . '_hirbnd';
                
                if (DB::getSchemaBuilder()->hasTable($tableName)) {
                    $revenueData = DB::table($tableName)
                        ->whereIn('TRX_Code', $revenueTrxCodes)
                        ->whereNotIn('Market_Code', ['House Use - H', 'Complimentary - N'])
                        ->where('Amount_1', '>', 0)
                        ->get();
                } else {
                    $revenueData = collect([]);
                }
            }
            
            return view('hirbnd.reports.revenue', compact('revenueData', 'month'));
            
        } catch (\Exception $e) {
            Log::error('HIRBND Revenue Report Error: ' . $e->getMessage());
            
            $revenueData = $month === 'all' ? [] : collect([]);
            return view('hirbnd.reports.revenue', compact('revenueData', 'month'));
        }
    }
}