<?php

namespace App\Http\Controllers\Hirbnd;

use App\Http\Controllers\Controller;
use App\Services\BusinessCentral\HirbndApiService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Exception;

class ManageDataController extends Controller
{
    protected $hirbndService;

    public function __construct(HirbndApiService $hirbndService)
    {
        $this->hirbndService = $hirbndService;
    }

    /**
     * Show manage data page
     */
    public function index()
    {
        // Get last update time dari salah satu table
        $lastDataLoad = null;
        
        try {
            // Check january table untuk last update
            $lastRecord = DB::table('january_hirbnd')
                ->orderBy('updated_at', 'desc')
                ->first();
            
            if ($lastRecord) {
                $lastDataLoad = $lastRecord->updated_at;
            }
        } catch (Exception $e) {
            // Kalau table belum ada data, set null
            $lastDataLoad = null;
        }

        // Count statistics
        $statistics = $this->getDataStatistics();

        return view('hirbnd.manage-data', compact('lastDataLoad', 'statistics'));
    }

    /**
     * Get data statistics
     */
    private function getDataStatistics(): array
    {
        $stats = [
            'total_rooms' => 0,
            'total_ooo' => 0,
            'room_types' => 0,
            'monthly_tables' => 0,
        ];

        try {
            // Count OOO records
            $stats['total_ooo'] = DB::table('ooo_hirbnd')->count();

            // Count monthly data
            $months = ['january', 'february', 'march', 'april', 'may', 'june',
                      'july', 'august', 'september', 'october', 'november', 'december'];
            
            $tablesWithData = 0;
            $totalRecords = 0;

            foreach ($months as $month) {
                $tableName = "{$month}_hirbnd";
                $count = DB::table($tableName)->count();
                
                if ($count > 0) {
                    $tablesWithData++;
                    $totalRecords += $count;
                }
            }

            $stats['total_rooms'] = $totalRecords;
            $stats['monthly_tables'] = $tablesWithData;

            // Count distinct room types
            if ($tablesWithData > 0) {
                $stats['room_types'] = DB::table('january_hirbnd')
                    ->distinct('RoomType')
                    ->count('RoomType');
            }

        } catch (Exception $e) {
            Log::warning('Failed to get statistics', ['error' => $e->getMessage()]);
        }

        return $stats;
    }

    /**
     * Refresh data from Business Central
     */
    public function refreshData(Request $request)
    {
        // Set unlimited execution time
        set_time_limit(0);
        ini_set('max_execution_time', 0);
        ini_set('memory_limit', '512M');

        try {
            Log::info('==============================================');
            Log::info('MANAGE DATA: Starting data refresh...');
            Log::info('==============================================');

            // Load all data (room + ooo)
            $year = $request->input('year', date('Y'));
            $statistics = $this->hirbndService->loadAllData($year);

            Log::info('==============================================');
            Log::info('MANAGE DATA: Data refresh completed!');
            Log::info('==============================================');
            Log::info('Statistics:', $statistics);

            return response()->json([
                'success' => true,
                'message' => 'Data berhasil di-refresh dari Business Central!',
                'statistics' => $statistics,
            ]);

        } catch (Exception $e) {
            Log::error('==============================================');
            Log::error('MANAGE DATA: Data refresh FAILED!');
            Log::error('==============================================');
            Log::error('Error: ' . $e->getMessage());
            Log::error('Trace: ' . $e->getTraceAsString());

            return response()->json([
                'success' => false,
                'message' => 'Gagal refresh data: ' . $e->getMessage(),
                'error' => $e->getMessage(),
            ], 500);
        }
    }
}