<?php

namespace App\Http\Controllers\Hirbc;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ReportController extends Controller
{
    /**
     * Show House Use Report
     */
    public function houseUse(Request $request)
    {
        $month = $request->get('month', 'all');
        
        try {
            if ($month === 'all') {
                $houseUseData = [];
                $months = [
                    'january', 'february', 'march', 'april', 'may', 'june',
                    'july', 'august', 'september', 'october', 'november', 'december'
                ];
                
                foreach ($months as $monthName) {
                    $tableName = $monthName . '_hirbc';
                    
                    if (DB::getSchemaBuilder()->hasTable($tableName)) {
                        $data = DB::table($tableName)
                            ->where(function($query) {
                                $query->where('Market_Code', 'LIKE', '%House%Use%')
                                      ->orWhere('Market_Code', 'LIKE', '%HOUSE%USE%')
                                      ->orWhere('Market_Code', '=', 'House Use - H')
                                      ->orWhere('Market_Code', '=', 'HOUSE USE - H');
                            })
                            ->get();
                        
                        $houseUseData[$monthName] = $data;
                        
                        Log::info("HIRBC House Use: {$monthName}", [
                            'records' => $data->count()
                        ]);
                    } else {
                        $houseUseData[$monthName] = collect([]);
                    }
                }
            } else {
                $tableName = $month . '_hirbc';
                
                if (DB::getSchemaBuilder()->hasTable($tableName)) {
                    $houseUseData = DB::table($tableName)
                        ->where(function($query) {
                            $query->where('Market_Code', 'LIKE', '%House%Use%')
                                  ->orWhere('Market_Code', 'LIKE', '%HOUSE%USE%')
                                  ->orWhere('Market_Code', '=', 'House Use - H')
                                  ->orWhere('Market_Code', '=', 'HOUSE USE - H');
                        })
                        ->get();
                    
                    Log::info("HIRBC House Use: {$month}", [
                        'records' => $houseUseData->count()
                    ]);
                } else {
                    $houseUseData = collect([]);
                }
            }
            
            return view('hirbc.reports.house-use', compact('houseUseData', 'month'));
            
        } catch (\Exception $e) {
            Log::error('HIRBC House Use Report Error: ' . $e->getMessage());
            
            $houseUseData = $month === 'all' ? [] : collect([]);
            return view('hirbc.reports.house-use', compact('houseUseData', 'month'));
        }
    }

    /**
     * Show Complimentary Report
     */
    public function complimentary(Request $request)
    {
        $month = $request->get('month', 'all');
        
        try {
            if ($month === 'all') {
                $complimentaryData = [];
                $months = [
                    'january', 'february', 'march', 'april', 'may', 'june',
                    'july', 'august', 'september', 'october', 'november', 'december'
                ];
                
                foreach ($months as $monthName) {
                    $tableName = $monthName . '_hirbc';
                    
                    if (DB::getSchemaBuilder()->hasTable($tableName)) {
                        $data = DB::table($tableName)
                            ->where(function($query) {
                                $query->where('Market_Code', 'LIKE', '%Complimentary%')
                                      ->orWhere('Market_Code', 'LIKE', '%COMPLIMENTARY%')
                                      ->orWhere('Market_Code', '=', 'Complimentary - N')
                                      ->orWhere('Market_Code', '=', 'COMPLIMENTARY - N');
                            })
                            ->get();
                        
                        $complimentaryData[$monthName] = $data;
                        
                        Log::info("HIRBC Complimentary: {$monthName}", [
                            'records' => $data->count()
                        ]);
                    } else {
                        $complimentaryData[$monthName] = collect([]);
                    }
                }
            } else {
                $tableName = $month . '_hirbc';
                
                if (DB::getSchemaBuilder()->hasTable($tableName)) {
                    $complimentaryData = DB::table($tableName)
                        ->where(function($query) {
                            $query->where('Market_Code', 'LIKE', '%Complimentary%')
                                  ->orWhere('Market_Code', 'LIKE', '%COMPLIMENTARY%')
                                  ->orWhere('Market_Code', '=', 'Complimentary - N')
                                  ->orWhere('Market_Code', '=', 'COMPLIMENTARY - N');
                        })
                        ->get();
                    
                    Log::info("HIRBC Complimentary: {$month}", [
                        'records' => $complimentaryData->count()
                    ]);
                } else {
                    $complimentaryData = collect([]);
                }
            }
            
            return view('hirbc.reports.complimentary', compact('complimentaryData', 'month'));
            
        } catch (\Exception $e) {
            Log::error('HIRBC Complimentary Report Error: ' . $e->getMessage());
            
            $complimentaryData = $month === 'all' ? [] : collect([]);
            return view('hirbc.reports.complimentary', compact('complimentaryData', 'month'));
        }
    }

    /**
 * Show OOO Report - MATRIX VIEW
 */
public function ooo(Request $request)
{
    $month = $request->get('month', 'all');
    
    try {
        // Get all OOO data where not completed
        $oooData = DB::table('ooo_hirbc')
            ->where('Completed_On', '0001-01-01')
            ->orderBy('Room_No')
            ->get();
        
        // Calculate statistics
        $stats = [
            'total' => $oooData->count(),
            'standard' => 0,
            'suite' => 0,
        ];
        
        // Suite patterns
        $suitePatterns = ['SUITE', 'Suite', 'CLUB', 'Club', 'OCEAN', 'Ocean', 'DELUXE', 'Deluxe'];
        
        foreach($oooData as $ooo) {
            $isSuite = false;
            foreach($suitePatterns as $pattern) {
                if (stripos($ooo->Room_No, $pattern) !== false) {
                    $isSuite = true;
                    break;
                }
            }
            
            if ($isSuite) {
                $stats['suite']++;
            } else {
                $stats['standard']++;
            }
        }
        
        return view('hirbc.reports.ooo', compact('oooData', 'stats', 'month'));
        
    } catch (\Exception $e) {
        Log::error('HIRBC OOO Report Error: ' . $e->getMessage());
        
        $oooData = collect([]);
        $stats = ['total' => 0, 'standard' => 0, 'suite' => 0];
        
        return view('hirbc.reports.ooo', compact('oooData', 'stats', 'month'));
    }
}
/**
 * Show OOO Room Detail Matrix
 */
public function oooRoomDetail(Request $request, $room)
{
    try {
        // Get OOO data for this room
        $oooData = DB::table('ooo_hirbc')
            ->where('Room_No', $room)
            ->where('Completed_On', '0001-01-01')
            ->get();
        
        if ($oooData->isEmpty()) {
            return redirect()->route('hirbc.reports.ooo')
                ->with('error', "No OOO data found for Room {$room}");
        }
        
        return view('hirbc.reports.ooo-room-detail', compact('room', 'oooData'));
        
    } catch (\Exception $e) {
        Log::error('HIRBC OOO Room Detail Error: ' . $e->getMessage());
        
        return redirect()->route('hirbc.reports.ooo')
            ->with('error', 'Error loading room detail');
    }
}
    /**
     * Show Revenue Report
     */
    public function revenue(Request $request)
    {
        $month = $request->get('month', 'all');
        
        try {
            $revenueTrxCodes = config('businesscentral.revenue_trx_codes', []);
            
            if (empty($revenueTrxCodes)) {
                Log::warning('HIRBC: revenue_trx_codes config is empty!');
                
                $revenueTrxCodes = [
                    '1000', '1001', '1002', '1010', '1011', '1020', '1021', '1030', '1031',
                    '1040', '1041', '1050', '1051', '1060', '1061', '1070', '1071', '1072',
                    '1075', '1076', '1080', '1081', '1082', '1083', '1084', '1085', '1086',
                    '1087', '1088', '1089', '1090', '1091', '1092', '1093', '1094', '1095',
                ];
            }
            
            if ($month === 'all') {
                $revenueData = [];
                $months = [
                    'january', 'february', 'march', 'april', 'may', 'june',
                    'july', 'august', 'september', 'october', 'november', 'december'
                ];
                
                foreach ($months as $monthName) {
                    $tableName = $monthName . '_hirbc';
                    
                    if (DB::getSchemaBuilder()->hasTable($tableName)) {
                        $data = DB::table($tableName)
                            ->whereIn('TRX_Code', $revenueTrxCodes)
                            ->whereNotIn('Market_Code', ['House Use - H', 'Complimentary - N'])
                            ->where('Amount_1', '>', 0)
                            ->get();
                        
                        $revenueData[$monthName] = $data;
                        
                        Log::info("HIRBC Revenue: {$monthName}", [
                            'records' => $data->count()
                        ]);
                    } else {
                        $revenueData[$monthName] = collect([]);
                        Log::warning("HIRBC Revenue: Table {$tableName} not found");
                    }
                }
                
            } else {
                $tableName = $month . '_hirbc';
                
                if (DB::getSchemaBuilder()->hasTable($tableName)) {
                    $revenueData = DB::table($tableName)
                        ->whereIn('TRX_Code', $revenueTrxCodes)
                        ->whereNotIn('Market_Code', ['House Use - H', 'Complimentary - N'])
                        ->where('Amount_1', '>', 0)
                        ->get();
                    
                    Log::info("HIRBC Revenue: {$month}", [
                        'records' => $revenueData->count()
                    ]);
                } else {
                    $revenueData = collect([]);
                    Log::warning("HIRBC Revenue: Table {$tableName} not found");
                }
            }
            
            return view('hirbc.reports.revenue', compact('revenueData', 'month'));
            
        } catch (\Exception $e) {
            Log::error('HIRBC Revenue Report Error: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            
            $revenueData = $month === 'all' ? [] : collect([]);
            return view('hirbc.reports.revenue', compact('revenueData', 'month'));
        }
    }
}