<?php

namespace App\Http\Controllers\Hirbc;

use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class DashboardController extends Controller
{
    /**
     * Show HIRBC Dashboard
     */
    public function index()
    {
        try {
            // Get room types from config
            $roomTypes = config('hirbc.room_types');
            
            // Calculate statistics
            $stats = $this->calculateStats();
            
            return view('hirbc.dashboard', compact('roomTypes', 'stats'));
            
        } catch (\Exception $e) {
            Log::error('HIRBC Dashboard Error: ' . $e->getMessage());
            
            return view('hirbc.dashboard', [
                'roomTypes' => config('hirbc.room_types'),
                'stats' => $this->getEmptyStats()
            ]);
        }
    }

    /**
     * Calculate dashboard statistics
     */
    private function calculateStats()
    {
        $stats = [
            'total_rooms' => config('hirbc.total_rooms', 0),
            'standard_rooms' => config('hirbc.standard_rooms', 0),
            'suite_rooms' => config('hirbc.suite_rooms', 0),
            'months_loaded' => 0,
            'total_records' => 0,
            'ooo_records' => 0,
            'last_update' => null
        ];

        try {
            // Count months with data
            $months = [
                'january', 'february', 'march', 'april', 'may', 'june',
                'july', 'august', 'september', 'october', 'november', 'december'
            ];
            
            $monthsLoaded = 0;
            $totalRecords = 0;
            
            foreach ($months as $month) {
                $tableName = $month . '_hirbc';
                
                if (DB::getSchemaBuilder()->hasTable($tableName)) {
                    $count = DB::table($tableName)->count();
                    
                    if ($count > 0) {
                        $monthsLoaded++;
                        $totalRecords += $count;
                    }
                }
            }
            
            $stats['months_loaded'] = $monthsLoaded;
            $stats['total_records'] = $totalRecords;
            
            // Count OOO records
            if (DB::getSchemaBuilder()->hasTable('ooo_hirbc')) {
                $stats['ooo_records'] = DB::table('ooo_hirbc')
                    ->where('Completed_On', '0001-01-01')
                    ->count();
            }
            
            // Get last update time
            if ($monthsLoaded > 0) {
                $lastMonth = null;
                
                foreach (array_reverse($months) as $month) {
                    $tableName = $month . '_hirbc';
                    
                    if (DB::getSchemaBuilder()->hasTable($tableName)) {
                        $count = DB::table($tableName)->count();
                        
                        if ($count > 0) {
                            $lastMonth = $tableName;
                            break;
                        }
                    }
                }
                
                if ($lastMonth) {
                    $lastRecord = DB::table($lastMonth)
                        ->orderBy('updated_at', 'desc')
                        ->first();
                    
                    if ($lastRecord && isset($lastRecord->updated_at)) {
                        $stats['last_update'] = $lastRecord->updated_at;
                    }
                }
            }
            
        } catch (\Exception $e) {
            Log::error('HIRBC Stats Calculation Error: ' . $e->getMessage());
        }
        
        return $stats;
    }

    /**
     * Get empty stats structure
     */
    private function getEmptyStats()
    {
        return [
            'total_rooms' => config('hirbc.total_rooms', 0),
            'standard_rooms' => config('hirbc.standard_rooms', 0),
            'suite_rooms' => config('hirbc.suite_rooms', 0),
            'months_loaded' => 0,
            'total_records' => 0,
            'ooo_records' => 0,
            'last_update' => null
        ];
    }
}