{{-- resources/views/positions/create.blade.php --}}
@extends('layouts.app')

@section('title', 'Tambah Posisi Baru')
@section('page-title', 'Tambah Posisi Baru')

@push('styles')
<style>
  /* ====== Halaman Create Position (khusus konten) ====== */
  .form-container{background:#fff;border-radius:12px;box-shadow:0 4px 20px rgba(0,0,0,.08);overflow:hidden;border:1px solid #e5e7eb}
  .form-header{background:linear-gradient(135deg,#4f46e5,#7c3aed);color:#fff;padding:20px 24px}
  .form-title{font-size:1.125rem;font-weight:700;margin-bottom:4px}
  .form-subtitle{font-size:.9rem;opacity:.95}

  .form-body{padding:24px}
  .form-grid{display:grid;grid-template-columns:1fr 1fr;gap:16px;margin-bottom:8px}
  .form-group{margin-bottom:14px}
  .form-group.full-width{grid-column:1 / -1}
  .form-label{display:block;font-size:.9rem;font-weight:600;color:#374151;margin-bottom:6px}
  .form-label.required::after{content:' *';color:#dc2626}

  .form-input,.form-select,.form-textarea{
    width:100%;padding:10px 12px;border:1px solid #d1d5db;border-radius:10px;font-size:.95rem;background:#fff;
    transition:box-shadow .2s,border-color .2s
  }
  .form-input:focus,.form-select:focus,.form-textarea:focus{
    outline:0;border-color:#4f46e5;box-shadow:0 0 0 3px rgba(79,70,229,.12)
  }
  .form-textarea{resize:vertical;min-height:100px}
  .form-textarea.large{min-height:120px}
  .form-row{display:grid;grid-template-columns:1fr 1fr;gap:12px}
  .form-help{font-size:.8rem;color:#6b7280;margin-top:5px}

  .checkbox-group{display:flex;align-items:center;gap:10px;margin-top:6px}
  .checkbox-input{width:18px;height:18px;border:2px solid #d1d5db;border-radius:4px;cursor:pointer}
  .checkbox-input:checked{background:#4f46e5;border-color:#4f46e5}
  .checkbox-label{font-size:.9rem;color:#374151;cursor:pointer}

  .form-actions{background:#f9fafb;padding:16px 20px;display:flex;justify-content:space-between;align-items:center;border-top:1px solid #e5e7eb}
  .form-actions-left{color:#6b7280;font-size:.85rem}
  .form-actions-right{display:flex;gap:10px}

  .btn{display:inline-flex;align-items:center;gap:8px;padding:10px 16px;border-radius:10px;font-weight:600;font-size:.9rem;text-decoration:none;border:1px solid transparent;transition:.2s}
  .btn-secondary{background:#6b7280;color:#fff}
  .btn-secondary:hover{background:#4b5563}
  .btn-success{background:linear-gradient(135deg,#10b981,#059669);color:#fff}
  .btn-success:hover{filter:brightness(.97)}
  .btn.loading{opacity:.7;pointer-events:none}
  .btn.loading::after{content:'';width:16px;height:16px;border:2px solid transparent;border-top:2px solid currentColor;border-radius:50%;animation:spin 1s linear infinite;margin-left:6px}
  @keyframes spin{to{transform:rotate(360deg)}}

  .alert{padding:12px 14px;border-radius:10px;margin-bottom:14px;display:flex;align-items:center;gap:8px;border:1px solid}
  .alert-success{background:#d1fae5;color:#065f46;border-color:#a7f3d0}
  .alert-error{background:#fee2e2;color:#991b1b;border-color:#fecaca}

  .form-input.error,.form-select.error,.form-textarea.error{border-color:#dc2626;box-shadow:0 0 0 3px rgba(220,38,38,.12)}
  .error-message{color:#dc2626;font-size:.8rem;margin-top:6px;display:flex;align-items:center;gap:6px}

  @media (max-width: 1024px){
    .form-grid{grid-template-columns:1fr}
    .form-row{grid-template-columns:1fr}
    .form-actions{flex-direction:column;gap:12px;text-align:center}
  }
</style>
@endpush

@section('content')
  {{-- Header kecil (breadcrumb + tombol kembali) --}}
  <div class="mb-6 flex flex-col gap-3 sm:flex-row sm:items-center sm:justify-between">
    <div>
      <div class="text-sm text-gray-500">
        <a href="{{ route('dashboard') }}" class="text-primary hover:underline">Dashboard</a>
        <span class="mx-1">/</span>
        <a href="{{ route('positions.index') }}" class="text-primary hover:underline">Posisi</a>
        <span class="mx-1">/</span>
        <span class="text-gray-600">Tambah Baru</span>
      </div>
      <h1 class="text-2xl font-bold text-gray-900 mt-1">Tambah Posisi Baru</h1>
    </div>
    <div class="flex gap-2">
      <a href="{{ route('positions.index') }}" class="btn btn-secondary">
        <i class="fas fa-arrow-left"></i> Kembali
      </a>
    </div>
  </div>

  {{-- Alert flash --}}
  @if(session('success'))
    <div class="alert alert-success">
      <i class="fas fa-check-circle"></i> {{ session('success') }}
    </div>
  @endif
  @if(session('error'))
    <div class="alert alert-error">
      <i class="fas fa-exclamation-circle"></i> {{ session('error') }}
    </div>
  @endif

  {{-- Form --}}
  <div class="form-container">
    <div class="form-header">
      <h2 class="form-title">Informasi Posisi</h2>
      <p class="form-subtitle">Lengkapi informasi untuk posisi baru yang akan dibuka</p>
    </div>

    <form method="POST" action="{{ route('positions.store') }}" id="positionForm" novalidate>
      @csrf

      <div class="form-body">
        {{-- Basic --}}
        <div class="form-grid">
          <div class="form-group">
            <label class="form-label required">Nama Posisi</label>
            <input type="text" name="position_name" class="form-input" placeholder="Contoh: Senior Software Engineer"
                   value="{{ old('position_name') }}" required>
            @error('position_name')
              <div class="error-message"><i class="fas fa-exclamation-triangle"></i>{{ $message }}</div>
            @enderror
          </div>

          <div class="form-group">
            <label class="form-label required">Departemen</label>
            <input type="text" name="department" class="form-input" placeholder="Contoh: Technology"
                   value="{{ old('department') }}" list="departments" required>
            <datalist id="departments">
              @foreach($departments as $dept)
                <option value="{{ $dept }}">
              @endforeach
            </datalist>
            @error('department')
              <div class="error-message"><i class="fas fa-exclamation-triangle"></i>{{ $message }}</div>
            @enderror
          </div>

          <div class="form-group">
            <label class="form-label">Lokasi</label>
            <input type="text" name="location" class="form-input" placeholder="Contoh: Jakarta Pusat"
                   value="{{ old('location') }}" list="locations">
            <datalist id="locations">
              @foreach($locations as $location)
                <option value="{{ $location }}">
              @endforeach
            </datalist>
            @error('location')
              <div class="error-message"><i class="fas fa-exclamation-triangle"></i>{{ $message }}</div>
            @enderror
          </div>

          <div class="form-group">
            <label class="form-label required">Tipe Pekerjaan</label>
            <select name="employment_type" class="form-select" required>
              <option value="">Pilih Tipe Pekerjaan</option>
              @foreach($employmentTypes as $key => $label)
                <option value="{{ $key }}" {{ old('employment_type') == $key ? 'selected' : '' }}>{{ $label }}</option>
              @endforeach
            </select>
            @error('employment_type')
              <div class="error-message"><i class="fas fa-exclamation-triangle"></i>{{ $message }}</div>
            @enderror
          </div>
        </div>

        {{-- Salary --}}
        <div class="form-group">
          <label class="form-label">Rentang Gaji</label>
          <div class="form-row">
            <div>
              <input type="number" name="salary_range_min" class="form-input" placeholder="Gaji minimum" min="0"
                     value="{{ old('salary_range_min') }}">
              <div class="form-help">Gaji minimum (Rp)</div>
              @error('salary_range_min')
                <div class="error-message"><i class="fas fa-exclamation-triangle"></i>{{ $message }}</div>
              @enderror
            </div>
            <div>
              <input type="number" name="salary_range_max" class="form-input" placeholder="Gaji maksimum" min="0"
                     value="{{ old('salary_range_max') }}">
              <div class="form-help">Gaji maksimum (Rp)</div>
              @error('salary_range_max')
                <div class="error-message"><i class="fas fa-exclamation-triangle"></i>{{ $message }}</div>
              @enderror
            </div>
          </div>
          <div class="form-help">Kosongkan jika gaji dapat dinegosiasikan</div>
        </div>

        {{-- Description --}}
        <div class="form-group full-width">
            <label class="form-label">Deskripsi Posisi</label>
            <textarea name="description" class="form-textarea large"
                      placeholder="Deskripsikan tugas, tanggung jawab, dan lingkup pekerjaan...">{{ old('description') }}</textarea>
            <div class="form-help">Jelaskan secara detail tentang posisi ini</div>
            @error('description')
              <div class="error-message"><i class="fas fa-exclamation-triangle"></i>{{ $message }}</div>
            @enderror
        </div>

        {{-- Requirements --}}
        <div class="form-group full-width">
          <label class="form-label">Persyaratan</label>
          <textarea name="requirements" class="form-textarea large"
                    placeholder="Sebutkan persyaratan pendidikan, pengalaman, keahlian yang dibutuhkan...">{{ old('requirements') }}</textarea>
          <div class="form-help">Jelaskan kriteria kandidat yang diharapkan</div>
          @error('requirements')
            <div class="error-message"><i class="fas fa-exclamation-triangle"></i>{{ $message }}</div>
          @enderror
        </div>

        {{-- Dates --}}
        <div class="form-grid">
          <div class="form-group">
            <label class="form-label">Tanggal Posting</label>
            <input type="date" name="posted_date" class="form-input"
                   value="{{ old('posted_date', date('Y-m-d')) }}">
            <div class="form-help">Tanggal posisi mulai dibuka</div>
            @error('posted_date')
              <div class="error-message"><i class="fas fa-exclamation-triangle"></i>{{ $message }}</div>
            @enderror
          </div>

          <div class="form-group">
            <label class="form-label">Tanggal Penutupan</label>
            <input type="date" name="closing_date" class="form-input"
                   value="{{ old('closing_date') }}">
            <div class="form-help">Batas akhir aplikasi (opsional)</div>
            @error('closing_date')
              <div class="error-message"><i class="fas fa-exclamation-triangle"></i>{{ $message }}</div>
            @enderror
          </div>
        </div>

        {{-- Status --}}
        <div class="form-group">
          <div class="checkbox-group">
            <input type="hidden" name="is_active" value="0">
            <input type="checkbox" name="is_active" value="1" class="checkbox-input" id="is_active"
                   {{ old('is_active', '1') ? 'checked' : '' }}>
            <label for="is_active" class="checkbox-label">Aktifkan posisi ini (dapat menerima aplikasi)</label>
          </div>
          <div class="form-help">Centang untuk langsung mengaktifkan posisi setelah dibuat</div>
        </div>
      </div>

      <div class="form-actions">
        <div class="form-actions-left">
          <i class="fas fa-info-circle"></i> Semua field yang wajib diisi ditandai dengan *
        </div>
        <div class="form-actions-right">
          <a href="{{ route('positions.index') }}" class="btn btn-secondary">
            <i class="fas fa-times"></i> Batal
          </a>
          <button type="submit" class="btn btn-success" id="submitBtn">
            <i class="fas fa-save"></i> Simpan Posisi
          </button>
        </div>
      </div>
    </form>
  </div>
@endsection

@push('scripts')
<script>
  // ==== Form handling (validasi ringan + UX) ====
  const form = document.getElementById('positionForm');
  const submitBtn = document.getElementById('submitBtn');

  form.addEventListener('submit', function(e){
    submitBtn.classList.add('loading');
    submitBtn.disabled = true;

    let isValid = true;
    const requiredFields = ['position_name','department','employment_type'];

    requiredFields.forEach(name => {
      const el = form.querySelector(`[name="${name}"]`);
      if (el && !String(el.value||'').trim()) {
        el.classList.add('error');
        isValid = false;
      } else {
        el?.classList.remove('error');
      }
    });

    // Salary range
    const minSalary = form.querySelector('[name="salary_range_min"]');
    const maxSalary = form.querySelector('[name="salary_range_max"]');
    if (minSalary.value && maxSalary.value) {
      if (parseFloat(minSalary.value) > parseFloat(maxSalary.value)) {
        maxSalary.classList.add('error');
        isValid = false;
        if (!maxSalary.parentNode.querySelector('.error-message')) {
          const em = document.createElement('div');
          em.className = 'error-message';
          em.innerHTML = '<i class="fas fa-exclamation-triangle"></i> Gaji maksimum harus lebih besar dari minimum';
          maxSalary.parentNode.appendChild(em);
        }
      } else {
        maxSalary.classList.remove('error');
        const em = maxSalary.parentNode.querySelector('.error-message');
        if (em) em.remove();
      }
    }

    // Date check
    const postedDate = form.querySelector('[name="posted_date"]');
    const closingDate = form.querySelector('[name="closing_date"]');
    if (postedDate.value && closingDate.value) {
      if (new Date(postedDate.value) > new Date(closingDate.value)) {
        closingDate.classList.add('error');
        isValid = false;
        if (!closingDate.parentNode.querySelector('.error-message')) {
          const em = document.createElement('div');
          em.className = 'error-message';
          em.innerHTML = '<i class="fas fa-exclamation-triangle"></i> Tanggal penutupan harus setelah tanggal posting';
          closingDate.parentNode.appendChild(em);
        }
      } else {
        closingDate.classList.remove('error');
        const em = closingDate.parentNode.querySelector('.error-message');
        if (em && em.textContent.includes('Tanggal penutupan')) em.remove();
      }
    }

    if (!isValid) {
      e.preventDefault();
      submitBtn.classList.remove('loading');
      submitBtn.disabled = false;
      const firstError = form.querySelector('.form-input.error, .form-select.error');
      if (firstError){
        firstError.scrollIntoView({behavior:'smooth', block:'center'});
        firstError.focus();
      }
    }
  });

  // Realtime cleanup
  form.querySelectorAll('.form-input, .form-select, .form-textarea').forEach(el => {
    el.addEventListener('input', function(){
      if (this.classList.contains('error') && String(this.value||'').trim()) {
        this.classList.remove('error');
      }
    });
  });

  // Salary live check
  const minSalaryInput = form.querySelector('[name="salary_range_min"]');
  const maxSalaryInput = form.querySelector('[name="salary_range_max"]');
  function validateSalaryRange(){
    if (minSalaryInput.value && maxSalaryInput.value) {
      if (parseFloat(minSalaryInput.value) > parseFloat(maxSalaryInput.value)) {
        maxSalaryInput.classList.add('error');
      } else {
        maxSalaryInput.classList.remove('error');
        const em = maxSalaryInput.parentNode.querySelector('.error-message');
        if (em && em.textContent.includes('maksimum harus')) em.remove();
      }
    }
  }
  minSalaryInput.addEventListener('input', validateSalaryRange);
  maxSalaryInput.addEventListener('input', validateSalaryRange);

  // Dates live check
  const postedDateInput = form.querySelector('[name="posted_date"]');
  const closingDateInput = form.querySelector('[name="closing_date"]');
  function validateDates(){
    if (postedDateInput.value && closingDateInput.value) {
      if (new Date(postedDateInput.value) > new Date(closingDateInput.value)) {
        closingDateInput.classList.add('error');
      } else {
        closingDateInput.classList.remove('error');
        const em = closingDateInput.parentNode.querySelector('.error-message');
        if (em && em.textContent.includes('Tanggal penutupan')) em.remove();
      }
    }
  }
  postedDateInput.addEventListener('change', validateDates);
  closingDateInput.addEventListener('change', validateDates);

  // Auto-hide alerts
  document.querySelectorAll('.alert').forEach(a=>{
    setTimeout(()=>{ a.style.transition='opacity .4s'; a.style.opacity='0';
      setTimeout(()=>a.remove(), 400);
    }, 5000);
  });

  // Draft autosave (localStorage)
  const storageKey = 'position_form_draft';
  window.addEventListener('load', ()=>{
    const draft = localStorage.getItem(storageKey);
    if (draft){
      try{
        const data = JSON.parse(draft);
        Object.keys(data).forEach(k=>{
          const el = form.querySelector(`[name="${k}"]`);
          if (!el) return;
          if (!el.value){
            if (el.type === 'checkbox'){ el.checked = !!data[k]; }
            else { el.value = data[k]; }
          }
        });
      }catch(e){ console.log('Error loading draft:', e); }
    }
  });
  form.querySelectorAll('input, select, textarea').forEach(el=>{
    el.addEventListener('input', ()=>{
      const fd = new FormData(form);
      const obj = {};
      for (const [k,v] of fd.entries()){ obj[k]=v; }
      // simpan state checkbox explicit
      const cb = form.querySelector('[name="is_active"][type="checkbox"]');
      if (cb) obj['is_active'] = cb.checked ? '1' : '0';
      localStorage.setItem(storageKey, JSON.stringify(obj));
    });
  });
  form.addEventListener('submit', ()=>{ setTimeout(()=>localStorage.removeItem(storageKey), 1000); });

  // Format helper untuk salary (opsional, menampilkan pada help)
  document.querySelectorAll('input[type="number"]').forEach(input=>{
    input.addEventListener('blur', function(){
      if (!this.value) return;
      const val = parseFloat(this.value);
      if (isNaN(val)) return;
      const help = this.parentNode.querySelector('.form-help');
      if (help && this.name.includes('salary_range_')){
        const formatted = new Intl.NumberFormat('id-ID',{style:'currency',currency:'IDR',minimumFractionDigits:0}).format(val);
        const base = help.textContent.split('(')[0].trim();
        help.textContent = `${base} (${formatted})`;
      }
    });
  });
</script>
@endpush
