/* ============================================================
 * positions-index.js  —  FINAL
 * ============================================================ */

/* ---------- util ---------- */
const CSRF = () => document.querySelector('meta[name="csrf-token"]')?.content || '';

const showLoading = () => {
  const el = document.getElementById('loadingOverlay');
  if (el) el.style.display = 'flex';
};
const hideLoading = () => {
  const el = document.getElementById('loadingOverlay');
  if (el) el.style.display = 'none';
};

/* ---------- dropdown simple ---------- */
window.toggleDropdown = (btn) => {
  const dd = btn.nextElementSibling;
  document.querySelectorAll('.dropdown-menu').forEach(d => (d === dd ? null : d.classList.remove('show')));
  dd.classList.toggle('show');
};
document.addEventListener('click', e => {
  if (!e.target.closest('.action-dropdown')) {
    document.querySelectorAll('.dropdown-menu').forEach(d => d.classList.remove('show'));
  }
});

/* ========================================================================
 * SWEETALERT: EXACT LOOK (sesuai screenshot)
 * =======================================================================
 * Kita styling via CSS injection agar tampilan dan warna sama.
 */
(function injectSwalSkin() {
  const css = `
    .swal2-popup{border-radius:14px}
    .swal2-title{font-size:2rem;font-weight:700}
    .swal2-html-container{font-size:1.02rem;color:#374151}
    .swal2-styled{min-width:160px;border-radius:10px;padding:.8rem 1.2rem}
    .btn-ghost{background:#6b7280;color:#fff}
    /* icon rings */
    .swal2-icon.swal2-warning{border-color:#fdba74;color:#fb923c}
    .swal2-icon.swal2-question{border-color:#93c5fd;color:#60a5fa}
    /* boxes */
    .box-yellow{background:#fff7ed;border:1px solid #fde68a;border-radius:10px;padding:12px 14px}
    .box-green{background:#ecfdf5;border:1px solid #a7f3d0;border-radius:10px;padding:12px 14px}
    .bullet{display:flex;align-items:center;gap:8px;margin:4px 0}
    .b-strong{font-weight:700;color:#b45309}
    .b-strong-red{font-weight:700;color:#7c2d12}
    .i-green{color:#16a34a}
    .i-amber{color:#f59e0b}
    .i-slate{color:#475569}
    .swal2-textarea{border-radius:10px}
    /* confirm colors persis */
    .btn-confirm-yellow{background:#f59e0b!important}
    .btn-confirm-green{background:#10b981!important}
    .btn-confirm-red{background:#ef4444!important}
  `;
  const style = document.createElement('style');
  style.textContent = css;
  document.head.appendChild(style);
})();

/* ========================================================================
 * DIALOG BUILDERS (HTML-nya dibuat persis seperti yang kamu tunjukkan)
 * =======================================================================
 */

/** Konfirmasi Nonaktifkan */
function askDeactivate(positionName, totals) {
  const { total, active } = totals;

  const html = `
    <p>Posisi "<b>${escapeHtml(positionName)}</b>" akan dinonaktifkan untuk aplikasi baru.</p>
    <div class="box-yellow" style="margin-top:14px">
      <div class="bullet"><span class="i-slate">📊</span>
        <span><b class="b-strong">${total} total kandidat</b> terdaftar</span>
      </div>
      <div class="bullet"><span class="i-amber">⚠️</span>
        <span><b class="b-strong-red">${active} kandidat sedang dalam proses</b></span>
      </div>
      <div class="bullet"><span class="i-amber">⚠️</span>
        <span>Menonaktifkan posisi akan menghentikan aplikasi baru</span>
      </div>
      <div class="bullet"><span class="i-green">✅</span>
        <span>Kandidat yang sudah mendaftar tetap dapat diproses</span>
      </div>
    </div>
    <textarea id="deactReason" class="swal2-textarea" placeholder="Alasan penonaktifan (opsional)..."></textarea>
  `;

  return Swal.fire({
    title: 'Nonaktifkan Posisi?',
    html,
    icon: 'warning',
    showCancelButton: true,
    confirmButtonText: 'Ya, Nonaktifkan',
    cancelButtonText: 'Batal',
    customClass: { confirmButton: 'btn-confirm-yellow', cancelButton: 'btn-ghost' },
    focusConfirm: false
  }).then(res => ({ confirmed: res.isConfirmed, reason: document.getElementById('deactReason')?.value || '' }));
}

/** Konfirmasi Aktifkan */
function askActivate(positionName) {
  const html = `
    <p>Posisi "<b>${escapeHtml(positionName)}</b>" akan diaktifkan dan dapat menerima aplikasi baru.</p>
    <div class="box-green" style="margin-top:14px">
      <div class="bullet"><span class="i-green">✅</span>
        <span>Posisi akan aktif untuk menerima aplikasi baru</span>
      </div>
    </div>
  `;
  return Swal.fire({
    title: 'Aktifkan Posisi?',
    html,
    icon: 'question',
    showCancelButton: true,
    confirmButtonText: 'Ya, Aktifkan',
    cancelButtonText: 'Batal',
    customClass: { confirmButton: 'btn-confirm-green', cancelButton: 'btn-ghost' }
  }).then(res => res.isConfirmed);
}

/** Konfirmasi Hapus (hanya jika total = 0) */
function askDelete(positionName) {
  const html = `
    <p>Apakah Anda yakin ingin menghapus posisi:<br><b>"${escapeHtml(positionName)}"</b></p>
    <div class="box-green" style="margin-top:14px">
      <div class="bullet"><span class="i-green">✅</span>
        <span>Posisi ini belum memiliki kandidat yang mendaftar, aman untuk dihapus.</span>
      </div>
    </div>
  `;
  return Swal.fire({
    title: 'Hapus Posisi?',
    html,
    icon: 'question',
    showCancelButton: true,
    confirmButtonText: 'Ya, Hapus',
    cancelButtonText: 'Batal',
    customClass: { confirmButton: 'btn-confirm-red', cancelButton: 'btn-ghost' }
  }).then(res => res.isConfirmed);
}

/* helper */
function escapeHtml(str) {
  return String(str).replace(/[&<>"']/g, s => ({
    '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#039;'
  }[s]));
}

/* ========================================================================
 * PUBLIC API dipanggil dari blade
 * =======================================================================
 */

/** Toggle status: 'open' | 'close' */
window.togglePositionStatus = async function (id, name, action, activeCandidates, totalCandidates) {
  try {
    if (action === 'open') {
      const ok = await askActivate(name);
      if (!ok) return;

      showLoading();
      const res = await fetch(`/positions/${id}/toggle-status`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json', 'X-CSRF-TOKEN': CSRF() },
        body: JSON.stringify({ action: 'open' })
      }).then(r => r.json());
      hideLoading();

      if (res?.success) {
        await Swal.fire({ icon: 'success', title: 'Berhasil', text: res.message, timer: 1500, showConfirmButton: false });
        location.reload();
      } else {
        Swal.fire({ icon: 'error', title: 'Gagal', text: res?.message || 'Tidak dapat mengaktifkan posisi' });
      }
      return;
    }

    // close (nonaktifkan)
    const { confirmed, reason } = await askDeactivate(name, { total: totalCandidates, active: activeCandidates });
    if (!confirmed) return;

    showLoading();
    const res = await fetch(`/positions/${id}/toggle-status`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json', 'X-CSRF-TOKEN': CSRF() },
      body: JSON.stringify({ action: 'close', reason })
    }).then(r => r.json());
    hideLoading();

    if (res?.success) {
      await Swal.fire({ icon: 'success', title: 'Berhasil', text: res.message, timer: 1500, showConfirmButton: false });
      location.reload();
    } else {
      Swal.fire({ icon: 'error', title: 'Gagal', text: res?.message || 'Tidak dapat menonaktifkan posisi' });
    }
  } catch (e) {
    hideLoading();
    console.error(e);
    Swal.fire({ icon: 'error', title: 'Error', text: 'Terjadi kesalahan jaringan.' });
  }
};

/** Hapus posisi — hanya benar-benar eksekusi setelah tombol merah diklik */
window.deletePosition = async function (id, name, totalCandidates, activeCandidates) {
  // Kalau ada kandidat -> arahkan ke nonaktifkan/transfer (tidak hapus langsung)
  if (totalCandidates > 0) {
    await askDeactivate(name, { total: totalCandidates, active: activeCandidates }); // tampilkan info peringatan yang sama
    // Tidak ada hapus otomatis di sini (sesuai permintaanmu).
    return;
  }

  // total = 0 -> boleh hapus, tapi hanya jika konfirmasi ditekan
  const ok = await askDelete(name);
  if (!ok) return;

  try {
    showLoading();
    const res = await fetch(`/positions/${id}`, {
      method: 'DELETE',
      headers: { 'Content-Type': 'application/json', 'X-CSRF-TOKEN': CSRF() }
    }).then(r => r.json());
    hideLoading();

    if (res?.success) {
      await Swal.fire({ icon: 'success', title: 'Berhasil', text: res.message, timer: 1500, showConfirmButton: false });
      location.reload();
    } else {
      Swal.fire({ icon: 'error', title: 'Gagal', text: res?.message || 'Tidak dapat menghapus posisi.' });
    }
  } catch (e) {
    hideLoading();
    console.error(e);
    Swal.fire({ icon: 'error', title: 'Error', text: 'Terjadi kesalahan jaringan.' });
  }
};

/* ========================================================================
 * FILTER: auto submit (debounce utk search)
 * ======================================================================= */
(function initFilters() {
  const form = document.getElementById('filterForm');
  if (!form) return;

  const search = form.querySelector('.search-input');
  form.querySelectorAll('.filter-select').forEach(sel => {
    sel.addEventListener('change', () => form.submit());
  });

  if (search) {
    let t;
    search.addEventListener('input', () => {
      clearTimeout(t);
      t = setTimeout(() => form.submit(), 500);
    });
  }
})();

/* ========================================================================
 * SIDEBAR: biar tidak numpuk konten saat mobile/desktop
 * (menggunakan layout kamu; di sini hanya close dropdown saat resize)
 * ======================================================================= */
window.addEventListener('resize', () => {
  document.querySelectorAll('.dropdown-menu').forEach(d => d.classList.remove('show'));
});
