<?php

namespace App\Services;

use App\Support\TalentaHmac;
use Illuminate\Http\Client\Pool;
use Illuminate\Http\Client\Response;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class MekariTalentaClient
{
    protected string $baseUrl;
    protected string $hmacUsername;
    protected string $hmacSecret;

    public function __construct()
    {
        $this->baseUrl      = rtrim(config('services.talenta.base_url'), '/');
        $this->hmacUsername = (string) config('services.talenta.hmac_username');
        $this->hmacSecret   = (string) config('services.talenta.hmac_secret');

        if (!$this->baseUrl || !$this->hmacUsername || !$this->hmacSecret) {
            throw new \RuntimeException('Talenta credentials are not configured. Check .env and config/services.php');
        }
    }

    public function get(string $path, array $query = [], int $timeout = 8): array
    {
        $url     = $this->baseUrl . '/' . ltrim($path, '/');
        $headers = TalentaHmac::buildHeaders('GET', $url, $query, $this->hmacUsername, $this->hmacSecret);

        $res = Http::withHeaders($headers)
            ->retry(2, 200)
            ->timeout($timeout)
            ->connectTimeout(3)
            ->get($url, $query);

        if (!$res->successful()) {
            $msg = 'Talenta API error: HTTP ' . $res->status();
            Log::channel('talenta')->error($msg, ['body' => $res->json()]);
            throw new \RuntimeException($msg);
        }

        return $res->json() ?? [];
    }

    /**
     * Ambil banyak halaman secara paralel. Hasil selalu array; request gagal -> [].
     *
     * @param  string $path e.g. '/v2/talenta/v2/employee'
     * @param  int[]  $pages e.g. [1,2,3]
     * @param  array  $commonQuery e.g. ['limit'=>100]
     * @return array [pageNumber => array_json]
     */
    public function getPool(string $path, array $pages, array $commonQuery = [], int $timeout = 8): array
    {
        $url = $this->baseUrl . '/' . ltrim($path, '/');

        $responses = Http::pool(function (Pool $pool) use ($pages, $url, $commonQuery, $timeout) {
            $reqs = [];
            foreach ($pages as $p) {
                $q       = array_merge($commonQuery, ['page' => (int) $p]);
                $headers = TalentaHmac::buildHeaders('GET', $url, $q, $this->hmacUsername, $this->hmacSecret);
                $reqs[]  = $pool->as("p{$p}")
                    ->withHeaders($headers)
                    ->timeout($timeout)
                    ->connectTimeout(3)
                    ->get($url, $q);
            }
            return $reqs;
        });

        $out = [];
        foreach ($responses as $key => $res) {
            $page = (int) str_replace('p', '', (string) $key);
            if ($res instanceof Response) {
                $out[$page] = $res->successful() ? ($res->json() ?? []) : [];
            } else {
                Log::channel('talenta')->warning('Talenta pool failed', [
                    'page' => $page,
                    'type' => is_object($res) ? get_class($res) : gettype($res),
                ]);
                $out[$page] = [];
            }
        }
        ksort($out);
        return $out;
    }
}
