<?php

namespace App\Http\Controllers;

use App\Models\Employee;
use App\Models\Candidate;
use App\Models\Position;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;
// use App\Models\{Employee, Position};

class EmployeeController extends Controller
{
    public function index(Request $request)
    {
        Gate::authorize('hr-access');

        // Ambil kandidat accepted (boleh sesuaikan eager load)
        $accepted = Candidate::query()
            ->where('application_status', 'accepted')
            ->with(['position']) // jika ada relasi position di Candidate
            ->orderByDesc('updated_at')
            ->get();

        // Map employee by candidate_id
        $employeesByCandidate = Employee::query()
            ->whereIn('candidate_id', $accepted->pluck('id'))
            ->get()
            ->keyBy('candidate_id');

        return view('employees.index', compact('accepted', 'employeesByCandidate'));
    }

    public function createFromCandidate($candidateId)
    {
        Gate::authorize('hr-access');

        $candidate = Candidate::findOrFail($candidateId);
        if ($candidate->application_status !== 'accepted') {
            abort(403, 'Kandidat belum berstatus accepted');
        }

        // Master dari positions
        $positions   = Position::orderBy('position_name')->get();
        $departments = Position::select('department')->whereNotNull('department')->distinct()->pluck('department');
        $locations   = Position::select('location')->whereNotNull('location')->distinct()->pluck('location');

        // Branch dibentuk: "PT. Kayu Mebel Indonesia - {Location}"
        $branches = $locations->map(fn($loc) => "PT. Kayu Mebel Indonesia - {$loc}");

        // Job levels (contoh)
        $jobLevels = collect(['MT / Management Trainee','Staff','Senior Staff','Supervisor','Assistant Manager','Manager','Head']);

        // Employment types (sinkron dengan enum positions)
        $employmentTypes = collect(['full-time','part-time','contract','internship']);

        // Contract types lokal HR
        $contractTypes = collect(['PKWT','PKWTT','Magang','Kontrak','Tetap']);

        // Prefill dari candidate
        $prefillPositionName = optional($candidate->position)->position_name ?? $candidate->position_applied;
        $prefillDepartment   = optional($candidate->position)->department     ?? null;
        $prefillLocation     = optional($candidate->position)->location       ?? null;
        $prefillBranch       = $prefillLocation ? "PT. Kayu Mebel Indonesia - {$prefillLocation}" : null;

        return view('employees.create', compact(
            'candidate','positions','departments','branches',
            'jobLevels','employmentTypes','contractTypes',
            'prefillPositionName','prefillDepartment','prefillBranch'
        ));
    }

    public function store(Request $request)
    {
        Gate::authorize('hr-access');

        $data = $request->validate([
            'candidate_id'     => 'integer',
            'position_id'      => 'nullable|integer',
            'employee_code'    => 'nullable|string|max:50',
            'full_name'        => 'nullable|string|max:255',

            'branch'           => 'string|max:255',
            'department'       => 'string|max:255',
            'position_name'    => 'string|max:255',
            'job_level'        => 'string|max:100',

            'employment_type'  => 'in:full-time,part-time,contract,internship',
            'contract_type'    => 'in:PKWT,PKWTT,Magang,Kontrak,Tetap',
            'start_date'       => 'nullable|date',
            'end_date'         => 'nullable|date|after_or_equal:start_date',
            'is_active'        => 'nullable|boolean',

            // 'base_salary'      => 'nullable|numeric|min:0',
            // 'salary_currency'  => 'nullable|string|max:8',
            // 'allowance'        => 'nullable|numeric|min:0',
            // 'meal_transport'   => 'nullable|numeric|min:0',
            // 'other_benefits'   => 'nullable|numeric|min:0',

            // 'npwp'             => 'nullable|string|max:50',
            // 'bpjs_ketenagakerjaan' => 'nullable|string|max:50',
            // 'bpjs_kesehatan'   => 'nullable|string|max:50',
            'tax_status'       => 'nullable|in:TK/0,TK/1,TK/2,TK/3,K/0,K/1,K/2,K/3',

            // 'bank_name'        => 'nullable|string|max:100',
            // 'bank_account_no'  => 'nullable|string|max:100',
            // 'bank_account_holder' => 'nullable|string|max:150',

            'supervisor_name'  => 'nullable|string|max:150',
            'work_schedule'    => 'nullable|string|max:100',
            'work_location_detail' => 'nullable|string|max:150',
        ]);

        $candidate = Candidate::findOrFail($data['candidate_id']);
        if ($candidate->application_status !== 'accepted') {
            return back()->withErrors(['candidate_id' => 'Kandidat belum berstatus accepted']);
        }

        // Default fallback
        $fullName = $data['full_name'] ?? $candidate->full_name ?? 'N/A';

        // Upsert by candidate_id (hindari duplikat)
        $employee = Employee::updateOrCreate(
            ['candidate_id' => $candidate->id],
            array_merge($data, [
                'full_name'       => $fullName,
                // 'salary_currency' => $data['salary_currency'] ?? 'IDR',
                'is_active'       => $data['is_active'] ?? true,
                'company_name'    => 'PT. Kayu Mebel Indonesia',
            ])
        );

        return redirect()->route('employees.show', $employee->id)
            ->with('success', 'Data karyawan berhasil disimpan.');
    }
    
    // public function store(Request $request)
    // {
    //     Gate::authorize('hr-access');

    //     $data = $request->validate([
    //         'candidate_id'     => 'required|integer',
    //         'position_id'      => 'nullable|integer',
    //         'employee_code'    => 'nullable|string|max:50',
    //         'full_name'        => 'nullable|string|max:255',

    //         'branch'           => 'required|string|max:255',
    //         'department'       => 'required|string|max:255',
    //         'position_name'    => 'required|string|max:255',
    //         'job_level'        => 'required|string|max:100',

    //         'employment_type'  => 'required|in:full-time,part-time,contract,internship',
    //         'contract_type'    => 'required|in:PKWT,PKWTT,Magang,Kontrak,Tetap',
    //         'start_date'       => 'nullable|date',
    //         'end_date'         => 'nullable|date|after_or_equal:start_date',
    //         'is_active'        => 'nullable|boolean',

    //         'base_salary'      => 'nullable|numeric|min:0',
    //         'salary_currency'  => 'nullable|string|max:8',
    //         'allowance'        => 'nullable|numeric|min:0',
    //         'meal_transport'   => 'nullable|numeric|min:0',
    //         'other_benefits'   => 'nullable|numeric|min:0',

    //         'npwp'             => 'nullable|string|max:50',
    //         'bpjs_ketenagakerjaan' => 'nullable|string|max:50',
    //         'bpjs_kesehatan'   => 'nullable|string|max:50',
    //         'tax_status'       => 'nullable|in:TK/0,TK/1,TK/2,TK/3,K/0,K/1,K/2,K/3',

    //         'bank_name'        => 'nullable|string|max:100',
    //         'bank_account_no'  => 'nullable|string|max:100',
    //         'bank_account_holder' => 'nullable|string|max:150',

    //         'supervisor_name'  => 'nullable|string|max:150',
    //         'work_schedule'    => 'nullable|string|max:100',
    //         'work_location_detail' => 'nullable|string|max:150',
    //     ]);

    //     $candidate = Candidate::findOrFail($data['candidate_id']);
    //     if ($candidate->application_status !== 'accepted') {
    //         return back()->withErrors(['candidate_id' => 'Kandidat belum berstatus accepted']);
    //     }

    //     // Default fallback
    //     $fullName = $data['full_name'] ?? $candidate->full_name ?? 'N/A';

    //     // Upsert by candidate_id (hindari duplikat)
    //     $employee = Employee::updateOrCreate(
    //         ['candidate_id' => $candidate->id],
    //         array_merge($data, [
    //             'full_name'       => $fullName,
    //             'salary_currency' => $data['salary_currency'] ?? 'IDR',
    //             'is_active'       => $data['is_active'] ?? true,
    //             'company_name'    => 'PT. Kayu Mebel Indonesia',
    //         ])
    //     );

    //     return redirect()->route('employees.show', $employee->id)
    //         ->with('success', 'Data karyawan berhasil disimpan.');
    // }
    public function edit(Employee $employee)
    {
        Gate::authorize('hr-access');

        $positions    = Position::where('is_active', true)->orderBy('position_name')->get();
        $departments  = Position::whereNotNull('department')->distinct()->pluck('department')->sort()->values();
        $locations    = Position::whereNotNull('location')->distinct()->pluck('location')->sort()->values();
        $branches     = $locations->map(fn($loc) => "PT. Kayu Mebel Indonesia - {$loc}");
        $jobLevels    = ['Intern','MT - Management Trainee','Staff','Senior Staff','Supervisor','Asst. Manager','Manager','Senior Manager','General Manager','Director'];
        $employmentTypes = ['full-time','part-time','contract','internship'];
        $contractTypes   = ['PKWT','PKWTT','Probation','Internship'];

        return view('employees.edit', compact(
            'employee','positions','departments','branches','jobLevels','employmentTypes','contractTypes'
        ));
    }

    public function update(Request $request, Employee $employee)
    {
        Gate::authorize('hr-access');

        $data = $request->validate([
            'position_id'      => ['nullable','exists:positions,id'],
            'position_name'    => ['required','string'],
            'department'       => ['required','string'],
            'branch'           => ['required','string'],
            'job_level'        => ['required','string'],
            'employment_type'  => ['required','in:full-time,part-time,contract,internship'],
            'contract_type'    => ['nullable','string'],
            'start_date'       => ['nullable','date'],
            'end_date'         => ['nullable','date','after_or_equal:start_date'],
            'is_active'        => ['required','boolean'],

            // 'salary_currency'  => ['nullable','string'],
            // 'base_salary'      => ['nullable','numeric'],
            // 'allowance'        => ['nullable','numeric'],
            // 'meal_transport'   => ['nullable','numeric'],
            // 'other_benefits'   => ['nullable','numeric'],

            // 'npwp'                 => ['nullable','string'],
            // 'bpjs_ketenagakerjaan' => ['nullable','string'],
            // 'bpjs_kesehatan'       => ['nullable','string'],
            'tax_status'           => ['nullable','string'],

            // 'bank_name'           => ['nullable','string'],
            // 'bank_account_no'     => ['nullable','string'],
            // 'bank_account_holder' => ['nullable','string'],

            'supervisor_name'     => ['nullable','string'],
            'work_schedule'       => ['nullable','string'],
            'work_location_detail'=> ['nullable','string'],
            'employee_code'       => ['nullable','string'],
        ]);

        $employee->update($data);

        return redirect()->route('employees.show', $employee->id)
            ->with('success', 'Data karyawan berhasil diperbarui.');
    }

    public function show($employeeId)
    {
        Gate::authorize('hr-access');

        $employee  = Employee::findOrFail($employeeId);
        $candidate = $employee->candidate; // via model relationship

        return view('employees.show', compact('employee','candidate'));
    }

    public function generateEmployeeCode()
    {
        // Ambil kode terakhir
        $lastEmployee = Employee::orderByDesc('id')->first();
        $lastCode = $lastEmployee ? $lastEmployee->employee_code : null;
        $lastNumber = 0;
        if ($lastCode && preg_match('/KM(\d{8})/', $lastCode, $matches)) {
            $lastNumber = (int)$matches[1];
        }
        $newNumber = $lastNumber + 1;
        $newCode = 'KM' . str_pad($newNumber, 8, '0', STR_PAD_LEFT);
        return $newCode;
    }
}