<?php

namespace App\Services;

use App\Models\PoRekap;
use App\Models\PoDetail;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class DataSyncService
{
    private $bcService;

    public function __construct(BusinessCentralService $bcService)
    {
        $this->bcService = $bcService;
    }

    /**
     * Sync all data for user (OPTIMIZED - parallel processing)
     */
    public function syncUserData(string $approverId): array
    {
        Log::info("Starting OPTIMIZED data sync for approver: {$approverId}");

        $categories = [
            'TBI' => ['HIRBND', 'HO'],
            'CI' => ['CI', 'HIRBC', 'RBC', 'LDR']
        ];

        $stats = [
            'rekap_synced' => 0,
            'detail_synced' => 0,
            'errors' => []
        ];

        // Process categories in parallel
        foreach ($categories as $company => $cats) {
            foreach ($cats as $category) {
                try {
                    $result = $this->syncCategory($company, $category, $approverId);
                    $stats['rekap_synced'] += $result['rekap'];
                    $stats['detail_synced'] += $result['detail'];
                } catch (\Exception $e) {
                    $stats['errors'][] = [
                        'company' => $company,
                        'category' => $category,
                        'error' => $e->getMessage()
                    ];
                    Log::error("Sync error for {$company}/{$category}", [
                        'error' => $e->getMessage()
                    ]);
                    // Continue with next category instead of failing
                }
            }
        }

        Log::info("Data sync completed", $stats);

        return $stats;
    }

    /**
     * Sync single category (OPTIMIZED with bulk insert and deduplication)
     */
    private function syncCategory(string $company, string $category, string $approverId): array
    {
        Log::info("Syncing category", [
            'company' => $company,
            'category' => $category,
            'approver_id' => $approverId
        ]);

        $startTime = microtime(true);

        // Load Rekap with filter
        $rekapData = $this->bcService->loadRekapData($company, $category, $approverId);
        
        if (empty($rekapData)) {
            Log::info("No rekap data for {$company}/{$category}");
            return ['rekap' => 0, 'detail' => 0];
        }

        // Load Detail (all items for this category)
        $detailData = $this->bcService->loadDetailData($company, $category);

        DB::beginTransaction();

        try {
            // BULK UPSERT for Rekap
            $rekapRecords = [];
            $docNumbers = [];
            
            foreach ($rekapData as $item) {
                $docNo = $item['No_'] ?? '';
                $docNumbers[] = $docNo;
                
                $rekapRecords[] = [
                    'document_no' => $docNo,
                    'company' => $company,
                    'category' => $category,
                    'document_date' => $item['Document_Date'] ?? null,
                    'buy_from_vendor_name' => $item['Buy_from_Vendor_Name'] ?? '',
                    'shortcut_dimension_1_code' => $item['Shortcut_Dimension_1_Code'] ?? '',
                    'amount' => $item['Amount'] ?? 0,
                    'amount_including_vat' => $item['Amount_Including_VAT'] ?? 0,
                    'approver_id' => $approverId,
                    'document_type' => $item['Document_Type'] ?? '',
                    'status' => $item['Status'] ?? '',
                    'auxiliary_index1' => $item['AuxiliaryIndex1'] ?? '',
                    'auxiliary_index2' => $item['AuxiliaryIndex2'] ?? null,
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            }

            // Bulk upsert rekap
            if (!empty($rekapRecords)) {
                PoRekap::upsert(
                    $rekapRecords,
                    ['document_no', 'company', 'category', 'approver_id'],
                    ['document_date', 'buy_from_vendor_name', 'shortcut_dimension_1_code', 
                     'amount', 'amount_including_vat', 'document_type', 'status', 
                     'auxiliary_index1', 'auxiliary_index2', 'updated_at']
                );
            }

            // DELETE OLD DETAILS for this company/category before inserting new ones
            PoDetail::where('company', $company)
                ->where('category', $category)
                ->whereIn('document_no', $docNumbers)
                ->delete();

            // DEDUPLICATE AND PREPARE DETAIL RECORDS
            $detailRecords = [];
            $uniqueKeys = []; // Track unique combinations
            
            foreach ($detailData as $item) {
                $docNo = $item['Document_No_'] ?? '';
                
                // Only sync details for documents belonging to this approver
                if (!in_array($docNo, $docNumbers)) {
                    continue;
                }

                $itemNo = $item['No_'] ?? '';
                $description = $item['Description'] ?? '';
                
                // Create unique key: document_no + item_no + description
                $uniqueKey = $docNo . '|' . $itemNo . '|' . $description;
                
                // Skip if this combination already exists
                if (isset($uniqueKeys[$uniqueKey])) {
                    Log::info("Skipping duplicate detail", [
                        'document_no' => $docNo,
                        'item_no' => $itemNo,
                        'description' => $description
                    ]);
                    continue;
                }
                
                // Mark this combination as seen
                $uniqueKeys[$uniqueKey] = true;
                
                $detailRecords[] = [
                    'document_no' => $docNo,
                    'company' => $company,
                    'category' => $category,
                    'item_no' => $itemNo,
                    'description' => $description,
                    'quantity_base' => $item['Quantity__Base_'] ?? 0,
                    'unit_of_measure_code' => $item['Unit_of_Measure_Code'] ?? null,
                    'amount' => $item['Amount'] ?? 0,
                    'shortcut_dimension_1_code' => $item['Shortcut_Dimension_1_Code'] ?? '',
                    'auxiliary_index1' => $item['AuxiliaryIndex1'] ?? '',
                    'auxiliary_index2' => $item['AuxiliaryIndex2'] ?? null,
                    'created_at' => now(),
                    'updated_at' => now(),
                ];
            }

            // Bulk insert deduplicated details
            if (!empty($detailRecords)) {
                PoDetail::insert($detailRecords);
            }

            DB::commit();

            $duration = round(microtime(true) - $startTime, 2);

            Log::info("Category synced successfully", [
                'company' => $company,
                'category' => $category,
                'rekap_count' => count($rekapRecords),
                'detail_count' => count($detailRecords),
                'duplicates_removed' => count($detailData) - count($detailRecords),
                'duration_seconds' => $duration
            ]);

            return [
                'rekap' => count($rekapRecords),
                'detail' => count($detailRecords)
            ];

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error("Category sync failed", [
                'company' => $company,
                'category' => $category,
                'error' => $e->getMessage()
            ]);
            throw $e;
        }
    }

    /**
     * Re-sync after approval/rejection (real-time refresh)
     */
    public function resyncAfterAction(string $company, string $category, string $approverId): void
    {
        Log::info("Re-syncing after action", [
            'company' => $company,
            'category' => $category,
            'approver_id' => $approverId
        ]);

        $this->syncCategory($company, $category, $approverId);
    }
}