<?php

namespace App\Services;

use App\Models\BcToken;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class BusinessCentralService
{
    private $baseUrl;
    private $tenantId;
    private $clientId;
    private $clientSecret;
    private $tokenUrl;
    
    public function __construct()
    {
        $this->baseUrl = env('BC_BASE_URL');
        $this->tenantId = env('AZURE_TENANT_ID');
        $this->clientId = env('AZURE_CLIENT_ID');
        $this->clientSecret = env('AZURE_CLIENT_SECRET');
        $this->tokenUrl = env('AZURE_TOKEN_URL');
    }

    /**
     * Get valid access token (with caching - 99% request reduction)
     */
    public function getAccessToken(): string
    {
        // Check cached token first
        $cachedToken = BcToken::getValidToken();
        
        if ($cachedToken) {
            Log::info('Using cached BC token');
            return $cachedToken;
        }

        // Request new token
        Log::info('Requesting new BC token from Azure');
        
        $response = Http::asForm()->post($this->tokenUrl, [
            'client_id' => $this->clientId,
            'client_secret' => $this->clientSecret,
            'scope' => env('AZURE_SCOPE'),
            'grant_type' => 'client_credentials',
        ]);

        if ($response->failed()) {
            Log::error('Failed to get BC token', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            throw new \Exception('Failed to get access token from Azure');
        }

        $data = $response->json();
        
        // Save token to cache (expires 5 minutes before actual expiry for safety)
        $expiresAt = Carbon::now()->addSeconds($data['expires_in'] - 300);
        
        BcToken::create([
            'access_token' => $data['access_token'],
            'token_type' => $data['token_type'] ?? 'Bearer',
            'expires_in' => $data['expires_in'],
            'expires_at' => $expiresAt,
        ]);

        Log::info('New BC token cached', ['expires_at' => $expiresAt]);

        return $data['access_token'];
    }

    /**
     * Fetch data from BC API with auto-pagination and filter
     */
    public function fetchData(string $endpoint, array $filters = [], int $timeout = 120): array
    {
        $token = $this->getAccessToken();
        $allData = [];
        
        // Build filter query
        $filterQuery = '';
        if (!empty($filters)) {
            $filterConditions = [];
            foreach ($filters as $field => $value) {
                $filterConditions[] = "{$field} eq '{$value}'";
            }
            $filterQuery = '$filter=' . implode(' and ', $filterConditions);
        }
        
        $url = $this->baseUrl . '/' . $endpoint;
        if ($filterQuery) {
            $url .= (strpos($url, '?') === false ? '?' : '&') . $filterQuery;
        }
        
        $pageCount = 0;

        do {
            $pageCount++;
            Log::info("Fetching page {$pageCount}", ['url' => substr($url, 0, 150)]);

            $response = Http::withToken($token)
                ->timeout($timeout)
                ->get($url);

            if ($response->failed()) {
                Log::error('BC API request failed', [
                    'url' => substr($url, 0, 150),
                    'status' => $response->status(),
                    'body' => substr($response->body(), 0, 500)
                ]);
                
                // Don't throw exception, return what we have
                break;
            }

            $data = $response->json();
            
            if (isset($data['value']) && is_array($data['value'])) {
                $allData = array_merge($allData, $data['value']);
                Log::info("Page {$pageCount} fetched", ['records' => count($data['value'])]);
            }

            // Check for next page link
            $url = $data['@odata.nextLink'] ?? null;

        } while ($url !== null && $pageCount < 10); // Limit to 10 pages max

        Log::info('Data fetch completed', [
            'total_pages' => $pageCount,
            'total_records' => count($allData)
        ]);

        return $allData;
    }

    /**
     * Load Rekap data for specific company and category with filter
     */
    public function loadRekapData(string $company, string $category, string $approverId): array
    {
        $companyEncoded = $company === 'TBI' ? env('BC_COMPANY_TBI') : env('BC_COMPANY_CI');
        $endpoint = "Company('{$companyEncoded}')/PO_{$category}";
        
        Log::info("Loading Rekap data with filter", [
            'company' => $company,
            'category' => $category,
            'approver_id' => $approverId,
            'endpoint' => $endpoint
        ]);

        // Use filter to get only relevant data
        $data = $this->fetchData($endpoint, ['Approver_ID' => $approverId]);
        
        return $data;
    }

    /**
     * Load Detail data for specific company and category (no filter - we'll filter in sync)
     */
    public function loadDetailData(string $company, string $category): array
    {
        $companyEncoded = $company === 'TBI' ? env('BC_COMPANY_TBI') : env('BC_COMPANY_CI');
        $endpoint = "Company('{$companyEncoded}')/PO_Item_{$category}";
        
        Log::info("Loading Detail data", [
            'company' => $company,
            'category' => $category,
            'endpoint' => $endpoint
        ]);

        return $this->fetchData($endpoint);
    }

    /**
     * Approve Purchase Order
     */
    public function approvePO(string $company, string $docNo, string $approverId): array
    {
        $token = $this->getAccessToken();
        $companyId = $company === 'TBI' 
            ? '1609a974-a1cf-ed11-a7c9-000d3ac8061e' 
            : '68899f26-1900-ee11-8f70-000d3ac804e2';

        $url = "https://api.businesscentral.dynamics.com/v2.0/{$this->tenantId}/Production/ODataV4/ApproveDocument_approvePO?Company={$companyId}";

        Log::info('Approving PO', [
            'company' => $company,
            'doc_no' => $docNo,
            'approver_id' => $approverId,
        ]);

        $response = Http::withToken($token)
            ->timeout(60)
            ->post($url, [
                'docNo' => $docNo,
                'approverId' => $approverId,
            ]);

        if ($response->failed()) {
            Log::error('Approval failed', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            throw new \Exception("Approval failed: " . $response->body());
        }

        Log::info('PO Approved successfully', ['doc_no' => $docNo]);

        return $response->json();
    }

    /**
     * Reject Purchase Order
     */
    public function rejectPO(string $company, string $docNo, string $approverId): array
    {
        $token = $this->getAccessToken();
        $companyId = $company === 'TBI' 
            ? '1609a974-a1cf-ed11-a7c9-000d3ac8061e' 
            : '68899f26-1900-ee11-8f70-000d3ac804e2';

        $url = "https://api.businesscentral.dynamics.com/v2.0/{$this->tenantId}/Production/ODataV4/ApproveDocument_rejectPO?Company={$companyId}";

        Log::info('Rejecting PO', [
            'company' => $company,
            'doc_no' => $docNo,
            'approver_id' => $approverId,
        ]);

        $response = Http::withToken($token)
            ->timeout(60)
            ->post($url, [
                'docNo' => $docNo,
                'approverId' => $approverId,
            ]);

        if ($response->failed()) {
            Log::error('Rejection failed', [
                'status' => $response->status(),
                'body' => $response->body()
            ]);
            throw new \Exception("Rejection failed: " . $response->body());
        }

        Log::info('PO Rejected successfully', ['doc_no' => $docNo]);

        return $response->json();
    }
}