<?php

namespace App\Http\Controllers;

use Google\Client;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class GoogleOAuthController extends Controller
{
    /**
     * Redirect user to Google OAuth consent screen
     */
    public function redirect()
    {
        $client = new Client();

        $client->setClientId(config('services.google.client_id'));
        $client->setClientSecret(config('services.google.client_secret'));

        // IMPORTANT: must match Google Cloud Console
        $client->setRedirectUri(route('google.oauth.callback'));

        // Required for refresh token
        $client->setAccessType('offline');
        $client->setPrompt('consent');

        // Minimal scope needed for uploads
        $client->addScope(\Google\Service\Drive::DRIVE_FILE);

        return redirect($client->createAuthUrl());
    }

    /**
     * Handle Google OAuth callback
     */
    public function callback(Request $request)
    {
        if (!$request->has('code')) {
            return redirect('/')
                ->with('error', 'Google OAuth failed: missing authorization code.');
        }

        $client = new Client();
        $client->setClientId(config('services.google.client_id'));
        $client->setClientSecret(config('services.google.client_secret'));
        $client->setRedirectUri(route('google.oauth.callback'));

        try {
            $token = $client->fetchAccessTokenWithAuthCode($request->code);

            if (isset($token['error'])) {
                Log::error('Google OAuth token error', $token);

                return redirect('/')
                    ->with('error', 'Google OAuth failed: ' . $token['error']);
            }

            $tokenPath = storage_path('app/google/oauth-token.json');

            // Ensure directory exists
            if (!is_dir(dirname($tokenPath))) {
                mkdir(dirname($tokenPath), 0755, true);
            }

            file_put_contents(
                $tokenPath,
                json_encode($token, JSON_PRETTY_PRINT)
            );

            return redirect('/')
                ->with('success', 'Google Drive connected successfully!');

        } catch (\Throwable $e) {
            Log::error('Google OAuth exception', [
                'error' => $e->getMessage(),
            ]);

            return redirect('/')
                ->with('error', 'Google OAuth exception: ' . $e->getMessage());
        }
    }
}
//90183433273-62v2dm9ufqam1f6n49qccgl50fol3kjo.apps.googleusercontent.com
//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