<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Laravel\Socialite\Facades\Socialite;
use Carbon\Carbon;
use App\Models\User;
use App\Models\Role;
use Illuminate\Support\Facades\Auth;

class AzureAuthController extends Controller
{
    protected $authService;

    public function __construct(\App\Services\AzureAuthService $authService)
    {
        $this->authService = $authService;
    }

    /**
     * ============================================================
     * REDIRECT TO MICROSOFT LOGIN
     * ============================================================
     */
    public function redirectToProvider()
    {
        // Jika token masih valid (< 24 jam), langsung masuk
        //if (session()->has('graph.expires_at') && Auth::check()) {
         //   $expiresAt = Carbon::parse(session('graph.expires_at'));

         //   if (now()->lessThan($expiresAt)) {
         //       return redirect('/forms/upload');
        //    }
        //}

        // Bersihkan session lama agar tidak terjadi state conflict

        return Socialite::driver('microsoft')
            ->scopes(['openid', 'offline_access', 'User.Read', 'Files.Read'])
            ->stateless()
            ->redirect();
    }

    /**
     * ============================================================
     * MICROSOFT CALLBACK HANDLER
     * ============================================================
     */
    public function handleProviderCallback(Request $request)
    {
        try {

            // ================================
            // GET USER FROM MICROSOFT
            // ================================
            $msUser = Socialite::driver('microsoft')
                ->stateless()
                ->user();

            $email = strtolower($msUser->getEmail());
            $name  = $msUser->getName();

            // ================================
            // FIND OR CREATE USER
            // ================================
            $user = User::where('email', $email)->first();

            if (!$user) {
                // get default USER role
                $userRole = Role::where('code', 'USER')->first();

                if (!$userRole) {
                    throw new \Exception('Default USER role not found');
                }

                $user = User::create([
                    'name'    => $name,
                    'email'   => $email,
                    'role_id'=> $userRole->id,
                ]);
            }

            // ================================
            // TOKEN EXPIRY
            // ================================
            $expiresIn = (int) ($msUser->expiresIn ?? 3600);
            $expiresAt = now()->addSeconds(min($expiresIn, 86400));

            // ================================
            // STORE SESSION
            // ================================
            $user->load('role');
            Auth::login($user);
            session([
                'user' => [
                    'id'       => $user->id,
                    'name'     => $user->name,
                    'email'    => $user->email,
                    'role'     => $user->roleCode(),
                    'login_at'=> now()->toDateTimeString(),
                ],
                'graph' => [
                    'access_token'  => $msUser->token,
                    'refresh_token' => $msUser->refreshToken ?? null,
                    'expires_at'    => $expiresAt->toIso8601String(),
                ],
            ]);

            return redirect('/forms/upload');

        } catch (\Throwable $e) {

            Log::error('Azure Login Failed', [
                'message' => $e->getMessage(),
                'file'    => $e->getFile(),
                'line'    => $e->getLine(),
            ]);

            $this->forceLogout($request);
            return redirect('/login');
        }
    }

    /**
     * ============================================================
     * SESSION CLEANER
     * ============================================================
     */
    public function logout(Request $request)
    {
        // 1. Logout dari Laravel Auth
        Auth::logout();

        // 2. Hapus semua session
        $request->session()->invalidate();
        $request->session()->flush();
        $request->session()->regenerateToken();

        // 3. Logout dari Microsoft (WAJIB agar bisa pilih akun ulang)
        $microsoftLogoutUrl =
            'https://login.microsoftonline.com/common/oauth2/v2.0/logout'
            . '?post_logout_redirect_uri=' . urlencode(route('login'));

        return redirect($microsoftLogoutUrl);
    }
}
