<?php
// Start session
session_start();

// Include database connection
require_once '../config/connection.php';

// Check if user is logged in and has IT privileges
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] != 'it') {
    header("Location: ../login.php");
    exit();
}

// Check if form is submitted
if ($_SERVER['REQUEST_METHOD'] != 'POST') {
    header("Location: form_data.php");
    exit();
}

// Initialize response
$success = false;
$message = '';

// Get input mode
$input_mode = $_POST['input_mode'] ?? '';

if (!in_array($input_mode, ['only_master', 'with_stock'])) {
    $_SESSION['error_message'] = "Mode input tidak valid!";
    header("Location: form_data.php?error=1");
    exit();
}

// Get component type if with_stock mode
$component_type = '';
if ($input_mode === 'with_stock') {
    $component_type = $_POST['component_type'] ?? '';
    if (!in_array($component_type, ['finish_only', 'with_steps'])) {
        $_SESSION['error_message'] = "Tipe component tidak valid!";
        header("Location: form_data.php?error=1");
        exit();
    }
}

// Function to get next global sequence number
function getNextSequenceNumber($conn) {
    $max_number = 0;
    
    $queries = [
        "SELECT CAST(code_color AS UNSIGNED) as num FROM master_color WHERE code_color REGEXP '^[0-9]+$' ORDER BY CAST(code_color AS UNSIGNED) DESC LIMIT 1",
        "SELECT CAST(panel_code AS UNSIGNED) as num FROM master_color_panel WHERE panel_code REGEXP '^[0-9]+$' ORDER BY CAST(panel_code AS UNSIGNED) DESC LIMIT 1",
        "SELECT CAST(code_color AS UNSIGNED) as num FROM color_stock_items WHERE code_color REGEXP '^[0-9]+$' ORDER BY CAST(code_color AS UNSIGNED) DESC LIMIT 1"
    ];
    
    foreach ($queries as $query) {
        $result = $conn->query($query);
        if ($result && $result->num_rows > 0) {
            $row = $result->fetch_assoc();
            $current_num = intval($row['num']);
            if ($current_num > $max_number) {
                $max_number = $current_num;
            }
        }
    }
    
    return $max_number + 1;
}

// Start transaction
$conn->begin_transaction();

try {
    // ========================================
    // SECTION 1: Validate and Insert Master Color
    // ========================================
    
    $required_fields = ['code_color', 'color_name', 'buyer_id', 'collection_id', 'lemari', 'lemari_number', 'rack_number', 'position_number'];
    foreach ($required_fields as $field) {
        if (!isset($_POST[$field]) || empty(trim($_POST[$field]))) {
            throw new Exception("Field {$field} wajib diisi!");
        }
    }
    
    // Sanitize inputs
    $code_color = trim($_POST['code_color']);
    $color_name = trim($_POST['color_name']);
    $buyer_id = intval($_POST['buyer_id']);
    $collection_id = intval($_POST['collection_id']);
    $lemari = trim($_POST['lemari']);
    $lemari_number = intval($_POST['lemari_number']);
    $rack_number = trim($_POST['rack_number']);
    $position_number = trim($_POST['position_number']);
    $is_master_only = isset($_POST['is_master_only']) && $_POST['is_master_only'] == '1' ? 1 : 0;
    $description = isset($_POST['description']) ? trim($_POST['description']) : null;
    
    // Validate lemari
    if (!in_array($lemari, ['A', 'B', 'C', 'D'])) {
        throw new Exception("Lemari tidak valid! Harus A, B, C, atau D");
    }
    
    // Validate code_color format
    if (!preg_match('/^\d{5}$/', $code_color)) {
        throw new Exception("Format code color tidak valid! Harus 5 digit.");
    }
    
    // Check if code_color already exists
    $check_query = "SELECT id FROM master_color WHERE code_color = ?";
    $check_stmt = $conn->prepare($check_query);
    $check_stmt->bind_param("s", $code_color);
    $check_stmt->execute();
    $check_result = $check_stmt->get_result();
    
    if ($check_result->num_rows > 0) {
        throw new Exception("Code Color '{$code_color}' sudah ada!");
    }
    $check_stmt->close();
    
    // Insert into master_color table
    $insert_master_query = "INSERT INTO master_color 
                            (code_color, color_name, buyer_id, collection_id, lemari_number, lemari, description, created_at, updated_at) 
                            VALUES (?, ?, ?, ?, ?, ?, ?, NOW(), NOW())";
    
    $stmt_master = $conn->prepare($insert_master_query);
    $stmt_master->bind_param("ssiisss", $code_color, $color_name, $buyer_id, $collection_id, $lemari_number, $lemari, $description);
    
    if (!$stmt_master->execute()) {
        throw new Exception("Gagal insert master color: " . $stmt_master->error);
    }
    
    $master_color_id = $conn->insert_id;
    $stmt_master->close();
    
    
    // ========================================
    // SECTION 2: Auto Insert Master Color Panel (mengikuti Section 1)
    // ========================================
    
    // Generate panel_code dengan nomor urut global
    $panel_code_number = getNextSequenceNumber($conn);
    $panel_code = str_pad($panel_code_number, 5, '0', STR_PAD_LEFT);
    
    // Validate rack_number format
    if (!preg_match('/^\d{4}$/', $rack_number)) {
        throw new Exception("Nomor rack harus 4 digit!");
    }
    
    // Validate position_number format
    if (!preg_match('/^\d{2}$/', $position_number)) {
        throw new Exception("Nomor posisi harus 2 digit!");
    }
    
    // Check if panel_code already exists
    $check_panel_query = "SELECT id FROM master_color_panel WHERE panel_code = ?";
    $check_panel_stmt = $conn->prepare($check_panel_query);
    $check_panel_stmt->bind_param("s", $panel_code);
    $check_panel_stmt->execute();
    $check_panel_result = $check_panel_stmt->get_result();
    
    if ($check_panel_result->num_rows > 0) {
        throw new Exception("Panel Code '{$panel_code}' sudah ada!");
    }
    $check_panel_stmt->close();
    
    // Insert into master_color_panel table (AUTO - mengikuti data Section 1)
    $insert_panel_query = "INSERT INTO master_color_panel 
                           (panel_code, master_color_id, buyer_id, collection_id, lemari, rack_number, position_number, color_name, is_master_only, notes, created_at, updated_at) 
                           VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW(), NOW())";
    
    $stmt_panel = $conn->prepare($insert_panel_query);
    $stmt_panel->bind_param("siiissssis", $panel_code, $master_color_id, $buyer_id, $collection_id, $lemari, $rack_number, $position_number, $color_name, $is_master_only, $description);
    
    if (!$stmt_panel->execute()) {
        throw new Exception("Gagal insert master color panel: " . $stmt_panel->error);
    }
    
    $master_panel_id = $conn->insert_id;
    $stmt_panel->close();
    
    
    // ========================================
    // SECTION 3: Handle Stock Items Based on Mode
    // ========================================
    
    $total_inserted_stock_count = 0;
    $code_barang_start_global = 0;
    $code_barang_end_global = 0;
    
    if ($input_mode === 'with_stock') {
        // Mode: WITH STOCK - Insert stock items
        
        if (!isset($_POST['stock_items']) || !is_array($_POST['stock_items']) || count($_POST['stock_items']) == 0) {
            throw new Exception("Minimal harus ada 1 stock item untuk mode 'With Components Stock'!");
        }
        
        $stock_items = $_POST['stock_items'];
        
        foreach ($stock_items as $index => $item) {
            // UPDATE: Tambahkan lemari_number ke required fields
            $stock_required_fields = ['component_id', 'qty', 'code_barang_start', 'lemari_number', 'lokasi_penyimpanan', 'manufacturing_date', 'expired_date'];
            foreach ($stock_required_fields as $field) {
                if (!isset($item[$field]) || (is_string($item[$field]) && empty(trim($item[$field])))) {
                    throw new Exception("Stock item #{$index}: Field {$field} wajib diisi!");
                }
            }
            
            $component_id = intval($item['component_id']);
            $qty = intval($item['qty']);
            $code_barang_start = trim($item['code_barang_start']); // Get user input
            $lemari_number = trim($item['lemari_number']); // NEW: Get lemari_number
            $lokasi_penyimpanan = trim($item['lokasi_penyimpanan']);
            $manufacturing_date = trim($item['manufacturing_date']);
            $expired_date = trim($item['expired_date']);
            
            // Validate lemari_number (NEW)
            if (empty($lemari_number)) {
                throw new Exception("Stock item #{$index}: Nomor Lemari wajib diisi!");
            }
            
            // Validate code_barang_start format
            if (!preg_match('/^\d{3}$/', $code_barang_start)) {
                throw new Exception("Stock item #{$index}: Code Barang harus 3 digit (001-999)!");
            }
            
            $code_barang_num = intval($code_barang_start);
            if ($code_barang_num < 1 || $code_barang_num > 999) {
                throw new Exception("Stock item #{$index}: Code Barang harus antara 001-999!");
            }
            
            // Validate quantity
            if ($qty <= 0) {
                throw new Exception("Stock item #{$index}: Quantity harus lebih dari 0!");
            }
            
            // Validate component
            $check_component_query = "SELECT id, component_name FROM color_components WHERE id = ?";
            $check_component_stmt = $conn->prepare($check_component_query);
            $check_component_stmt->bind_param("i", $component_id);
            $check_component_stmt->execute();
            $check_component_result = $check_component_stmt->get_result();
            
            if ($check_component_result->num_rows == 0) {
                throw new Exception("Stock item #{$index}: Component tidak valid!");
            }
            
            $component_data = $check_component_result->fetch_assoc();
            $component_name = $component_data['component_name'];
            $check_component_stmt->close();
            
            // Validate dates
            if (!strtotime($manufacturing_date) || !strtotime($expired_date)) {
                throw new Exception("Stock item #{$index}: Format tanggal tidak valid!");
            }
            
            if (strtotime($expired_date) <= strtotime($manufacturing_date)) {
                throw new Exception("Stock item #{$index}: Tanggal kadaluarsa harus setelah tanggal produksi!");
            }
            
            // Set status
            $status = 'available';
            if (strtotime($expired_date) <= time()) {
                $status = 'expired';
            }
            
            // Track code barang range for this item
            if ($code_barang_start_global == 0 || $code_barang_num < $code_barang_start_global) {
                $code_barang_start_global = $code_barang_num;
            }
            
            // ========================================
            // INSERT MULTIPLE ROWS BASED ON QUANTITY
            // ========================================
            // Use user-provided code_barang_start and increment from there
            
            for ($i = 0; $i < $qty; $i++) {
                $current_code_barang = $code_barang_num + $i;
                
                // Validate not exceeding 999
                if ($current_code_barang > 999) {
                    throw new Exception("Stock item #{$index}: Code Barang melebihi batas maksimum 999!");
                }
                
                $code_barang_lemari5 = str_pad($current_code_barang, 3, '0', STR_PAD_LEFT);
                
                // Check if code_barang_lemari5 already exists
                $check_code_barang_query = "SELECT id, master_color_id, component_id FROM color_stock_items WHERE code_barang_lemari5 = ?";
                $check_code_barang_stmt = $conn->prepare($check_code_barang_query);
                $check_code_barang_stmt->bind_param("s", $code_barang_lemari5);
                $check_code_barang_stmt->execute();
                $check_code_barang_result = $check_code_barang_stmt->get_result();
                
                if ($check_code_barang_result->num_rows > 0) {
                    throw new Exception("Stock item #{$index}: Code Barang {$code_barang_lemari5} sudah digunakan!");
                }
                $check_code_barang_stmt->close();
                
                // Generate unique stock item code untuk setiap unit
                $stock_code_number = getNextSequenceNumber($conn);
                $stock_code_color = str_pad($stock_code_number, 5, '0', STR_PAD_LEFT);
                
                // UPDATE: Insert stock item dengan lemari_number (qty tetap = 1 untuk setiap unit)
                $insert_stock_query = "INSERT INTO color_stock_items 
                                       (master_color_id, code_color, component_id, qty, lokasi_penyimpanan, code_barang_lemari5, lemari_number, manufacturing_date, expired_date, status, created_at, updated_at) 
                                       VALUES (?, ?, ?, 1, ?, ?, ?, ?, ?, ?, NOW(), NOW())";
                
                $stmt_stock = $conn->prepare($insert_stock_query);
                $stmt_stock->bind_param("isissssss", $master_color_id, $stock_code_color, $component_id, $lokasi_penyimpanan, $code_barang_lemari5, $lemari_number, $manufacturing_date, $expired_date, $status);
                
                if (!$stmt_stock->execute()) {
                    throw new Exception("Gagal insert stock item #{$index} unit #{$i}: " . $stmt_stock->error);
                }
                
                $stmt_stock->close();
                $total_inserted_stock_count++;
                
                // Track the highest code
                if ($current_code_barang > $code_barang_end_global) {
                    $code_barang_end_global = $current_code_barang;
                }
            }
        }
        
        $component_type_label = $component_type === 'finish_only' ? 'Hanya Finish' : 'Beserta Step';
        $code_barang_range = str_pad($code_barang_start_global, 3, '0', STR_PAD_LEFT) . ' - ' . str_pad($code_barang_end_global, 3, '0', STR_PAD_LEFT);
        
        $message = "Data berhasil disimpan!\n" .
                   "Master Color: {$code_color}\n" .
                   "Panel: {$panel_code}\n" .
                   "Tipe Component: {$component_type_label}\n" .
                   "Total Stock Items: {$total_inserted_stock_count} units\n" .
                   "Range Code Barang Lemari 5: {$code_barang_range}";
        
    } else {
        // Mode: ONLY MASTER - Skip stock items
        
        $message = "Data berhasil disimpan! Master Color: {$code_color}, Panel: {$panel_code} (Mode: Only Master - Tanpa Stock Items)";
    }
    
    
    // ========================================
    // COMMIT TRANSACTION
    // ========================================
    
    $conn->commit();
    
    $success = true;
    
    // Log audit
    $audit_query = "INSERT INTO audit_logs (action_type, user_id, user_type, details, timestamp, ip_address) 
                    VALUES ('INSERT_COLOR_DATA', ?, 'it', ?, NOW(), ?)";
    $audit_stmt = $conn->prepare($audit_query);
    $user_id = $_SESSION['user_id'];
    
    if ($input_mode === 'with_stock') {
        $audit_details = "Inserted Master Color: {$code_color}, Panel: {$panel_code}, Tipe Component: {$component_type_label}, Total Stock Items: {$total_inserted_stock_count} units, Code Barang Range: {$code_barang_range} (Mode: With Stock)";
    } else {
        $audit_details = "Inserted Master Color: {$code_color}, Panel: {$panel_code} (Mode: Only Master)";
    }
    
    $ip_address = $_SERVER['REMOTE_ADDR'];
    $audit_stmt->bind_param("iss", $user_id, $audit_details, $ip_address);
    $audit_stmt->execute();
    $audit_stmt->close();
    
    $_SESSION['success_message'] = $message;
    
    header("Location: form_data.php?success=1");
    exit();
    
} catch (Exception $e) {
    $conn->rollback();
    
    $_SESSION['error_message'] = "Error: " . $e->getMessage();
    
    header("Location: form_data.php?error=1");
    exit();
    
} finally {
    $conn->close();
}
?>