<?php
session_start();
require_once '../config/connection.php';

// Check if user is logged in
if (!isset($_SESSION['user_id']) || $_SESSION['user_type'] != 'it') {
    echo json_encode(['success' => false, 'message' => 'Unauthorized']);
    exit();
}

header('Content-Type: application/json');

$action = isset($_POST['action']) ? $_POST['action'] : '';

// Function to get next global sequence number
function getNextSequenceNumber($conn) {
    $max_number = 0;
    
    $queries = [
        "SELECT CAST(code_color AS UNSIGNED) as num FROM master_color WHERE code_color REGEXP '^[0-9]+$' ORDER BY CAST(code_color AS UNSIGNED) DESC LIMIT 1",
        "SELECT CAST(panel_code AS UNSIGNED) as num FROM master_color_panel WHERE panel_code REGEXP '^[0-9]+$' ORDER BY CAST(panel_code AS UNSIGNED) DESC LIMIT 1",
        "SELECT CAST(code_color AS UNSIGNED) as num FROM color_stock_items WHERE code_color REGEXP '^[0-9]+$' ORDER BY CAST(code_color AS UNSIGNED) DESC LIMIT 1"
    ];
    
    foreach ($queries as $query) {
        $result = $conn->query($query);
        if ($result && $result->num_rows > 0) {
            $row = $result->fetch_assoc();
            $current_num = intval($row['num']);
            if ($current_num > $max_number) {
                $max_number = $current_num;
            }
        }
    }
    
    return $max_number + 1;
}

$conn->begin_transaction();

try {
    switch($action) {
        case 'add_component_to_master':
            // Menambah component baru atau existing ke master color
            
            $master_color_id = isset($_POST['master_color_id']) ? (int)$_POST['master_color_id'] : 0;
            $component_type = isset($_POST['component_type']) ? $_POST['component_type'] : '';
            $qty = isset($_POST['qty']) ? (int)$_POST['qty'] : 0;
            $code_barang_start = isset($_POST['code_barang_start']) ? trim($_POST['code_barang_start']) : '';
            $lemari_number = isset($_POST['lemari_number']) ? trim($_POST['lemari_number']) : ''; // NEW
            $lokasi_penyimpanan = isset($_POST['lokasi_penyimpanan']) ? $conn->real_escape_string(trim($_POST['lokasi_penyimpanan'])) : '';
            $manufacturing_date = isset($_POST['manufacturing_date']) ? $_POST['manufacturing_date'] : '';
            $expired_date = isset($_POST['expired_date']) ? $_POST['expired_date'] : '';
            
            // Validasi
            if ($master_color_id <= 0) {
                throw new Exception('Master color tidak valid');
            }
            
            if ($qty <= 0) {
                throw new Exception('Quantity harus lebih dari 0');
            }
            
            if (!preg_match('/^\d{3}$/', $code_barang_start)) {
                throw new Exception('Code barang harus 3 digit (001-999)');
            }
            
            // NEW: Validasi lemari_number
            if (empty($lemari_number)) {
                throw new Exception('Nomor Lemari wajib diisi');
            }
            
            if (empty($lokasi_penyimpanan) || empty($manufacturing_date) || empty($expired_date)) {
                throw new Exception('Semua field wajib diisi');
            }
            
            // Validasi tanggal
            if (strtotime($expired_date) <= strtotime($manufacturing_date)) {
                throw new Exception('Tanggal kadaluarsa harus setelah tanggal produksi');
            }
            
            // Determine component_id
            $component_id = 0;
            
            if ($component_type === 'existing') {
                // Gunakan component yang sudah ada
                $component_id = isset($_POST['existing_component_id']) ? (int)$_POST['existing_component_id'] : 0;
                
                if ($component_id <= 0) {
                    throw new Exception('Silakan pilih component');
                }
                
                // Validasi component exists
                $check_component = "SELECT id FROM color_components WHERE id = $component_id";
                $result_check = $conn->query($check_component);
                
                if ($result_check->num_rows == 0) {
                    throw new Exception('Component tidak ditemukan');
                }
                
            } else if ($component_type === 'new') {
                // Buat component baru
                $new_component_name = isset($_POST['new_component_name']) ? $conn->real_escape_string(trim($_POST['new_component_name'])) : '';
                $new_component_description = isset($_POST['new_component_description']) ? $conn->real_escape_string(trim($_POST['new_component_description'])) : '';
                
                if (empty($new_component_name)) {
                    throw new Exception('Nama component baru wajib diisi');
                }
                
                // Cek apakah component name sudah ada
                $check_name = "SELECT id FROM color_components WHERE component_name = '$new_component_name'";
                $result_name = $conn->query($check_name);
                
                if ($result_name->num_rows > 0) {
                    throw new Exception('Nama component sudah ada. Silakan gunakan component yang sudah ada atau gunakan nama lain.');
                }
                
                // Insert new component
                $insert_component = "INSERT INTO color_components (component_name, description, created_at) 
                                    VALUES ('$new_component_name', '$new_component_description', NOW())";
                
                if (!$conn->query($insert_component)) {
                    throw new Exception('Gagal menambahkan component baru: ' . $conn->error);
                }
                
                $component_id = $conn->insert_id;
                
            } else {
                throw new Exception('Tipe component tidak valid');
            }
            
            // Validasi code barang start
            $code_barang_num = intval($code_barang_start);
            if ($code_barang_num < 1 || $code_barang_num > 999) {
                throw new Exception('Code barang harus antara 001-999');
            }
            
            // Validasi range tidak melebihi 999
            $code_barang_end = $code_barang_num + $qty - 1;
            if ($code_barang_end > 999) {
                throw new Exception('Range code barang melebihi batas maksimal 999');
            }
            
            // Set status
            $status = 'available';
            if (strtotime($expired_date) <= time()) {
                $status = 'expired';
            }
            
            // Insert multiple stock items based on qty
            $inserted_count = 0;
            
            for ($i = 0; $i < $qty; $i++) {
                $current_code_barang = $code_barang_num + $i;
                $code_barang_lemari5 = str_pad($current_code_barang, 3, '0', STR_PAD_LEFT);
                
                // Check if code_barang_lemari5 already exists
                $check_code_query = "SELECT id FROM color_stock_items WHERE code_barang_lemari5 = '$code_barang_lemari5'";
                $check_code_result = $conn->query($check_code_query);
                
                if ($check_code_result->num_rows > 0) {
                    throw new Exception("Code Barang {$code_barang_lemari5} sudah digunakan!");
                }
                
                // Generate unique stock code_color
                $stock_code_number = getNextSequenceNumber($conn);
                $stock_code_color = str_pad($stock_code_number, 5, '0', STR_PAD_LEFT);
                
                // UPDATE: Insert stock item dengan lemari_number (qty tetap = 1)
                $insert_stock = "INSERT INTO color_stock_items 
                                (master_color_id, code_color, component_id, qty, lokasi_penyimpanan, code_barang_lemari5, lemari_number,
                                manufacturing_date, expired_date, status, created_at, updated_at) 
                                VALUES 
                                ($master_color_id, '$stock_code_color', $component_id, 1, '$lokasi_penyimpanan', '$code_barang_lemari5', '$lemari_number',
                                '$manufacturing_date', '$expired_date', '$status', NOW(), NOW())";
                
                if (!$conn->query($insert_stock)) {
                    throw new Exception('Gagal insert stock item #' . ($i + 1) . ': ' . $conn->error);
                }
                
                $inserted_count++;
            }
            
            $conn->commit();
            
            // Log audit
            $audit_query = "INSERT INTO audit_logs (action_type, user_id, user_type, details, timestamp, ip_address) 
                           VALUES ('ADD_COMPONENT_TO_MASTER', ?, 'it', ?, NOW(), ?)";
            $audit_stmt = $conn->prepare($audit_query);
            $user_id = $_SESSION['user_id'];
            $code_range = str_pad($code_barang_num, 3, '0', STR_PAD_LEFT) . ' - ' . str_pad($code_barang_end, 3, '0', STR_PAD_LEFT);
            $audit_details = "Added $inserted_count stock items to master_color_id: $master_color_id, component_id: $component_id, lemari_number: $lemari_number, code_barang_range: $code_range";
            $ip_address = $_SERVER['REMOTE_ADDR'];
            $audit_stmt->bind_param("iss", $user_id, $audit_details, $ip_address);
            $audit_stmt->execute();
            $audit_stmt->close();
            
            echo json_encode([
                'success' => true, 
                'message' => "Berhasil menambahkan $inserted_count stock items dengan range code barang: $code_range"
            ]);
            break;
            
        case 'update_code_barang':
            // Update code_barang_lemari5 untuk stock item
            
            $stock_id = isset($_POST['stock_id']) ? (int)$_POST['stock_id'] : 0;
            $new_code = isset($_POST['new_code']) ? trim($_POST['new_code']) : '';
            
            // Validasi
            if ($stock_id <= 0) {
                throw new Exception('Stock ID tidak valid');
            }
            
            if (!preg_match('/^\d{3}$/', $new_code)) {
                throw new Exception('Code barang harus 3 digit (001-999)');
            }
            
            // Check if stock exists
            $check_stock = "SELECT id, code_barang_lemari5 FROM color_stock_items WHERE id = $stock_id";
            $result_stock = $conn->query($check_stock);
            
            if ($result_stock->num_rows == 0) {
                throw new Exception('Stock item tidak ditemukan');
            }
            
            $stock_data = $result_stock->fetch_assoc();
            $old_code = $stock_data['code_barang_lemari5'];
            
            // Check if new code already exists (except current stock)
            $check_code = "SELECT id FROM color_stock_items WHERE code_barang_lemari5 = '$new_code' AND id != $stock_id";
            $result_code = $conn->query($check_code);
            
            if ($result_code->num_rows > 0) {
                throw new Exception('Code barang sudah digunakan oleh stock item lain');
            }
            
            // Update code barang
            $update_query = "UPDATE color_stock_items SET code_barang_lemari5 = '$new_code', updated_at = NOW() WHERE id = $stock_id";
            
            if (!$conn->query($update_query)) {
                throw new Exception('Gagal update code barang: ' . $conn->error);
            }
            
            $conn->commit();
            
            // Log audit
            $audit_query = "INSERT INTO audit_logs (action_type, user_id, user_type, details, timestamp, ip_address) 
                           VALUES ('UPDATE_CODE_BARANG', ?, 'it', ?, NOW(), ?)";
            $audit_stmt = $conn->prepare($audit_query);
            $user_id = $_SESSION['user_id'];
            $audit_details = "Updated code_barang for stock_id: $stock_id from $old_code to $new_code";
            $ip_address = $_SERVER['REMOTE_ADDR'];
            $audit_stmt->bind_param("iss", $user_id, $audit_details, $ip_address);
            $audit_stmt->execute();
            $audit_stmt->close();
            
            echo json_encode([
                'success' => true, 
                'message' => 'Code barang berhasil diupdate'
            ]);
            break;
            
        case 'delete_stock_item':
            // Menghapus stock item
            
            $stock_id = isset($_POST['stock_id']) ? (int)$_POST['stock_id'] : 0;
            
            if ($stock_id <= 0) {
                throw new Exception('Stock ID tidak valid');
            }
            
            // Check if stock exists
            $check_stock = "SELECT id, code_barang_lemari5, status FROM color_stock_items WHERE id = $stock_id";
            $result_stock = $conn->query($check_stock);
            
            if ($result_stock->num_rows == 0) {
                throw new Exception('Stock item tidak ditemukan');
            }
            
            $stock_data = $result_stock->fetch_assoc();
            
            // Check if stock is borrowed
            if ($stock_data['status'] === 'borrowed') {
                throw new Exception('Stock item sedang dipinjam, tidak dapat dihapus');
            }
            
            // Delete stock item
            $delete_query = "DELETE FROM color_stock_items WHERE id = $stock_id";
            
            if (!$conn->query($delete_query)) {
                throw new Exception('Gagal menghapus stock item: ' . $conn->error);
            }
            
            $conn->commit();
            
            // Log audit
            $audit_query = "INSERT INTO audit_logs (action_type, user_id, user_type, details, timestamp, ip_address) 
                           VALUES ('DELETE_STOCK_ITEM', ?, 'it', ?, NOW(), ?)";
            $audit_stmt = $conn->prepare($audit_query);
            $user_id = $_SESSION['user_id'];
            $code_barang = $stock_data['code_barang_lemari5'];
            $audit_details = "Deleted stock_id: $stock_id with code_barang: $code_barang";
            $ip_address = $_SERVER['REMOTE_ADDR'];
            $audit_stmt->bind_param("iss", $user_id, $audit_details, $ip_address);
            $audit_stmt->execute();
            $audit_stmt->close();
            
            echo json_encode([
                'success' => true, 
                'message' => 'Stock item berhasil dihapus'
            ]);
            break;
            
        case 'add_component':
            // Menambah component baru ke tabel color_components (standalone)
            
            $component_name = isset($_POST['component_name']) ? $conn->real_escape_string(trim($_POST['component_name'])) : '';
            $description = isset($_POST['description']) ? $conn->real_escape_string(trim($_POST['description'])) : '';
            
            if (empty($component_name)) {
                throw new Exception('Nama component wajib diisi');
            }
            
            // Cek apakah component sudah ada
            $check_query = "SELECT id FROM color_components WHERE component_name = '$component_name'";
            $check_result = $conn->query($check_query);
            
            if($check_result->num_rows > 0) {
                throw new Exception('Component dengan nama tersebut sudah ada');
            }
            
            $query = "INSERT INTO color_components (component_name, description, created_at) 
                     VALUES ('$component_name', '$description', NOW())";
            
            if(!$conn->query($query)) {
                throw new Exception($conn->error);
            }
            
            $conn->commit();
            
            // Log audit
            $audit_query = "INSERT INTO audit_logs (action_type, user_id, user_type, details, timestamp, ip_address) 
                           VALUES ('ADD_COMPONENT', ?, 'it', ?, NOW(), ?)";
            $audit_stmt = $conn->prepare($audit_query);
            $user_id = $_SESSION['user_id'];
            $audit_details = "Added new component: $component_name";
            $ip_address = $_SERVER['REMOTE_ADDR'];
            $audit_stmt->bind_param("iss", $user_id, $audit_details, $ip_address);
            $audit_stmt->execute();
            $audit_stmt->close();
            
            echo json_encode(['success' => true, 'message' => 'Component berhasil ditambahkan']);
            break;
            
        case 'delete_component':
            // Menghapus component dari tabel color_components
            
            $component_id = isset($_POST['component_id']) ? (int)$_POST['component_id'] : 0;
            
            if ($component_id <= 0) {
                throw new Exception('Component ID tidak valid');
            }
            
            // Cek apakah component masih digunakan
            $check_query = "SELECT COUNT(*) as total FROM color_stock_items WHERE component_id = $component_id";
            $check_result = $conn->query($check_query);
            $check_data = $check_result->fetch_assoc();
            
            if($check_data['total'] > 0) {
                throw new Exception('Component masih digunakan di ' . $check_data['total'] . ' stock item(s). Hapus stock items terlebih dahulu.');
            }
            
            // Get component name for audit
            $get_name = "SELECT component_name FROM color_components WHERE id = $component_id";
            $result_name = $conn->query($get_name);
            $component_name = '';
            if ($result_name->num_rows > 0) {
                $row = $result_name->fetch_assoc();
                $component_name = $row['component_name'];
            }
            
            $query = "DELETE FROM color_components WHERE id = $component_id";
            
            if(!$conn->query($query)) {
                throw new Exception($conn->error);
            }
            
            $conn->commit();
            
            // Log audit
            $audit_query = "INSERT INTO audit_logs (action_type, user_id, user_type, details, timestamp, ip_address) 
                           VALUES ('DELETE_COMPONENT', ?, 'it', ?, NOW(), ?)";
            $audit_stmt = $conn->prepare($audit_query);
            $user_id = $_SESSION['user_id'];
            $audit_details = "Deleted component: $component_name (ID: $component_id)";
            $ip_address = $_SERVER['REMOTE_ADDR'];
            $audit_stmt->bind_param("iss", $user_id, $audit_details, $ip_address);
            $audit_stmt->execute();
            $audit_stmt->close();
            
            echo json_encode(['success' => true, 'message' => 'Component berhasil dihapus']);
            break;
            
        case 'edit_stock_item':
            // Edit/Update stock item
            
            $stock_id = isset($_POST['stock_id']) ? (int)$_POST['stock_id'] : 0;
            $code_barang_lemari5 = isset($_POST['code_barang_lemari5']) ? trim($_POST['code_barang_lemari5']) : '';
            $lemari_number = isset($_POST['lemari_number']) ? trim($_POST['lemari_number']) : ''; // NEW
            $qty = isset($_POST['qty']) ? (int)$_POST['qty'] : 0;
            $lokasi_penyimpanan = isset($_POST['lokasi_penyimpanan']) ? $conn->real_escape_string(trim($_POST['lokasi_penyimpanan'])) : '';
            $manufacturing_date = isset($_POST['manufacturing_date']) ? $_POST['manufacturing_date'] : '';
            $expired_date = isset($_POST['expired_date']) ? $_POST['expired_date'] : '';
            $status = isset($_POST['status']) ? $_POST['status'] : '';
            
            // Validasi
            if ($stock_id <= 0) {
                throw new Exception('Stock ID tidak valid');
            }
            
            if (!preg_match('/^\d{3}$/', $code_barang_lemari5)) {
                throw new Exception('Code barang harus 3 digit (001-999)');
            }
            
            // NEW: Validasi lemari_number
            if (empty($lemari_number)) {
                throw new Exception('Nomor Lemari wajib diisi');
            }
            
            if ($qty <= 0) {
                throw new Exception('Quantity harus lebih dari 0');
            }
            
            if (empty($lokasi_penyimpanan) || empty($manufacturing_date) || empty($expired_date)) {
                throw new Exception('Semua field wajib diisi');
            }
            
            if (!in_array($status, ['available', 'borrowed', 'expired'])) {
                throw new Exception('Status tidak valid');
            }
            
            // Validasi tanggal
            if (strtotime($expired_date) <= strtotime($manufacturing_date)) {
                throw new Exception('Tanggal kadaluarsa harus setelah tanggal produksi');
            }
            
            // Check if stock exists
            $check_stock = "SELECT id, code_barang_lemari5 FROM color_stock_items WHERE id = $stock_id";
            $result_stock = $conn->query($check_stock);
            
            if ($result_stock->num_rows == 0) {
                throw new Exception('Stock item tidak ditemukan');
            }
            
            $stock_data = $result_stock->fetch_assoc();
            $old_code = $stock_data['code_barang_lemari5'];
            
            // Check if new code already exists (except current stock)
            if ($code_barang_lemari5 !== $old_code) {
                $check_code = "SELECT id FROM color_stock_items WHERE code_barang_lemari5 = '$code_barang_lemari5' AND id != $stock_id";
                $result_code = $conn->query($check_code);
                
                if ($result_code->num_rows > 0) {
                    throw new Exception('Code barang sudah digunakan oleh stock item lain');
                }
            }
            
            // UPDATE: Update stock item dengan lemari_number
            $update_query = "UPDATE color_stock_items SET 
                             code_barang_lemari5 = '$code_barang_lemari5',
                             lemari_number = '$lemari_number',
                             qty = $qty,
                             lokasi_penyimpanan = '$lokasi_penyimpanan',
                             manufacturing_date = '$manufacturing_date',
                             expired_date = '$expired_date',
                             status = '$status',
                             updated_at = NOW()
                             WHERE id = $stock_id";
            
            if (!$conn->query($update_query)) {
                throw new Exception('Gagal update stock item: ' . $conn->error);
            }
            
            $conn->commit();
            
            // Log audit
            $audit_query = "INSERT INTO audit_logs (action_type, user_id, user_type, details, timestamp, ip_address) 
                           VALUES ('EDIT_STOCK_ITEM', ?, 'it', ?, NOW(), ?)";
            $audit_stmt = $conn->prepare($audit_query);
            $user_id = $_SESSION['user_id'];
            $audit_details = "Updated stock_id: $stock_id, code_barang: $code_barang_lemari5, lemari_number: $lemari_number, qty: $qty, status: $status";
            $ip_address = $_SERVER['REMOTE_ADDR'];
            $audit_stmt->bind_param("iss", $user_id, $audit_details, $ip_address);
            $audit_stmt->execute();
            $audit_stmt->close();
            
            echo json_encode([
                'success' => true, 
                'message' => 'Stock item berhasil diupdate'
            ]);
            break;
            
        default:
            throw new Exception('Invalid action');
            break;
    }
    
} catch(Exception $e) {
    $conn->rollback();
    echo json_encode(['success' => false, 'message' => $e->getMessage()]);
} finally {
    $conn->close();
}
?>