<?php
// delete_all_qr.php
require_once '../config/connection.php';

// Handle AJAX requests
if (isset($_POST['action']) && $_POST['action'] === 'delete_all_qr') {
    header('Content-Type: application/json');
    
    try {
        // Mulai transaction
        $conn->autocommit(FALSE);
        
        // Hitung QR code yang ada sebelum dihapus
        $count_stock_qr = "SELECT COUNT(*) as total FROM color_stock_items WHERE qr_code IS NOT NULL";
        $result_stock_qr = $conn->query($count_stock_qr);
        $total_stock_qr = $result_stock_qr ? $result_stock_qr->fetch_assoc()['total'] : 0;
        
        $count_master_qr = "SELECT COUNT(*) as total FROM master_color WHERE qr_code IS NOT NULL";
        $result_master_qr = $conn->query($count_master_qr);
        $total_master_qr = $result_master_qr ? $result_master_qr->fetch_assoc()['total'] : 0;
        
        // Hapus QR code dari tabel color_stock_items
        $query_stock = "UPDATE color_stock_items SET qr_code = NULL WHERE qr_code IS NOT NULL";
        $result_stock = $conn->query($query_stock);
        
        if (!$result_stock) {
            throw new Exception("Gagal menghapus QR code dari color stock items: " . $conn->error);
        }
        
        // Hapus QR code dari tabel master_color
        $query_master = "UPDATE master_color SET qr_code = NULL WHERE qr_code IS NOT NULL";
        $result_master = $conn->query($query_master);
        
        if (!$result_master) {
            throw new Exception("Gagal menghapus QR code dari master color: " . $conn->error);
        }
        
        // Commit transaction
        $conn->commit();
        $conn->autocommit(TRUE);
        
        // Log audit trail
        $log_details = "Deleted all QR codes - Stock Items: {$total_stock_qr}, Master Colors: {$total_master_qr}";
        $log_query = "INSERT INTO audit_logs (action_type, user_id, user_type, details, ip_address) 
                      VALUES ('DELETE_ALL_QR', 1, 'admin', ?, ?)";
        $stmt = $conn->prepare($log_query);
        $ip_address = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $stmt->bind_param("ss", $log_details, $ip_address);
        $stmt->execute();
        $stmt->close();
        
        echo json_encode([
            'success' => true,
            'message' => 'Semua QR code berhasil dihapus',
            'summary' => [
                'stock_items_cleared' => $total_stock_qr,
                'master_colors_cleared' => $total_master_qr,
                'total_cleared' => $total_stock_qr + $total_master_qr
            ]
        ]);
        
    } catch (Exception $e) {
        // Rollback jika ada error
        $conn->rollback();
        $conn->autocommit(TRUE);
        
        echo json_encode([
            'success' => false,
            'message' => 'Error: ' . $e->getMessage()
        ]);
    }
    
    $conn->close();
    exit;
}

// Set page title
$page_title = "Delete All QR Codes";

// Include header
include '../template_it/header.php';

// Get current QR code counts
try {
    $count_stock_qr = "SELECT COUNT(*) as total FROM color_stock_items WHERE qr_code IS NOT NULL";
    $result_stock_qr = $conn->query($count_stock_qr);
    $total_stock_qr = $result_stock_qr ? $result_stock_qr->fetch_assoc()['total'] : 0;
    
    $count_master_qr = "SELECT COUNT(*) as total FROM master_color WHERE qr_code IS NOT NULL";
    $result_master_qr = $conn->query($count_master_qr);
    $total_master_qr = $result_master_qr ? $result_master_qr->fetch_assoc()['total'] : 0;
    
    $total_qr = $total_stock_qr + $total_master_qr;
    
    // Get some sample data
    $sample_stock_query = "SELECT csi.code_color, mc.code_color as master_code, mc.color_name, b.nama_buyer 
                           FROM color_stock_items csi 
                           LEFT JOIN master_color mc ON csi.master_color_id = mc.id 
                           LEFT JOIN buyers b ON mc.buyer_id = b.id 
                           WHERE csi.qr_code IS NOT NULL 
                           ORDER BY csi.created_at DESC LIMIT 5";
    $result_sample_stock = $conn->query($sample_stock_query);
    
    $sample_master_query = "SELECT mc.code_color, mc.color_name, b.nama_buyer 
                            FROM master_color mc 
                            LEFT JOIN buyers b ON mc.buyer_id = b.id 
                            WHERE mc.qr_code IS NOT NULL 
                            ORDER BY mc.created_at DESC LIMIT 5";
    $result_sample_master = $conn->query($sample_master_query);
    
} catch (Exception $e) {
    $total_stock_qr = 0;
    $total_master_qr = 0;
    $total_qr = 0;
}
?>

<!-- Delete All QR Codes Content -->
<div class="container mx-auto">
    <!-- Page Header -->
    <div class="bg-white rounded-lg shadow-md p-6 mb-6">
        <div class="flex items-center justify-between">
            <div>
                <h2 class="text-2xl font-bold text-gray-800">
                    <i class="fas fa-trash-alt text-red-600 mr-2"></i>
                    Delete All QR Codes
                </h2>
                <p class="text-gray-600 mt-1">Hapus semua QR code dari sistem (Stock Items & Master Color)</p>
            </div>
            <div class="text-right">
                <span class="bg-red-100 text-red-800 px-4 py-2 rounded-full text-sm font-semibold">
                    <i class="fas fa-exclamation-triangle mr-1"></i>
                    Danger Zone
                </span>
            </div>
        </div>
    </div>

    <!-- Warning Notice -->
    <div class="bg-yellow-50 border-l-4 border-yellow-400 p-6 mb-6">
        <div class="flex">
            <div class="flex-shrink-0">
                <i class="fas fa-exclamation-triangle text-yellow-400 text-xl"></i>
            </div>
            <div class="ml-3">
                <h3 class="text-lg font-medium text-yellow-800">Peringatan Penting!</h3>
                <div class="mt-2 text-sm text-yellow-700">
                    <ul class="list-disc list-inside space-y-1">
                        <li>Aksi ini akan menghapus SEMUA QR code yang telah di-generate</li>
                        <li>QR code yang sudah dicetak akan menjadi tidak valid</li>
                        <li>Data master color dan stock items tidak akan terhapus, hanya QR code-nya saja</li>
                        <li>Proses ini tidak dapat dibatalkan (irreversible)</li>
                        <li>Sistem akan mencatat aktivitas ini dalam audit log</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>

    <!-- Current Status -->
    <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
        <div class="bg-white rounded-lg shadow-md p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <div class="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
                        <i class="fas fa-boxes text-blue-600 text-xl"></i>
                    </div>
                </div>
                <div class="ml-4">
                    <h3 class="text-lg font-semibold text-gray-800">Stock Items QR</h3>
                    <p class="text-2xl font-bold text-blue-600"><?php echo number_format($total_stock_qr); ?></p>
                    <p class="text-sm text-gray-500">QR codes tersedia</p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-lg shadow-md p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <div class="w-12 h-12 bg-green-100 rounded-lg flex items-center justify-center">
                        <i class="fas fa-palette text-green-600 text-xl"></i>
                    </div>
                </div>
                <div class="ml-4">
                    <h3 class="text-lg font-semibold text-gray-800">Master Color QR</h3>
                    <p class="text-2xl font-bold text-green-600"><?php echo number_format($total_master_qr); ?></p>
                    <p class="text-sm text-gray-500">QR codes tersedia</p>
                </div>
            </div>
        </div>

        <div class="bg-white rounded-lg shadow-md p-6">
            <div class="flex items-center">
                <div class="flex-shrink-0">
                    <div class="w-12 h-12 bg-red-100 rounded-lg flex items-center justify-center">
                        <i class="fas fa-qrcode text-red-600 text-xl"></i>
                    </div>
                </div>
                <div class="ml-4">
                    <h3 class="text-lg font-semibold text-gray-800">Total QR Codes</h3>
                    <p class="text-2xl font-bold text-red-600"><?php echo number_format($total_qr); ?></p>
                    <p class="text-sm text-gray-500">Yang akan dihapus</p>
                </div>
            </div>
        </div>
    </div>

    <?php if ($total_qr > 0): ?>
    <!-- Sample Data Preview -->
    <div class="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-6">
        <!-- Sample Stock Items -->
        <div class="bg-white rounded-lg shadow-md p-6">
            <h3 class="text-lg font-bold text-gray-800 mb-4 flex items-center">
                <i class="fas fa-boxes text-blue-600 mr-2"></i>
                Sample Stock Items QR
            </h3>
            
            <?php if ($result_sample_stock && $result_sample_stock->num_rows > 0): ?>
                <div class="space-y-3">
                    <?php while($stock = $result_sample_stock->fetch_assoc()): ?>
                        <div class="flex items-center justify-between p-3 bg-blue-50 rounded-lg">
                            <div>
                                <p class="font-semibold text-gray-800"><?php echo htmlspecialchars($stock['code_color']); ?></p>
                                <p class="text-sm text-gray-600">
                                    Master: <?php echo htmlspecialchars($stock['master_code']); ?> - 
                                    <?php echo htmlspecialchars($stock['color_name']); ?>
                                </p>
                                <p class="text-xs text-gray-500"><?php echo htmlspecialchars($stock['nama_buyer']); ?></p>
                            </div>
                            <i class="fas fa-qrcode text-blue-600"></i>
                        </div>
                    <?php endwhile; ?>
                </div>
            <?php else: ?>
                <p class="text-gray-500 text-center py-4">Tidak ada QR code stock items</p>
            <?php endif; ?>
        </div>

        <!-- Sample Master Colors -->
        <div class="bg-white rounded-lg shadow-md p-6">
            <h3 class="text-lg font-bold text-gray-800 mb-4 flex items-center">
                <i class="fas fa-palette text-green-600 mr-2"></i>
                Sample Master Color QR
            </h3>
            
            <?php if ($result_sample_master && $result_sample_master->num_rows > 0): ?>
                <div class="space-y-3">
                    <?php while($master = $result_sample_master->fetch_assoc()): ?>
                        <div class="flex items-center justify-between p-3 bg-green-50 rounded-lg">
                            <div>
                                <p class="font-semibold text-gray-800"><?php echo htmlspecialchars($master['code_color']); ?></p>
                                <p class="text-sm text-gray-600"><?php echo htmlspecialchars($master['color_name']); ?></p>
                                <p class="text-xs text-gray-500"><?php echo htmlspecialchars($master['nama_buyer']); ?></p>
                            </div>
                            <i class="fas fa-qrcode text-green-600"></i>
                        </div>
                    <?php endwhile; ?>
                </div>
            <?php else: ?>
                <p class="text-gray-500 text-center py-4">Tidak ada QR code master color</p>
            <?php endif; ?>
        </div>
    </div>

    <!-- Delete Action -->
    <div class="bg-white rounded-lg shadow-md p-6">
        <div class="text-center">
            <h3 class="text-xl font-bold text-gray-800 mb-4">Hapus Semua QR Codes</h3>
            <p class="text-gray-600 mb-6">
                Klik tombol di bawah untuk menghapus <strong><?php echo number_format($total_qr); ?> QR codes</strong> dari sistem.
                <br>Pastikan Anda sudah memahami konsekuensi dari aksi ini.
            </p>
            
            <div class="flex justify-center gap-4">
                <button onclick="deleteAllQR()" 
                        class="bg-red-600 hover:bg-red-700 text-white px-8 py-3 rounded-lg font-bold transition-colors">
                    <i class="fas fa-trash-alt mr-2"></i>
                    Hapus Semua QR Codes (<?php echo $total_qr; ?>)
                </button>
                
                <a href="data_qr_stock.php" 
                   class="bg-gray-500 hover:bg-gray-600 text-white px-8 py-3 rounded-lg font-bold transition-colors">
                    <i class="fas fa-arrow-left mr-2"></i>
                    Kembali ke QR Stock
                </a>
            </div>
        </div>
    </div>

    <?php else: ?>
    <!-- No QR Codes -->
    <div class="bg-white rounded-lg shadow-md p-12 text-center">
        <i class="fas fa-check-circle text-green-500 text-6xl mb-4"></i>
        <h3 class="text-xl font-bold text-gray-800 mb-2">Tidak Ada QR Code</h3>
        <p class="text-gray-600 mb-6">
            Sistem sudah bersih! Tidak ada QR code yang perlu dihapus.
        </p>
        
        <div class="flex justify-center gap-4">
            <a href="data_qr_stock.php" 
               class="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 rounded-lg transition-colors">
                <i class="fas fa-qrcode mr-2"></i>
                Generate QR Stock
            </a>
            
            <a href="data_qr_master.php" 
               class="bg-green-600 hover:bg-green-700 text-white px-6 py-2 rounded-lg transition-colors">
                <i class="fas fa-palette mr-2"></i>
                Generate QR Master
            </a>
        </div>
    </div>
    <?php endif; ?>
</div>

<!-- Loading Modal -->
<div id="loadingModal" class="fixed inset-0 bg-black bg-opacity-50 hidden z-50 flex items-center justify-center">
    <div class="bg-white rounded-lg p-8 max-w-md w-full mx-4">
        <div class="text-center">
            <div class="animate-spin rounded-full h-16 w-16 border-b-2 border-red-600 mx-auto mb-4"></div>
            <h3 class="text-lg font-bold text-gray-800 mb-2">Menghapus QR Codes...</h3>
            <p class="text-gray-600">Mohon tunggu, proses sedang berlangsung.</p>
        </div>
    </div>
</div>

<script>
function deleteAllQR() {
    const totalQR = <?php echo $total_qr; ?>;
    
    if (totalQR === 0) {
        alert('Tidak ada QR code yang perlu dihapus!');
        return;
    }
    
    const confirmation = confirm(
        `⚠️ PERINGATAN SERIUS! ⚠️\n\n` +
        `Anda akan menghapus ${totalQR.toLocaleString()} QR codes dari sistem:\n` +
        `• Stock Items QR: <?php echo $total_stock_qr; ?>\n` +
        `• Master Color QR: <?php echo $total_master_qr; ?>\n\n` +
        `Aksi ini TIDAK DAPAT DIBATALKAN!\n\n` +
        `Apakah Anda yakin ingin melanjutkan?`
    );
    
    if (!confirmation) return;
    
    const secondConfirmation = confirm(
        `🔥 KONFIRMASI TERAKHIR! 🔥\n\n` +
        `Semua QR code yang sudah dicetak akan menjadi TIDAK VALID!\n\n` +
        `Ketik "DELETE" untuk melanjutkan atau Cancel untuk membatalkan.`
    );
    
    if (!secondConfirmation) return;
    
    // Show loading modal
    document.getElementById('loadingModal').classList.remove('hidden');
    
    // Send delete request
    const formData = new FormData();
    formData.append('action', 'delete_all_qr');
    
    fetch('delete_all_qr.php', {
        method: 'POST',
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        // Hide loading modal
        document.getElementById('loadingModal').classList.add('hidden');
        
        if (data.success) {
            alert(
                `✅ BERHASIL!\n\n` +
                `Semua QR code telah dihapus:\n` +
                `• Stock Items: ${data.summary.stock_items_cleared.toLocaleString()}\n` +
                `• Master Colors: ${data.summary.master_colors_cleared.toLocaleString()}\n` +
                `• Total: ${data.summary.total_cleared.toLocaleString()}\n\n` +
                `Halaman akan di-refresh untuk menampilkan status terbaru.`
            );
            
            // Refresh halaman
            location.reload();
        } else {
            alert('❌ Error: ' + data.message);
        }
    })
    .catch(error => {
        // Hide loading modal
        document.getElementById('loadingModal').classList.add('hidden');
        alert('❌ Network error: ' + error);
    });
}

// Auto refresh every 30 seconds to show updated counts
setInterval(function() {
    location.reload();
}, 30000);
</script>

<?php
// Include footer
include '../template_it/footer.php';
?>