<?php

namespace App\Http\Controllers;

use App\Models\{ItemRequest, ItemRequestLine, ApprovalFlowTemplate};
use App\Models\{Company, User, ApprovalAction};
use App\Services\ApprovalEngine;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;

class ApprovalWebController extends Controller
{
    protected function resolveCurrentUser(): User
    {
        $sess = session('user') ?? [];
        $email = $sess['email'] ?? null;
        $name  = $sess['name'] ?? ($sess['displayName'] ?? 'MS User');
        abort_unless($email, 401, 'Not authenticated');
        return User::firstOrCreate(['email'=>$email], ['name'=>$name]);
    }

    protected function activeCompanyId(): ?int
    {
        $key = session('current_company_name') ?? data_get(session('user'), 'role');

        $map = [
            'Regent' => 'CI',   
            'HIN'    => 'TBI',  
        ];

        $code = $map[$key] ?? null;

        if (!$code) {
            return null; // SUPER with no selection => show everything
        }

        return Company::where('code', $code)->value('id');
    }

    // public function create()
    // {
    //     $flows = ApprovalFlowTemplate::where('is_active',1)->orderBy('name')->get(['id','name']);
    //     return view('approvals.new', compact('flows'));
    // }

    // public function store(Request $r)
    // {
    //     $me = $this->resolveCurrentUser();

    //     $data = $r->validate([
    //         'company_id'        => 'required|integer|exists:companies,id',
    //         'flow_template_id'  => 'required|exists:approval_flow_templates,id',
    //         'posting_date'      => 'nullable|date',
    //         'remarks'           => 'nullable|string',
    //         'vendor_name'       => 'nullable|string|max:255',
    //         'lines'                           => 'required|array|min:1',
    //         'lines.*.article_name'            => 'required|string',
    //         'lines.*.type'                    => 'required|in:Inventory,Service,NonInventory',
    //         'lines.*.base_unit_code'          => 'required|string',
    //         'lines.*.gl_account_no'           => 'nullable|string',
    //         'lines.*.inventory_posting_group_code' => 'nullable|string',
    //         'lines.*.gen_prod_posting_group_code'  => 'nullable|string',
    //         'lines.*.l1'                      => 'nullable|string',
    //         'lines.*.l2'                      => 'nullable|string',
    //         'lines.*.article_no'              => 'nullable|string',
    //         'lines.*.line_remarks'            => 'nullable|string',
    //         'lines.*.vendor_quotes'           => 'nullable|array',
    //         'lines.*.vendor_quotes.*.name'    => 'nullable|string|max:120',
    //         'lines.*.vendor_quotes.*.price'   => 'nullable|numeric|min:0',
    //         'lines.*.vendor_quotes.*.currency'=> 'nullable|string|max:10',
    //         'lines.*.vendor_selected'         => 'nullable|integer',
    //     ]);

    //     foreach ($data['lines'] as $i => $ln) {
    //         if ($ln['type'] === 'Inventory' && !empty($ln['gl_account_no'])) {
    //             return back()->withErrors(["lines.$i.gl_account_no" => 'Inventory line cannot have GL account'])->withInput();
    //         }
    //         if (in_array($ln['type'], ['Service','NonInventory'], true) && empty($ln['gl_account_no'])) {
    //             return back()->withErrors(["lines.$i.gl_account_no" => 'Service/NonInventory requires GL account'])->withInput();
    //         }
    //     }

    //     $req = null;
    //     DB::transaction(function () use ($me, $data, &$req) {
    //         $req = ItemRequest::create([
    //             'company_id'       => (int) $data['company_id'],
    //             'requester_id'     => $me->id,
    //             'flow_template_id' => $data['flow_template_id'],
    //             'posting_date'     => $data['posting_date'] ?? null,
    //             'remarks'          => $data['remarks'] ?? null,
    //             'vendor_name'      => $data['vendor_name'] ?? null,
    //             'status'           => 'Draft',
    //             'current_step_no'  => 0,
    //         ]);

    //         $payloads = [];
    //         foreach ($data['lines'] as $ln) {
    //             $quotesRaw   = $ln['vendor_quotes'] ?? [];
    //             $selectedIdx = $ln['vendor_selected'] ?? null;

    //             $quotes = [];
    //             foreach ($quotesRaw as $idx => $q) {
    //                 if (blank($q['name'] ?? null) && blank($q['price'] ?? null)) continue;
    //                 $quotes[] = [
    //                     'name'     => trim((string)($q['name'] ?? '')),
    //                     'price'    => is_numeric($q['price'] ?? null) ? (float)$q['price'] : null,
    //                     'currency' => $q['currency'] ?? 'IDR',
    //                     'selected' => (string)$idx === (string)$selectedIdx,
    //                 ];
    //             }

    //             $clean = Arr::only($ln, [
    //                 'article_name','type','base_unit_code','gl_account_no',
    //                 'inventory_posting_group_code','gen_prod_posting_group_code',
    //                 'l1','l2','article_no','line_remarks',
    //             ]);
    //             $clean['vendor_quotes']   = $quotes ?: null;
    //             $clean['item_request_id'] = $req->id;

    //             $payloads[] = $clean;
    //         }

    //         $req->lines()->createMany($payloads);
    //     });

    //     return redirect()->route('approvals.show', $req->id)->with('success', 'Draft created');
    // }


    public function show($id)
    {
        $req = \App\Models\ItemRequest::with([
            'requester',
            'lines',
            'flow.steps',   
            'instance.actions.actor', 
        ])->findOrFail($id);

        return view('approvals.show', compact('req'));
    }
    
    

    public function all(\Illuminate\Http\Request $request)
    {
        $q       = trim((string) $request->query('q', ''));
        $cid     = $this->activeCompanyId();
        $status  = $request->query('status');
        $from    = $request->query('from'); // YYYY-MM-DD
        $to      = $request->query('to');   // YYYY-MM-DD
    
        $builder = \App\Models\ItemRequest::query()
            ->with([
                'requester:id,name,email',
                'flow:id,name,version',
                'instance:id,item_request_id,status,started_at',
            ])
            ->when($cid, fn($q) => $q->where('company_id', $cid))
            ->withCount('lines')
            ->orderByDesc('id');
    
        if ($q !== '') {
            $builder->where(function ($w) use ($q) {
                $w->where('number', 'like', "%{$q}%")
                  ->orWhere('vendor_name', 'like', "%{$q}%")
                  ->orWhereHas('requester', function ($qr) use ($q) {
                      $qr->where('name', 'like', "%{$q}%")
                         ->orWhere('email', 'like', "%{$q}%");
                  });
            });
        }
    
        if ($status) {
            $builder->where('status', $status);
        }
    
        // Posting date range
        if ($from && $to) {
            $builder->whereBetween('posting_date', [$from, $to]);
        } elseif ($from) {
            $builder->whereDate('posting_date', '>=', $from);
        } elseif ($to) {
            $builder->whereDate('posting_date', '<=', $to);
        }
    
        $items = $builder->paginate(20)->withQueryString();
    
        return view('approvals.all', compact('items', 'q', 'status'));
    }

    public function submit($id, ApprovalEngine $eng)
    {
        $me  = $this->resolveCurrentUser();
        $req = ItemRequest::with('flow.steps','instance')->findOrFail($id);
    
        // Only these may be (re)submitted
        abort_unless(in_array($req->status, ['Draft','Rejected','Cancelled'], true), 422,
            'Only Draft/Rejected/Cancelled can be submitted.');
    
        // Owner or admin only
        $isOwner = (int)$req->requester_id === (int)$me->id;
        $isAdmin = method_exists($me, 'isAdmin')
            ? (bool)$me->isAdmin()
            : (optional($me->role)->code === 'ADMIN');
        abort_unless($isOwner || $isAdmin, 403, 'You do not have permission to submit this request.');
    
        // Must have at least one line
        abort_if(!$req->lines()->exists(), 422, 'Add at least one line before submitting.');
    
        // Don’t allow submit while an approval instance is in progress
        abort_if(optional($req->instance)->status === 'InProgress', 422, 'An approval is already in progress.');
    
        // If it’s already fully approved, block resubmit explicitly (defensive)
        abort_if($req->status === 'Approved', 422, 'This request is already approved.');
    
        // Submit via engine
        $eng->submit($req, $me);
    
        // Notify current approvers (engine should have advanced to step 1)
        \App\Support\ApprovalNotifier::notifyCurrentApprovers($req->fresh(), 'requested', $me->name ?? $me->email);
    
        return redirect()->route('approvals.show', $id)
            ->with('success', 'Submitted for approval');
    }

    public function approve($id, Request $r, ApprovalEngine $eng)
    {
        $me = $this->resolveCurrentUser();
        $req = ItemRequest::with('flow.steps','instance')->findOrFail($id);
        $result = $eng->act($req, $me->id, 'Approve', $r->input('comment'));

        $flowFinished = false;
        if (is_array($result) && array_key_exists('finished', $result)) {
            $flowFinished = (bool) $result['finished'];
        } elseif (is_bool($result)) {
            $flowFinished = $result;
        } else {
            $flowFinished =
                $req->status === 'Approved' ||
                optional($req->instance)->status === 'Approved' ||
                !is_null(optional($req->instance)->completed_at);
        }

        if ($flowFinished) {
        $requester = $req->requester;
            if ($requester) {
                \App\Support\ApprovalNotifier::notifyUser($requester, $req, 'approved', 'Your request has been fully approved', $me->name ?? $me->email);
            }
        } else {
            \App\Support\ApprovalNotifier::notifyCurrentApprovers($req, 'requested', $me->name ?? $me->email);
        }
        
        return back()->with('success','Approved.');
    }
    
    public function reject($id, Request $r, ApprovalEngine $eng)
    {
        $me = $this->resolveCurrentUser();
        $req = ItemRequest::with('flow.steps','instance')->findOrFail($id);
        $eng->act($req, $me->id, 'Reject', $r->input('comment'));
        $requester = $req->requester;
        if ($requester) {
             \App\Support\ApprovalNotifier::notifyUser($requester, $req, 'rejected', 'Your request was rejected', $me->name ?? $me->email);
        }
        return back()->with('success','Rejected.');
    }
    

    public function inbox(Request $request)
    {
        $me = $this->resolveCurrentUser();
        $cid = $this->activeCompanyId();
        $base = ItemRequest::query()
            ->with(['requester','flow','instance','lines'])
            ->when($cid, fn($q) => $q->where('company_id', $cid))
            ->where('status', 'InReview')
            ->whereExists(function ($q) {
                $q->from('approval_instances as ai')
                ->whereColumn('ai.item_request_id','item_requests.id')
                ->where('ai.status','InProgress');
            });

        if (!$me->isAdmin()) {
            $base->whereExists(function ($q) use ($me) {
                $q->from('approval_step_templates as ast')
                  ->join('approval_step_approvers as asa', 'asa.step_template_id', '=', 'ast.id')
                  ->whereColumn('ast.flow_template_id','item_requests.flow_template_id')
                  ->whereColumn('ast.step_no','item_requests.current_step_no')
                  ->where(function ($qq) use ($me) {
                      $qq->where('asa.approver_user_id', $me->id)
                         ->orWhere('asa.approver_role_id', $me->role_id);
                  });
            });
        }

        $items = $base->orderByDesc('id')->paginate(15);

        $stepsCount = ApprovalFlowTemplate::withCount('steps')
            ->whereIn('id', $items->pluck('flow_template_id'))->get()
            ->pluck('steps_count','id');

        return view('approvals.inbox', compact('items','stepsCount'));
    }

    public function myForms(\Illuminate\Http\Request $request)
    {
        $me  = $this->resolveCurrentUser();
        $cid = $this->activeCompanyId();
    
        $items = \App\Models\ItemRequest::query()
            ->where('requester_id', $me->id)
            ->when($cid, fn($q) => $q->where('company_id', $cid))
            ->withCount('lines')
            ->latest()
            ->paginate(12)
            ->withQueryString();
    
        $userId = (string) $me->id;
        $roleId = (string) $me->role_id;
    
        $pendingForMe = \App\Models\ItemRequest::query()
            ->with(['requester','flow','instance'])
            ->when($cid, fn($q) => $q->where('company_id', $cid))
            ->where('status', 'InReview')
            ->whereExists(function ($q) {
                $q->from('approval_instances as ai')
                  ->whereColumn('ai.item_request_id','item_requests.id')
                  ->where('ai.status','InProgress');
            })
            ->whereExists(function ($q) use ($userId, $roleId) {
                $q->from('approval_step_templates as ast')
                  ->join('approval_step_approvers as asa', 'asa.step_template_id', '=', 'ast.id')
                  ->whereColumn('ast.flow_template_id','item_requests.flow_template_id')
                  ->whereColumn('ast.step_no','item_requests.current_step_no')
                  ->where(function ($qq) use ($userId, $roleId) {
                      $qq->where('asa.approver_user_id', $userId)
                         ->orWhere('asa.approver_role_id', $roleId);
                  });
            })
            ->orderByDesc('id')
            ->paginate(10)
            ->withQueryString();
    
        // collect debugging info
        $debugData = [
            'me' => [
                'id'        => $me->id,
                'role_id'   => $me->role_id,
                'company_id'=> $cid,
            ],
            'all_item_requests' => \App\Models\ItemRequest::query()
                ->select('id','requester_id','flow_template_id','current_step_no','company_id','status')
                ->latest()->limit(10)->get()->toArray(),
            'steps' => \DB::table('approval_step_templates')
                ->select('id','flow_template_id','step_no','approver_user_id','approver_role_id')
                ->whereIn('flow_template_id', [1,3])
                ->whereIn('step_no', [1,2])
                ->get()
                ->toArray(),
        ];
    
    
        $stepsCount = \App\Models\ApprovalFlowTemplate::withCount('steps')
            ->whereIn('id', $pendingForMe->pluck('flow_template_id'))
            ->get()
            ->pluck('steps_count','id');
    
        return view('approvals.my_forms', compact('items','pendingForMe','stepsCount'));
    }


        public function edit($id, ApprovalEngine $eng)
        {
            $me  = $this->resolveCurrentUser();
            $req = ItemRequest::with('lines')->findOrFail($id);
    
            abort_unless(in_array($req->status, ['Draft','Rejected','Approved', 'Cancelled', 'InReview']), 422, 'This request cannot be edited.');
    
            $isOwner = ((int) $req->requester_id === (int) $me->id);
            $isAdmin = method_exists($me, 'isAdmin')
                ? (bool) $me->isAdmin()
                : (optional($me->role)->code === 'ADMIN');
                
    
            $canEdit = false;
    
            if (in_array($req->status, ['Draft','Rejected', 'InReview', 'Cancelled'], true)) {
                $canEdit = $isOwner || $isAdmin;
            } else if ($req->status === 'Approved') {
                $canEdit = $isAdmin || $eng->isLastApprover($req, $me);
            }
            
            abort_unless($canEdit, 403, 'You do not have permission to edit this request.');
    
            $flows = ApprovalFlowTemplate::where('object_type','NewItem')
                ->where('is_active',1)->orderByDesc('version')->get();
    
            $companies = $me->isAdmin()
                ? Company::orderBy('name')->get()
                : Company::whereIn('id', $me->allowedCompanyIds())->orderBy('name')->get();
    
            return view('approvals.edit', [
                'req' => $req,
                'flows' => $flows,
                'companies' => $companies,
            ]);
        }
    
        public function update($id, Request $r, ApprovalEngine $eng)
        {
            $me  = $this->resolveCurrentUser();
            $req = ItemRequest::with('lines')->findOrFail($id);
    
            abort_unless(in_array($req->status, ['Draft','Rejected','Approved','Cancelled', 'InReview']), 422, 'This request cannot be edited.');
    
            $isOwner = ((int) $req->requester_id === (int) $me->id);
            $isAdmin = method_exists($me, 'isAdmin')
                ? (bool) $me->isAdmin()
                : (optional($me->role)->code === 'ADMIN');
                
    
            $canEdit = false;
    
            if (in_array($req->status, ['Draft','Rejected', 'InReview', 'Cancelled'], true)) {
                $canEdit = $isOwner || $isAdmin;
            } else if ($req->status === 'Approved') {
                $canEdit = $isAdmin || $eng->isLastApprover($req, $me);
            }
            abort_unless($canEdit, 403, 'You do not have permission to edit this request.');
    
    
            $data = $r->validate([
                'company_id'        => 'required|integer|exists:companies,id',
                'flow_template_id'  => 'required|exists:approval_flow_templates,id',
                'posting_date'      => 'nullable|date',
                'remarks'           => 'nullable|string',
                'vendor_name'       => 'nullable|string|max:255',
                'lines'                           => 'required|array|min:1',
                'lines.*.article_name'            => 'required|string',
                'lines.*.type'                    => 'required|in:Inventory,Service,NonInventory',
                'lines.*.base_unit_code'          => 'required|string',
                'lines.*.gl_account_no'           => 'nullable|string',
                'lines.*.inventory_posting_group_code' => 'nullable|string',
                'lines.*.gen_prod_posting_group_code'  => 'nullable|string',
                'lines.*.l1'                      => 'nullable|string',
                'lines.*.l2'                      => 'nullable|string',
                'lines.*.article_no'              => 'nullable|string',
                'lines.*.line_remarks'            => 'nullable|string',
                'lines.*.vendor_quotes'           => 'nullable|array',
                'lines.*.vendor_quotes.*.name'    => 'nullable|string|max:120',
                'lines.*.vendor_quotes.*.price'   => 'nullable|numeric|min:0',
                'lines.*.vendor_quotes.*.currency'=> 'nullable|string|max:10',
                'lines.*.vendor_selected'         => 'nullable|integer',
                'attachments'       => ['array'],
                'attachments.*'     => ['file','max:5120'],
            ]);
    
            foreach ($data['lines'] as $i => $ln) {
                if ($ln['type'] === 'Inventory' && !empty($ln['gl_account_no'])) {
                    return back()->withErrors(["lines.$i.gl_account_no" => 'Inventory line cannot have GL account'])->withInput();
                }
                if (in_array($ln['type'], ['Service','NonInventory'], true) && empty($ln['gl_account_no'])) {
                    return back()->withErrors(["lines.$i.gl_account_no" => 'Service/NonInventory requires GL account'])->withInput();
                }
            }
    
            DB::transaction(function () use ($req, $data, $r, $me) {
                $req->update([
                    'company_id'       => (int) $data['company_id'],
                    'flow_template_id' => $data['flow_template_id'],
                    'posting_date'     => $data['posting_date'] ?? null,
                    'remarks'          => $data['remarks'] ?? null,
                    'vendor_name'      => $data['vendor_name'] ?? null,
                ]);
    
                $req->lines()->delete();
    
                $payloads = [];
                foreach ($data['lines'] as $ln) {
                    $quotesRaw   = $ln['vendor_quotes'] ?? [];
                    $selectedIdx = $ln['vendor_selected'] ?? null;
    
                    $quotes = [];
                    foreach ($quotesRaw as $idx => $q) {
                        $hasName  = isset($q['name'])  && trim((string)$q['name']) !== '';
                        $hasPrice = isset($q['price']) && $q['price'] !== null && $q['price'] !== '';
                        if (!$hasName && !$hasPrice) continue;
    
                        $quotes[] = [
                            'name'     => trim((string)($q['name'] ?? '')),
                            'price'    => is_numeric($q['price'] ?? null) ? (float)$q['price'] : null,
                            'currency' => $q['currency'] ?? 'IDR',
                            'selected' => (string)$idx === (string)$selectedIdx,
                        ];
                    }
    
                    $clean = Arr::only($ln, [
                        'article_name','type','base_unit_code','gl_account_no',
                        'inventory_posting_group_code','gen_prod_posting_group_code',
                        'l1','l2','article_no','line_remarks',
                    ]);
                    $clean['vendor_quotes']   = $quotes ?: null;
                    $clean['item_request_id'] = $req->id;
    
                    $payloads[] = $clean;
                }
                $req->lines()->createMany($payloads);
                $deleteIds = (array) $r->input('delete_attachments', []);
                if ($deleteIds) {
                    $req->attachments()
                        ->whereIn('id', $deleteIds)
                        ->get()
                        ->each(function ($att) {
                            \Storage::disk('public')->delete($att->path);
                            $att->delete();
                        });
                }
    
                foreach ($r->file('attachments', []) as $file) {
                        $path = $file->store("requests/{$req->id}", 'public');
                        $req->attachments()->create([
                            'uploader_user_id' => $me->id,
                            'original_name'    => $file->getClientOriginalName(),
                            'path'             => $path,
                            'mime'             => $file->getClientMimeType(),
                            'size'             => $file->getSize(),
                        ]);
                }
    
            });
    
            return redirect()->route('approvals.show', $req->id)->with('success', 'Draft updated.');
        }
    
    public function destroyDraft($id, Request $r)
    {
        $me  = $this->resolveCurrentUser(); // you already use this helper elsewhere
        $req = ItemRequest::with(['lines','attachments','instance.actions'])->findOrFail($id);

        abort_if($req->status !== 'Draft', 422, 'Only draft requests can be deleted.');
        $isOwner = ((int) $req->requester_id === (int) $me->id);
        $isAdmin = method_exists($me, 'isAdmin')
            ? (bool) $me->isAdmin()
            : (optional($me->role)->code === 'ADMIN');
    
        abort_unless($isOwner || $isAdmin, 403, 'You do not have permission to delete this request.');


        DB::transaction(function () use ($req) {
            if ($req->instance) {
                $req->instance->actions()->delete();
                $req->instance()->delete();
            }

            $req->attachments->each(function($att){
                if ($att->path) {
                    \Storage::disk('public')->delete($att->path);
                }
                $att->delete();
            });

            $req->lines()->delete();
            $req->delete();
        });

        return redirect()->route('approvals.mine')->with('success', 'Draft deleted.');
    }

    public function cancel($id, Request $r)
    {
        $me  = $this->resolveCurrentUser();
        $req = ItemRequest::with(['instance'])->findOrFail($id);

        abort_unless($req->status === 'InReview', 422, 'Only in-review requests can be cancelled.');
        $isOwner = ((int) $req->requester_id === (int) $me->id);
        $isAdmin = method_exists($me, 'isAdmin')
            ? (bool) $me->isAdmin()
            : (optional($me->role)->code === 'ADMIN');
    
        abort_unless($isOwner || $isAdmin, 403, 'You do not have permission to edit this request.');


        $ins = $req->instance;
        abort_if(!$ins, 422, 'No approval instance found.');
        abort_unless($ins->status === 'InProgress', 422, 'Instance is not in progress.');

        DB::transaction(function () use ($req, $ins, $me, $r) {
            ApprovalAction::create([
                'approval_instance_id' => $ins->id,
                'step_no'              => $req->current_step_no ?? 0,
                'actor_user_id'        => $me->id,
                'action'               => 'Cancel',            
                'comment'              => $r->input('comment'),
                'acted_at'             => now(),
            ]);

            // close the instance
            $ins->status       = 'Cancelled';
            $ins->ended_at = now();
            $ins->save();

            $req->status = 'Cancelled';
            $req->save();
        });

        return redirect()->route('approvals.show', $req->id)
            ->with('success', 'Request has been cancelled.');
    }

    public function downloadAttachment(\App\Models\ItemRequest $req, \App\Models\ItemRequestAttachment $att)
    {
        if ((int)$att->item_request_id !== (int)$req->id) {
            Log::warning('DL abort: attachment not owned by request', [
                'req_id' => $req->id, 'att_id' => $att->id,
            ]);
            abort(404);
        }

        return \Storage::disk('public')->download($att->path, $att->original_name);
    }
    
    public function previewAttachment(\App\Models\ItemRequest $req, \App\Models\ItemRequestAttachment $att)
    {
        if ((int) $att->item_request_id !== (int) $req->id) {
            \Log::warning('Preview abort: attachment not owned by request', [
                'req_id' => $req->id,
                'att_id' => $att->id,
            ]);
            abort(404);
        }
    
        $path = \Storage::disk('public')->path($att->path);
    
        abort_unless(file_exists($path), 404);
    
        return response()->file($path, [
            'Content-Type'        => $att->mime,
            'Content-Disposition' => 'inline; filename="'.$att->original_name.'"',
        ]);
    }

    public function duplicate($id, \Illuminate\Http\Request $r)
    {
        $me  = $this->resolveCurrentUser();

        $src = ItemRequest::with(['lines','attachments'])->findOrFail($id);

        abort_unless($src->status === 'Rejected', 422, 'Only rejected requests can be duplicated.');
        $isOwner = (int)$src->requester_id === (int)$me->id;
        $isAdmin = method_exists($me, 'isAdmin') ? $me->isAdmin() : (optional($me->role)->code === 'ADMIN');
        
        abort_unless($isOwner || $isAdmin, 403, 'You are not allowed to duplicate this request.');

        $copyAttachments = (bool) $r->boolean('copy_attachments'); 

        $new = DB::transaction(function () use ($src, $me, $copyAttachments) {

            $new = ItemRequest::create([
                'company_id'       => (int) $src->company_id,
                'requester_id'     => $me->id,
                'flow_template_id' => $src->flow_template_id,
                'posting_date'     => $src->posting_date,        
                'remarks'          => $src->remarks,
                'vendor_name'      => $src->vendor_name,
                'status'           => 'Draft',
                'current_step_no'  => 0,
            ]);

            $payloads = [];
            foreach ($src->lines as $ln) {
                $payloads[] = [
                    'article_name'                   => $ln->article_name,
                    'type'                           => $ln->type,
                    'base_unit_code'                 => $ln->base_unit_code,
                    'gl_account_no'                  => $ln->gl_account_no,
                    'inventory_posting_group_code'   => $ln->inventory_posting_group_code,
                    'gen_prod_posting_group_code'    => $ln->gen_prod_posting_group_code,
                    'l1'                             => $ln->l1,
                    'l2'                             => $ln->l2,
                    'article_no'                     => $ln->article_no,
                    'line_remarks'                   => $ln->line_remarks,
                    'vendor_quotes'                  => $ln->vendor_quotes,
                ];
            }
            if (!empty($payloads)) {
                $new->lines()->createMany($payloads);
            }

            // (Optional) copy attachments
            if ($copyAttachments && $src->attachments?->count()) {
                foreach ($src->attachments as $att) {
                    $from = $att->path; // e.g. requests/{oldId}/file.pdf
                    $to   = 'requests/'.$new->id.'/'.basename($att->path);
                    // if stored on 'public' disk:
                    if (\Storage::disk('public')->exists($from)) {
                        \Storage::disk('public')->copy($from, $to);
                        $new->attachments()->create([
                            'uploader_user_id' => $me->id,
                            'original_name'    => $att->original_name,
                            'path'             => $to,
                            'mime'             => $att->mime,
                            'size'             => $att->size,
                        ]);
                    }
                }
            }

            return $new;
        });

        return redirect()->route('approvals.edit', $new->id)
            ->with('success', "Draft #{$new->id} created from rejected request #{$src->id}.");
    }
    public function print($id)
    {
        $req = \App\Models\ItemRequest::with([
            'requester','company','flow.steps.approverUser',
            'instance.actions.actor','lines'
        ])->findOrFail($id);
    
        $vendorHeader = collect($req->lines)
            ->flatMap(fn($l) => collect($l->vendor_quotes ?? [])->pluck('name')->filter())
            ->countBy()->sortDesc()->keys()->take(3)->values()->all();
    
        // If there’s a selected vendor anywhere, make it the first header column
        $firstSelected = collect($req->lines)
            ->flatMap(fn($l) => collect($l->vendor_quotes ?? []))
            ->firstWhere('selected', true);
        if ($firstSelected) {
            $name = $firstSelected['name'] ?? null;
            if ($name) {
                $vendorHeader = collect([$name])
                    ->merge(collect($vendorHeader)->reject(fn($v) => $v === $name))
                    ->take(3)->values()->all();
            }
        }
    
        return view('approvals.print', compact('req','vendorHeader'));
    }
}
