@extends('layouts.app')

@section('title', 'Replace Material – Pastry Division')

@section('content')
    <div class="container py-4 pastel-bg rounded-4 shadow-sm">

        <div class="d-flex justify-content-between align-items-center mb-3 position-relative">
            <div>
                <h3 class="fw-bold text-peach mb-1">Replace Material</h3>
                <small class="text-muted">Manage recipe materials and substitute ingredients if needed.</small>
            </div>

            <button id="openReplaceModal" class="btn btn-outline-peach" disabled>🔄 Replace Selected</button>



        </div>
        <div id="localLoading"
            class="d-none position-absolute top-0 start-0 w-100 h-100 d-flex justify-content-center align-items-center bg-white bg-opacity-75"
            style="display:none;">
            <div class="spinner-border text-peach" style="width:2.5rem;height:2.5rem;"></div>
        </div>

        <!-- Filter Section -->
        <div class="d-flex gap-3 mb-3 flex-wrap">
            <div class="flex-grow-1">
                <input type="text" id="searchInput" class="form-control rounded-3 shadow-sm"
                    placeholder="🔍 Search recipe or material...">
            </div>
            <div>
                <select id="recipeFilter" class="form-select rounded-3 shadow-sm">
                    <option value="">All Recipes</option>
                </select>
            </div>
        </div>

        <!-- Table -->
        <div class="table-responsive rounded-4 shadow-sm">
            <table class="table align-middle pastel-table mb-0">
                <thead class="table-cream">
                    <tr>
                        <th style="width:40px;">#</th>
                        <th>Item No</th>
                        <th>Recipe</th>
                        <th>Business Unit</th>
                        <th>Action</th>
                    </tr>
                </thead>
                <tbody id="recipeBody"></tbody>
            </table>
        </div>

        <!-- Pagination -->
        <div class="d-flex justify-content-center align-items-center mt-3 gap-2">
            <button class="btn btn-outline-peach btn-sm" id="prevPage">⬅️ Prev</button>
            <span id="pageInfo" class="text-muted small"></span>
            <button class="btn btn-outline-peach btn-sm" id="nextPage">Next ➡️</button>
        </div>
    </div>

    <!-- Modal Replace -->
    <div class="modal fade" id="replaceModal" tabindex="-1" aria-hidden="true">
        <div class="modal-dialog modal-lg modal-dialog-scrollable">
            <div class="modal-content rounded-4 shadow">
                <div class="modal-header bg-peach text-white">
                    <h5 class="modal-title fw-bold">Replace Selected Materials</h5>
                    <button class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body pastel-bg" id="replaceBody"></div>
                <div class="modal-footer">
                    <button class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button id="applyReplace" class="btn btn-peach">✅ Apply Changes</button>
                </div>
            </div>
        </div>
    </div>
@endsection

@push('styles')
    <style>
        .text-peach {
            color: #e89b7e;
        }

        .bg-peach {
            background-color: #e89b7e;
        }

        .btn-peach {
            background-color: #e89b7e;
            color: white;
            border: none;
        }

        .btn-outline-peach {
            color: #e89b7e;
            border-color: #e89b7e;
        }

        .btn-outline-peach:hover {
            background-color: #e89b7e;
            color: white;
        }

        .pastel-bg {
            background-color: #fffaf6;
        }

        .table-cream thead {
            background-color: #fff3e6;
        }

        .rounded-4 {
            border-radius: 1rem;
        }

        .fade-in {
            animation: fadeIn .3s ease-in-out;
        }

        @keyframes fadeIn {
            from {
                opacity: 0;
                transform: translateY(5px);
            }

            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
    </style>
@endpush

@push('scripts')
    <script>
        document.addEventListener('DOMContentLoaded', () => {

            const rawData = @json($rawData);

            const recipeBody = document.getElementById('recipeBody');
            const searchInput = document.getElementById('searchInput');
            const recipeFilter = document.getElementById('recipeFilter');
            const openReplaceModal = document.getElementById('openReplaceModal');
            const replaceModal = new bootstrap.Modal(document.getElementById('replaceModal'));
            const replaceBody = document.getElementById('replaceBody');
            const applyReplace = document.getElementById('applyReplace');
            const pageInfo = document.getElementById('pageInfo');
            const prevPage = document.getElementById('prevPage');
            const nextPage = document.getElementById('nextPage');

            let filtered = [];
            let currentPage = 1;
            const perPage = 10;
            let selectedItems = [];
            let availableItems = [];

            // ============================================
            // === ADDED: Penyimpanan Checkbox Global =====
            // ============================================
            let savedRecipeChecked = new Set();
            let savedItemChecked = new Set();
            // savedItemChecked menyimpan format: "BOM_PARENT::ITEM_NO"
            // ============================================


            // === Populate dropdown recipe list ===
            const recipes = [...new Set(rawData.map(b => b.BOM_Parent))];
            recipes.forEach(r => {
                const opt = document.createElement('option');
                opt.value = r;
                opt.textContent = r;
                recipeFilter.appendChild(opt);
            });

            // === Filter + Pagination ===
            function getFilteredData() {
                const keyword = searchInput.value.toLowerCase();
                const recipeSel = recipeFilter.value;

                const matched = rawData.filter(bom => {
                    const matchRecipe = recipeSel ? bom.BOM_Parent === recipeSel : true;
                    const matchKeyword =
                        (bom.Description_BOM ?? '').toLowerCase().includes(keyword) ||
                        (bom.Item_Operation || []).some(it =>
                            (it.BOM_Description ?? '').toLowerCase().includes(keyword)
                        );
                    return matchRecipe && matchKeyword;
                });

                if (keyword) {
                    matched.sort((a, b) => {
                        const aHas = (a.Item_Operation || []).some(it => (it.BOM_Description ?? '')
                            .toLowerCase().includes(keyword));
                        const bHas = (b.Item_Operation || []).some(it => (it.BOM_Description ?? '')
                            .toLowerCase().includes(keyword));
                        return (bHas === aHas) ? 0 : bHas ? 1 : -1;
                    }).reverse();
                }
                return matched;
            }

            function paginateData(data) {
                const start = (currentPage - 1) * perPage;
                return data.slice(start, start + perPage);
            }

            // === Render Table ===
            function renderTable() {

                filtered = getFilteredData();
                const paginated = paginateData(filtered);
                recipeBody.innerHTML = '';

                paginated.forEach((bom, index) => {

                    const recipeChecked = savedRecipeChecked.has(bom.BOM_Parent);

                    const tr = document.createElement('tr');
                    tr.classList.add('fade-in');
                    tr.innerHTML = `
                <td class="fw-bold">
                    <input type="checkbox" class="recipe-check me-2"
                        data-recipe='${JSON.stringify(bom.BOM_Parent)}'
                        ${recipeChecked ? "checked" : ""}>
                </td>
                <td>${bom.BOM_Parent}</td>
                <td>${bom.Description_BOM}</td>
                <td>${bom.Business_Unit}</td>
                <td><button class="btn btn-outline-peach btn-sm toggle-items" data-index="${index}">⬇️ Show Items</button></td>
            `;
                    recipeBody.appendChild(tr);

                    const subTr = document.createElement('tr');
                    subTr.classList.add('d-none');
                    subTr.innerHTML = `
                <td colspan="4">
                    <div class="p-2 bg-white rounded-4 shadow-sm fade-in">
                        ${(bom.Item_Operation || []).map(it => {
                            const key = `${bom.BOM_Parent}::${it.BOM}`;
                            const itemChecked = savedItemChecked.has(key);
                            return `
                                    <div class="d-flex justify-content-between border-bottom py-1 align-items-center">
                                        <div>
                                            <input type="checkbox" class="item-check me-2"
                                                data-item='${JSON.stringify({ bomParent: bom.BOM_Parent, item: it })}'
                                                ${itemChecked ? "checked" : ""}>
                                            <strong>${it.BOM}</strong> — ${it.BOM_Description}
                                        </div>
                                        <div>
                                            <small>${it.Quantity} ${it.Uom_BOM ?? bom.Uom_BOM_Parent}</small> |
                                            ${it.Inventory > 0 ? '🟢 Ready' : '⚫ Empty'}
                                        </div>
                                    </div>
                                `;
                        }).join('')}
                    </div>
                </td>
            `;
                    recipeBody.appendChild(subTr);
                });

                pageInfo.textContent = `Page ${currentPage} of ${Math.ceil(filtered.length / perPage)}`;
            }

            // === Toggle Items ===
            document.addEventListener('click', e => {
                if (e.target.classList.contains('toggle-items')) {
                    const btn = e.target;
                    const tr = btn.closest('tr');
                    const nextTr = tr.nextElementSibling;
                    const visible = !nextTr.classList.contains('d-none');
                    nextTr.classList.toggle('d-none', visible);
                    btn.textContent = visible ? '⬇️ Show Items' : '⬆️ Hide Items';
                }
            });

            // === Checkbox interlink + SAVE STATE ===
            document.addEventListener('change', e => {
                if (e.target.classList.contains('recipe-check')) {
                    const bomParent = JSON.parse(e.target.dataset.recipe);

                    if (e.target.checked) {
                        savedRecipeChecked.add(bomParent);
                    } else {
                        savedRecipeChecked.delete(bomParent);
                    }

                    const subTr = e.target.closest('tr').nextElementSibling;
                    const itemChecks = subTr.querySelectorAll('.item-check');
                    itemChecks.forEach(c => {
                        c.checked = e.target.checked;

                        const itemData = JSON.parse(c.dataset.item);
                        const key = `${itemData.bomParent}::${itemData.item.BOM}`;

                        if (e.target.checked) savedItemChecked.add(key);
                        else savedItemChecked.delete(key);
                    });
                }

                if (e.target.classList.contains('item-check')) {
                    const data = JSON.parse(e.target.dataset.item);
                    const key = `${data.bomParent}::${data.item.BOM}`;

                    if (e.target.checked) {
                        savedItemChecked.add(key);
                    } else {
                        savedItemChecked.delete(key);
                    }
                }

                updateSelected();
            });

            // === Update selected ===
            function updateSelected() {
                selectedItems = [];

                savedRecipeChecked.forEach(recipeNo => {
                    const recipeData = rawData.find(r => r.BOM_Parent === recipeNo);
                    selectedItems.push({
                        type: 'recipe',
                        bomParent: recipeData.BOM_Parent,
                        description: recipeData.Description_BOM,
                        items: recipeData.Item_Operation || []
                    });
                });

                savedItemChecked.forEach(key => {
                    const [bomParent, itemNo] = key.split("::");
                    const parentData = rawData.find(r => r.BOM_Parent === bomParent);
                    const item = (parentData.Item_Operation || []).find(it => it.BOM === itemNo);

                    if (!item) return;

                    const existing = selectedItems.find(sel =>
                        sel.bomParent === bomParent && sel.type === 'recipe'
                    );
                    if (!existing) {
                        let recSel = selectedItems.find(sel => sel.bomParent === bomParent);
                        if (!recSel) {
                            recSel = {
                                type: 'partial',
                                bomParent: bomParent,
                                description: parentData.Description_BOM,
                                items: []
                            };
                            selectedItems.push(recSel);
                        }
                        recSel.items.push(item);
                    }
                });

                openReplaceModal.disabled = selectedItems.length === 0;
            }

            searchInput.addEventListener('input', () => {
                currentPage = 1;
                renderTable();
            });
            recipeFilter.addEventListener('change', () => {
                currentPage = 1;
                renderTable();
            });

            prevPage.addEventListener('click', () => {
                if (currentPage > 1) {
                    currentPage--;
                    renderTable();
                }
            });
            nextPage.addEventListener('click', () => {
                if (currentPage < Math.ceil(filtered.length / perPage)) {
                    currentPage++;
                    renderTable();
                }
            });

            async function fetchAvailableItems() {
                const res = await fetch("{{ route('replace_material.available') }}");
                const data = await res.json();
                availableItems = data;
            }
            
            // === Render Modal ===
            async function renderModal() {
                replaceBody.innerHTML = '';
                await fetchAvailableItems();

                replaceBody.style.height = '80vh';
                replaceBody.style.overflowY = 'auto';

                if (selectedItems.length === 0) {
                    replaceBody.innerHTML =
                        `<div class="d-flex flex-column justify-content-center align-items-center h-100 text-muted fs-5">No item selected</div>`;
                    return;
                }

                selectedItems.forEach(sel => {
                    const section = document.createElement('div');
                    section.className = 'mb-3 p-3 bg-white rounded-4 shadow-sm fade-in';
                    section.innerHTML = `
                <h6 class="fw-bold text-peach mb-2">${sel.bomParent} — ${sel.description}</h6>
                <div class="ps-3" id="matContainer_${sel.bomParent}">
                    ${(sel.items || []).map((it, i) => `
                            <div class="d-flex align-items-center justify-content-between mb-2">
                                <div style="max-width: 400px; overflow: hidden; text-overflow: ellipsis; white-space: nowrap;">
                                    <small class="text-muted">${it.BOM} — ${it.BOM_Description}</small>
                                </div>
                                <div class="d-flex gap-2 position-relative align-items-center">
                                    <select class="form-select form-select-sm action-select" id="action_${sel.bomParent}_${i}" style="width:120px">
                                        <option value="no">No Change</option>
                                        <option value="yes">Change To</option>
                                    </select>
                                    <input class="form-control form-control-sm d-none search-input" placeholder="Search new item" autocomplete="off">
                                    <div class="dropdown-menu shadow-sm border rounded-3 p-1"
                                        style="max-height: 200px; overflow-y: auto; width: 500px; position: absolute; top: 100%; left: auto; inset-inline-end: 0; z-index: 1000; display: none;"></div>
                                    <select class="form-select form-select-sm d-none uom-select" style="width:100px;"></select>
                                    <input type="number" step="0.01" min="0" value="${it.Quantity ?? 0}" class="form-control form-control-sm d-none qty-input" placeholder="Qty" style="width:90px;">
                                </div>
                            </div>
                        `).join('')}
                </div>
            `;
                    replaceBody.appendChild(section);

                    const container = section.querySelectorAll('.search-input');
                    container.forEach(input => {
                        const dropdown = input.nextElementSibling;
                        const uomSelect = dropdown.nextElementSibling;
                        const qtyInput = uomSelect.nextElementSibling;

                        function renderDropdown(list) {
                            dropdown.innerHTML = '';
                            if (list.length === 0) {
                                dropdown.innerHTML =
                                    `<div class="px-2 py-1 text-muted small">No match found</div>`;
                                return;
                            }
                            list.slice(0, 50).forEach(it => {
                                const itemDiv = document.createElement('div');
                                itemDiv.className = 'dropdown-item py-1 px-2 small';
                                itemDiv.style.cursor = 'pointer';
                                itemDiv.textContent =
                                    `${it.item_no} — ${it.description}`;
                                itemDiv.addEventListener('click', () => {
                                    input.value =
                                        `${it.item_no} — ${it.description}`;
                                    input.dataset.selectedItem = JSON.stringify(
                                        it);
                                    dropdown.style.display = 'none';

                                    if (it.item_uom && Array.isArray(it
                                            .item_uom) && it.item_uom.length >
                                        0) {
                                        uomSelect.innerHTML = it.item_uom.map(
                                            u =>
                                            `<option value="${u}">${u}</option>`
                                        ).join('');
                                        uomSelect.classList.remove('d-none');
                                    } else {
                                        uomSelect.classList.add('d-none');
                                        uomSelect.innerHTML = '';
                                    }
                                    qtyInput.classList.remove('d-none');
                                });
                                dropdown.appendChild(itemDiv);
                            });
                        }

                        function fuzzyMatch(str, query) {
                            str = (str ?? '').toLowerCase();
                            query = query.toLowerCase();
                            let i = 0;
                            for (let c of str) {
                                if (c === query[i]) i++;
                                if (i === query.length) return true;
                            }
                            return false;
                        }

                        input.addEventListener('focus', () => {
                            renderDropdown(availableItems);
                            dropdown.style.display = 'block';
                        });
                        input.addEventListener('input', e => {
                            const v = e.target.value.toLowerCase();
                            const filteredList = availableItems.filter(i =>
                                fuzzyMatch(i.item_no ?? '', v) || fuzzyMatch(i
                                    .description ?? '', v)
                            );
                            renderDropdown(filteredList);
                            dropdown.style.display = 'block';
                        });
                        input.addEventListener('blur', () => {
                            setTimeout(() => dropdown.style.display = 'none', 200);
                        });

                        const select = input.previousElementSibling;
                        select.addEventListener('change', () => {
                            const show = select.value === 'yes';
                            input.classList.toggle('d-none', !show);
                            uomSelect.classList.toggle('d-none', !show);
                            qtyInput.classList.toggle('d-none', !show);
                            if (!show) {
                                dropdown.style.display = 'none';
                                uomSelect.innerHTML = '';
                                qtyInput.value = '';
                            }
                        });
                    });
                });
            }

            // === Open Modal ===
            openReplaceModal.addEventListener('click', async () => {
                const loader = document.getElementById('localLoading');
                loader.classList.remove('d-none');
                loader.classList.add('d-flex');
                try {
                    await renderModal();
                    replaceModal.show();
                } catch (err) {
                    console.error('Error rendering modal:', err);
                    Swal.fire('Error', 'Failed to load replacement data', 'error');
                } finally {
                    loader.classList.remove('d-flex');
                    loader.classList.add('d-none');
                }
            });

            // === Apply Replace ===
            applyReplace.addEventListener('click', async () => {
                const payload = [];

                const sections = replaceBody.querySelectorAll('.mb-3.p-3');
                sections.forEach(section => {
                    const [bomParent, desc] = section.querySelector('h6').textContent.split('—')
                        .map(s => s.trim());
                    const materials = [];

                    section.querySelectorAll('.d-flex.align-items-center.mb-2').forEach(row => {
                        const select = row.querySelector('.action-select');
                        const input = row.querySelector('.search-input');
                        const uomSelect = row.querySelector('.uom-select');
                        const qtyInput = row.querySelector('.qty-input');
                        if (select && select.value === 'yes' && input.dataset
                            .selectedItem) {
                            const oldText = row.querySelector('small').textContent;
                            const [oldNo, oldDesc] = oldText.split('—').map(s => s
                                .trim());
                            const newItem = JSON.parse(input.dataset.selectedItem);
                            const chosenUom = uomSelect && !uomSelect.classList
                                .contains('d-none') ? uomSelect.value : null;
                            const chosenQty = qtyInput && !qtyInput.classList.contains(
                                'd-none') && qtyInput.value ? parseFloat(qtyInput
                                .value) : null;
                            materials.push({
                                old_material: {
                                    item_no: oldNo,
                                    description: oldDesc
                                },
                                new_material: {
                                    ...newItem,
                                    chosen_uom: chosenUom,
                                    chosen_quantity: chosenQty
                                }
                            });
                        }
                    });

                    if (materials.length > 0) {
                        payload.push({
                            parent_recipe: {
                                bom_parent: bomParent,
                                description: desc
                            },
                            materials: materials
                        });
                    }
                });

                if (payload.length === 0) {
                    Swal.fire('No changes selected', '', 'info');
                    return;
                }

                console.log('Payload:', payload);

                Swal.fire({
                    title: 'Processing...',
                    text: 'Please wait while updating BOM items.',
                    allowOutsideClick: false,
                    didOpen: () => {
                        Swal.showLoading();
                    }
                });

                try {
                    const res = await fetch("{{ route('bc.bom.patch') }}", {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')
                                .content
                        },
                        body: JSON.stringify(payload)
                    });

                    const data = await res.json();

                    if (res.ok && data.status === 'success') {
                        Swal.fire({
                            icon: 'success',
                            title: 'Success',
                            text: data.message || 'BOM successfully updated.',
                            confirmButtonColor: '#ff9f68'
                        }).then(() => {
                            replaceModal.hide();
                            location.reload();
                        });
                    } else {
                        Swal.fire({
                            icon: 'error',
                            title: 'Failed',
                            text: data.message || 'Something went wrong while updating BOM.',
                            confirmButtonColor: '#ff9f68'
                        });
                    }
                } catch (err) {
                    console.error('POST error:', err);
                    Swal.fire({
                        icon: 'error',
                        title: 'Request Failed',
                        text: err.message || 'Unable to reach server.',
                        confirmButtonColor: '#ff9f68'
                    });
                }
            });

            renderTable();
        });
    </script>
@endpush
