<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use GuzzleHttp\Client;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;
use GuzzleHttp\Promise;
use GuzzleHttp\Promise\Create;
use GuzzleHttp\Promise\Utils;
use Illuminate\Support\Str;
class InventoryBC {
    public function getAccessToken()
    {
        $client = new Client();
        $response = $client->post("https://login.microsoftonline.com/".env('AZURE_TENANT_ID')."/oauth2/v2.0/token", [
            'form_params' => [
                'grant_type' => 'client_credentials',
                'client_id' => env('AZURE_CLIENT_ID'),
                'client_secret' => env('AZURE_CLIENT_SECRET'),
                'scope' => 'https://api.businesscentral.dynamics.com/.default'
            ]
        ]);
        $body = json_decode((string)$response->getBody(), true);

        return $body['access_token'];
    }

    public function getCompanyId($token)
    {
        $client = new Client();
        $response = $client->get("https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') . "/api/v2.0/companies", [
            'headers' => [
                'Authorization' => "Bearer {$token}",
                'Accept'        => 'application/json'
            ]
        ]);
        
        $companies = json_decode((string)$response->getBody(), true);
        $role = session('user')['role'] ?? null; 
        $companyId1 = $companies['value'][0]['id'];
        $companyId2 = $companies['value'][1]['id'];

        return [
            'companyId1' => $companyId1,
            'companyId2' => $companyId2
        ];
    }

    public function __construct()
    {
     $this->token = $this->getAccessToken();
     $companyIds = $this->getCompanyId($this->token);
     $this->companyRegent = $companyIds['companyId1'];
     $this->companyHIN    = $companyIds['companyId2'];
     
     $activeCompany = session('current_company_name') ?? session('user')['role'];
    if ($activeCompany === 'Regent' or $activeCompany === 'SUPER') {
        $this->companyId = $this->companyRegent;
    } elseif ($activeCompany === 'HIN') {
        $this->companyId = $this->companyHIN;
    }
     $this->client = new Client();
    }

    private function readForecastFile(string $filePath, string $delimiter = "\t"): array
    {
        $result = [
            'forecast' => 0,
            'history'  => 0,
        ];

        if (!file_exists($filePath)) {
            return $result;
        }

        if (($handle = fopen($filePath, 'r')) === false) {
            return $result;
        }

        $headers = fgetcsv($handle, 0, $delimiter);
        $colNoRooms = array_search('NO_ROOMS', $headers);
        $colDate    = array_search('CONSIDERED_DATE', $headers);

        if ($colNoRooms === false || $colDate === false) {
            fclose($handle);
            \Log::warning("File $filePath missing required columns", ['headers' => $headers]);
            return $result;
        }

        $today  = new \DateTime('today', new \DateTimeZone('UTC'));
        $start  = (clone $today)->modify('-1 month');
        $end    = (clone $today)->modify('+1 month');

        \Log::debug('Forecast window', [
            'start'  => $start->format(DATE_ATOM),
            'today'  => $today->format(DATE_ATOM),
            'end'    => $end->format(DATE_ATOM),
        ]);

        while (($row = fgetcsv($handle, 0, $delimiter)) !== false) {
            if (!isset($row[$colDate]) || !isset($row[$colNoRooms])) {
                continue;
            }

            $rowDate = \DateTime::createFromFormat('d-M-y', strtoupper($row[$colDate]), new \DateTimeZone('UTC'));
            if (!$rowDate) {
                continue;
            }
            $rooms = (int) $row[$colNoRooms];

            if ($rowDate >= $today && $rowDate <= $end) {
                $result['forecast'] += (int) $row[$colNoRooms];
            \Log::debug('Forecast add', ['date' => $rowDate->format('Y-m-d'), 'rooms' => $rooms]);
            }
            if ($rowDate <= $today && $rowDate >= $start) {
                $result['history'] += (int) $row[$colNoRooms];
            \Log::debug('History add', ['date' => $rowDate->format('Y-m-d'), 'rooms' => $rooms]);

            }
        }

        fclose($handle);
        return $result;
    }


    public function getItemLedgerAndSkuMappingAsync()
    {
        return Cache::remember('sku_mapping_cache', now()->addMinutes(120), function () {
        $stockkeeping = [];   // <-- add before decoding results
        $skuData      = [];   // <-- add before decoding results
        $filters      = [];
        $forecastRoomsRBC = $historyRoomsRBC = $forecastRoomsHIC = $historyRoomsHIC = $forecastRoomsHIN = $historyRoomsHIN = 0;

        $data1 = $this->readForecastFile(storage_path('app/onedrive/history_forecast.txt'), ',');
        $forecastRoomsRBC += $data1['forecast'];
        $historyRoomsRBC  += $data1['history'];

        $data2 = $this->readForecastFile(storage_path('app/onedrive/history_forecast(HIC).txt'), "\t");
        $forecastRoomsHIC += $data2['forecast'];            
        $historyRoomsHIC  += $data2['history'];

        $data3 = $this->readForecastFile(storage_path('app/onedrive/history_forecast(HIND).txt'), "\t");
        $forecastRoomsHIN += $data3['forecast'];
        $historyRoomsHIN  += $data3['history'];

        $forecastRoomsHIC = round($forecastRoomsHIC * 1.3, 2);
        $forecastRoomsRBC = round($forecastRoomsRBC * 1.3, 2);
        $forecastRoomsHIN = round($forecastRoomsHIN * 1.3, 2);

        $historyRoomsHIC  = round($historyRoomsHIC, 2);
        $historyRoomsRBC  = round($historyRoomsRBC, 2);
        $historyRoomsHIN  = round($historyRoomsHIN, 2);

        \Log::info('Forecast/HIC History Rooms', [
            'forecast' => $forecastRoomsHIC,
            'history'  => $historyRoomsHIC
        ]);

        \Log::info('Forecast/HIND History Rooms', [
            'forecast' => $forecastRoomsHIN,
            'history'  => $historyRoomsHIN
        ]);

        \Log::info('Forecast/RBC History Rooms', [
            'forecast' => $forecastRoomsRBC,
            'history'  => $historyRoomsRBC
        ]);

        $twoMonthsAgo = now()->subMonths(2)->format('Y-m-d');
        $matchedKeys = [];
        $baseUrl = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT');
        $company = "/ODataV4/Company('" . $this->companyId . "')";
        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept' => 'application/json',
                'Prefer' => 'odata.maxpagesize=20000'
            ]
        ];
        $activeCompany = session('current_company_name') ?? session('user')['role'];
        if ($activeCompany === 'Regent' or $activeCompany === 'SUPER') {
            $skuFilter = urlencode("(Location eq 'CI.1011') and Max_Inventory gt 0");
            $skuUrl = "$baseUrl$company/SKU_Import?\$filter={$skuFilter}";
            $stockFilter = urlencode("(Location_Code eq 'CI.1011')");
            $filters = [
            "(Location_Code eq 'CI.1011') and (Entry_Type eq 'Negative Adjmt.') and (Quantity lt 0) and (Posting_Date ge $twoMonthsAgo)",
            "(Location_Code eq 'CI.1011') and (Entry_Type eq 'Assembly Consumption	') and (Quantity lt 0) and (Posting_Date ge $twoMonthsAgo)",
            "(Location_Code eq 'CI.1011') and (Entry_Type eq 'Transfer') and (Quantity lt 0) and (Posting_Date ge $twoMonthsAgo)",
        ];
        // $activeCompany = session('current_company_name') ?? session('user')['role'];
        // if ($activeCompany === 'Regent' or $activeCompany === 'SUPER') {
        //     $skuFilter = urlencode("(Location eq 'RBC.2040') and Max_Inventory gt 0");
        //     $skuUrl = "$baseUrl$company/SKU_Import?\$filter={$skuFilter}";
        //     $stockFilter = urlencode("(Location_Code eq 'RBC.2040')");
        //     $filters = [
        //     "(Location_Code eq 'RBC.2040') and (Entry_Type eq 'Negative Adjmt.') and (Quantity lt 0) and (Posting_Date ge $twoMonthsAgo)",
        //     "(Location_Code eq 'RBC.2040') and (Entry_Type eq 'Assembly Consumption	') and (Quantity lt 0) and (Posting_Date ge $twoMonthsAgo)",
        //     "(Location_Code eq 'RBC.2040') and (Entry_Type eq 'Transfer') and (Quantity lt 0) and (Posting_Date ge $twoMonthsAgo)",
        // ];
        // $activeCompany = session('current_company_name') ?? session('user')['role'];
        // if ($activeCompany === 'Regent' or $activeCompany === 'SUPER') {
        //     $skuFilter = urlencode("(Location eq 'CI.1012') and Max_Inventory gt 0");
        //     $skuUrl = "$baseUrl$company/SKU_Import?\$filter={$skuFilter}";
        //     $stockFilter = urlencode("(Location_Code eq 'CI.1012')");
        //     $filters = [
        //     "(Location_Code eq 'CI.1012') and (Entry_Type eq 'Negative Adjmt.') and (Quantity lt 0) and (Posting_Date ge $twoMonthsAgo)",
        //     "(Location_Code eq 'CI.1012') and (Entry_Type eq 'Assembly Consumption	') and (Quantity lt 0) and (Posting_Date ge $twoMonthsAgo)",
        //     "(Location_Code eq 'CI.1012') and (Entry_Type eq 'Transfer') and (Quantity lt 0) and (Posting_Date ge $twoMonthsAgo)",
        // ];
        } elseif ($activeCompany === 'HIN') {
            $skuFilter = urlencode("(Location eq 'HIN.1200' or Location eq 'HIN.1300' or Location eq 'HIN.1000' or Location eq 'HIN.3000')");
            $skuUrl = "$baseUrl$company/SKU_Import?\$filter={$skuFilter}";
            $stockFilter = urlencode("(Location_Code eq 'HIN.1200' or Location_Code eq 'HIN.1300' or Location_Code eq 'HIN.1000' or Location_Code eq 'HIN.3000')");
            $filters = [
            "(Location_Code eq 'HIN.1200') and (Entry_Type eq 'Negative Adjmt.') and (Quantity lt 0) and (Posting_Date ge $twoMonthsAgo)",
            "(Location_Code eq 'HIN.1200') and (Entry_Type eq 'Transfer') and (Quantity lt 0) and (Posting_Date ge $twoMonthsAgo)",
            "(Location_Code eq 'HIN.1300') and (Entry_Type eq 'Negative Adjmt.') and (Quantity lt 0) and (Posting_Date ge $twoMonthsAgo)",
            "(Location_Code eq 'HIN.1300') and (Entry_Type eq 'Transfer') and (Quantity lt 0) and (Posting_Date ge $twoMonthsAgo)",
            "(Location_Code eq 'HIN.1000') and (Entry_Type eq 'Negative Adjmt.') and (Quantity lt 0) and (Posting_Date ge $twoMonthsAgo)",
            "(Location_Code eq 'HIN.1000') and (Entry_Type eq 'Transfer') and (Quantity lt 0) and (Posting_Date ge $twoMonthsAgo)",
            "(Location_Code eq 'HIN.3000') and (Entry_Type eq 'Negative Adjmt.') and (Quantity lt 0) and (Posting_Date ge $twoMonthsAgo)",
            "(Location_Code eq 'HIN.3000') and (Entry_Type eq 'Transfer') and (Quantity lt 0) and (Posting_Date ge $twoMonthsAgo)"
        ];
        }
        $skuUrl = "$baseUrl$company/SKU_Import?\$filter={$skuFilter}";
        $promises = [
            'sku' => $this->client->getAsync($skuUrl, $headers),
        ];
        foreach ($filters as $i => $filter) {
            $encodedFilter = urlencode($filter);
            $ledgerSelect = urlencode("Quantity,Item_No,Item_Description,Base_Unit,Location_Code,Unit_of_Measure_Code,Qty_per_Unit_of_Measure");
            $url = "$baseUrl$company/Item_Ledger_Entries_Excel?\$filter=$encodedFilter&\$select=$ledgerSelect";
            $promises["ledger_$i"] = $this->client->getAsync($url, $headers);
        }

        $promises["stockkeeping"] = $this->client->getAsync(
            "$baseUrl$company/APIStockkeeping?\$filter=$stockFilter",
            $headers
        );

        $results = \GuzzleHttp\Promise\Utils::settle($promises)->wait();

        $ledgerData = [];
        $skuData = [];

        foreach ($results as $key => $result) {
            if (str_starts_with($key, 'ledger_') && $result['state'] === 'fulfilled') {
                $decoded = json_decode((string)$result['value']->getBody(), true);
                $ledgerData = array_merge($ledgerData, $decoded['value'] ?? []);
            }
        }
        if ($results['stockkeeping']['state'] === 'fulfilled') {
            $decoded = json_decode((string)$results['stockkeeping']['value']->getBody(), true);
            $stockkeeping = $decoded['value'] ?? [];
        }

        if ($results['sku']['state'] === 'fulfilled') {
            $decoded = json_decode((string)$results['sku']['value']->getBody(), true);
            $skuData = $decoded['value'] ?? [];
        } else {
            \Log::error('Failed to fetch SKU Mapping: ' . $results['sku']['reason']);
        }

        \Log::info('SKU count: ' . count($skuData));
        \Log::info('Ledger count: ' . count($ledgerData));

        $ledgerQuantities = [];

        foreach ($ledgerData as $entry) {
            $itemNo = $entry['Item_No'];
            $location = $entry['Location_Code'];
            $key = $itemNo . '|' . $location;

            $qty = abs(floatval($entry['Quantity'] ?? 0));

            if (!isset($ledgerQuantities[$key])) {
                $ledgerQuantities[$key] = 0;
            }
            $ledgerQuantities[$key] += $qty;
        }
        $result = [];
        foreach ($skuData as $sku) {
            $itemNo = $sku['Item_No'];
            $stockFor = intval($sku['Stock_For'] ?? 0);
            if ($stockFor === 0) {
                $stockFor = 14;
            }
            $leadTime = intval($sku['Lead_Time'] ?? 0);

            $key = $itemNo . '|' . $sku['Location'];
            $matchedKeys[$key] = true; 
            $consumption = ceil($ledgerQuantities[$key] ?? 0);

            $matchedStock = collect($stockkeeping)
                ->first(fn($s) => $s['Item_No'] === $itemNo && $s['Location_Code'] === $sku['Location']);
            $realMin = $matchedStock['MinInven'] ?? 0;
            $realMax = $matchedStock['MaxInven'] ?? 0;
            if ($realMin > 0  or $realMax > 0  or $consumption > 0) {
                $lastModified = $matchedStock['LastModifiedAt'] ?? null;
                if ($lastModified !== null) {
                    $lastModified = date('d F Y', strtotime($lastModified));
                }

                $suggestMin = ceil(($leadTime + 1) * ($consumption / 60));
                $suggestMax = ceil(($stockFor) * ($consumption / 60));

                if ($activeCompany === 'Regent' || $activeCompany === 'SUPER') {
                    $safeDiv = function (float $a, float $b): float {
                        return $b != 0.0 ? ($a / $b) : 0.0;
                    };

                    $hic_up = $safeDiv($forecastRoomsHIC, $historyRoomsHIC);
                    $rbc_up = $safeDiv($forecastRoomsRBC, $historyRoomsRBC);
                    $avg_up = ($hic_up + $rbc_up) / 2.0;

                    $forecastMinHIC = (int) ceil($suggestMin * $hic_up);
                    $forecastMinRBC = (int) ceil($suggestMin * $rbc_up);
                    $forecastMinAvg = (int) ceil($suggestMin * $avg_up);

                    $forecastMaxHIC = (int) ceil($suggestMax * $hic_up);
                    $forecastMaxRBC = (int) ceil($suggestMax * $rbc_up);
                    $forecastMaxAvg = (int) ceil($suggestMax * $avg_up);
                } elseif ($activeCompany === 'HIN') {
                    $safeDiv = function (float $a, float $b): float {
                        return $b != 0.0 ? ($a / $b) : 0.0;
                    };
                    $hin_up       = $safeDiv($forecastRoomsHIN, $historyRoomsHIN);
                    $forecastMinHIC = $forecastMinRBC = 0;
                    $forecastMaxHIC = $forecastMaxRBC = 0;
                    $forecastMinAvg = (int) ceil($suggestMin * $hin_up);
                    $forecastMaxAvg = (int) ceil($suggestMax * $hin_up);
                } 
                $result[] = [
                    'Item_No' => $itemNo,
                    'Description' => $matchedStock['Description'] ?? $sku['Description'] ,
                    'Location' => $sku['Location'],
                    'Stock_For' => $stockFor,
                    'Lead Time' => $leadTime,
                    'SuggestMinQty' => $suggestMin,
                    'ActMinQty' => $sku['Min_Inventory'],
                    'RealMinQty' => $realMin,
                    'SuggestMaxQty' => $suggestMax,
                    'ActMaxQty' => $sku['Max_Inventory'],
                    'RealMaxQty' => $realMax,

                    'ForecMin_HIC' => $forecastMinHIC,
                    'ForecMin_RBC' => $forecastMinRBC,
                    'ForecMin_Avg' => $forecastMinAvg,

                    'ForecMax_HIC' => $forecastMaxHIC,
                    'ForecMax_RBC' => $forecastMaxRBC,
                    'ForecMax_Avg' => $forecastMaxAvg,
                    'avg_rn'       => $avg_up ?? $hin_up ?? 0.0,
                    'Consumption / 2 months' => $consumption,
                    'Comment' => $sku['Comment'],
                    'LastModifiedAt' => $lastModified
                ];
            }
        }
        foreach ($stockkeeping as $stock) {
            $key = $stock['Item_No'] . '|' . $stock['Location_Code'];
            if (!isset($matchedKeys[$key])) {
                $lastModified = $stock['LastModifiedAt'] ?? null;
                $consumption = ceil($ledgerQuantities[$key] ?? 0);
                if ($lastModified !== null) {
                    $lastModified = date('d F Y', strtotime($lastModified));
                }
                $realMin = $stock['MinInven'] ?? 0;
                
                $realMax = $stock['MaxInven'] ?? 0;
                if ($realMin > 0  or $realMax > 0  or $consumption > 0) {
                        $suggestMin = ceil((7) * ($consumption / 60));
                        $suggestMax = ceil((30) * ($consumption / 60));

                        if ($activeCompany === 'Regent' || $activeCompany === 'SUPER') {
                            $safeDiv = function (float $a, float $b): float {
                                return $b != 0.0 ? ($a / $b) : 0.0;
                            };

                            $hic_up = $safeDiv($forecastRoomsHIC, $historyRoomsHIC);
                            $rbc_up = $safeDiv($forecastRoomsRBC, $historyRoomsRBC);
                            $avg_up = ($hic_up + $rbc_up) / 2.0;

                            $forecastMinHIC = (int) ceil($suggestMin * $hic_up);
                            $forecastMinRBC = (int) ceil($suggestMin * $rbc_up);
                            $forecastMinAvg = (int) ceil($suggestMin * $avg_up);

                            $forecastMaxHIC = (int) ceil($suggestMax * $hic_up);
                            $forecastMaxRBC = (int) ceil($suggestMax * $rbc_up);
                            $forecastMaxAvg = (int) ceil($suggestMax * $avg_up);
                        } elseif ($activeCompany === 'HIN') {
                            $safeDiv = function (float $a, float $b): float {
                                return $b != 0.0 ? ($a / $b) : 0.0;
                            };
                            $hin_up       = $safeDiv($forecastRoomsHIN, $historyRoomsHIN);
                            $forecastMinHIC = $forecastMinRBC = 0;
                            $forecastMaxHIC = $forecastMaxRBC = 0;
                            $forecastMinAvg = (int) ceil($suggestMin * $hin_up);
                            $forecastMaxAvg = (int) ceil($suggestMax * $hin_up);
                        } 
                        $result[] = [
                            'Item_No' => $stock['Item_No'],
                            'Description' => $stock['Description']  ?? '',
                            'Location' => $stock['Location_Code'],
                            'Stock_For' => 14,
                            'Lead Time' => 7,
                            'SuggestMinQty' => $suggestMin,
                            'ActMinQty' => 0,
                            'RealMinQty' => $stock['MinInven'] ?? 0,
                            'SuggestMaxQty' => $suggestMax,
                            'ActMaxQty' => 0,
                            'RealMaxQty' => $stock['MaxInven'] ?? 0,

                            'ForecMin_HIC' => $forecastMinHIC,
                            'ForecMin_RBC' => $forecastMinRBC,
                            'ForecMin_Avg' => $forecastMinAvg,

                            'ForecMax_HIC' => $forecastMaxHIC,
                            'ForecMax_RBC' => $forecastMaxRBC,
                            'ForecMax_Avg' => $forecastMaxAvg,
                            'avg_rn'       => $avg_up ?? $hin_up ?? 0.0,
                            'Consumption / 2 months' => $consumption,
                            'Comment' => '',
                            'LastModifiedAt' => $lastModified,
                            'IsMatched' => false
                        ];
                    }
            }
        }
        return [
            'ledger' => $ledgerData,
            'sku' => $skuData,
            'result' => $result
        ];
        });
    }


    public function getSKUImport()
    {
        $baseUrl = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT');
        $url = "/ODataV4/Company('{$this->companyId}')/APIStockkeeping";

        $fullUrl = $baseUrl . $url;

        $response = $this->client->get($fullUrl, [
            'headers' => [
                'Authorization' => 'Bearer ' . $this->token,
                'Accept'        => 'application/json',
            ]
        ]);

        return json_decode((string) $response->getBody(), true);
    }

    public function getItemIdByNo(string $itemNo)
    {
        $filter = urlencode("number eq '$itemNo'");
        $response = $this->client->get("https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
       "/api/v2.0/companies({$this->companyId})/items?\$filter=$filter", [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Content-Type'  => 'application/json',
            ]
        ]);

        $items = json_decode($response->getBody(), true);
        return $items['value'][0]['id'] ?? null;
    }

    public function getLocationIdByCode($locationCode)
    {
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/api/v2.0/companies({$this->companyId})/locations?\$filter=code eq '{$locationCode}'";

        $response = $this->client->get($url, [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept' => 'application/json'
            ]
        ]);

        $data = json_decode((string)$response->getBody(), true);
        
        return $data['value'][0]['id'] ?? null;
    }

    public function updateSKUBuffer($itemNo, $MinStock, $MaxStock, $stockFor, $leadTime, $comment, $userName)
    {   
        set_time_limit(300);
        $start = microtime(true);
        $itemId = $this->getItemIdByNo($itemNo);
        $locationCode = str_contains($userName, 'titania@citbi.onmicrosoft.com') ? 'CI.1010' : 'CI.1020';
        $locationId = $this->getLocationIdByCode($locationCode);
        $payload = json_encode([
            'Min_Inventory' => (float) $MinStock,
            'Max_Inventory' => (float) $MaxStock,
            'Stock_For' => $stockFor,
            'Comment' => $comment,
            'Lead_Time' => $leadTime
        ]);
        $headers = [
            'Authorization' => "Bearer {$this->token}",
            'Accept' => 'application/json',
            'Content-Type' => 'application/json'
        ];
        $baseUrl = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT');
        $company = "/ODataV4/Company('" . $this->companyId . "')";
        $patchUrl = $baseUrl . $company . "/SKU_Import(Item_No='" . rawurlencode($itemNo) . "',Location='" . rawurlencode($locationCode) . "')";

        try {
            $response = $this->client->request('PATCH', $patchUrl, [
                'headers' => $headers,
                'body'    => $payload
            ]);
        } catch (\Exception $e) {
            Log::error("Failed to update SKU for {$itemNo} at {$locationCode}: " . $e->getMessage());
            return null;
        }

        $elapsed = microtime(true) - $start;
        Log::info("Create SKU runtime: {$elapsed} seconds");
        return json_decode($response->getBody(), true);
    }


    function getItemLedger()
    {
        $twoMonthsAgo = now()->subMonths(2)->format('Y-m-d');
        $filter = urlencode("(Location_Code eq 'CI.1010' or Location_Code eq 'CI.1020') and Entry_Type eq 'Negative Adjmt.'");
        $select = urlencode("Quantity,Item_No,Base_Unit,Location_Code,Unit_of_Measure_Code");
        // $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
        //     "/ODataV4/Company('" . $this->companyId . "')/Item_Ledger_Entries_Excel?\$filter={$filter}&\$select={$select}";
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/Company('" . $this->companyId . "')/Item_Ledger_Entries_Excel?\$filter={$filter}";
        $totalCount = 0;
        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer' => 'odata.maxpagesize=5000'
            ]
        ];
        $allData = [];
        do {
            $response = $this->client->get($url, $headers);
            $decoded = json_decode((string)$response->getBody(), true);

            if (isset($decoded['value'])) {
                $allData = array_merge($allData, $decoded['value']);
            }
            
            if (isset($decoded['@odata.count'])) {
                $totalCount = $decoded['@odata.count'];
            }


            $url = $decoded['@odata.nextLink'] ?? null;
        } while ($url);
        return ['value' => $allData];
    }

    function getSKUMapping()
    {
        $filter = urlencode("(Location eq 'CI.1010' or Location eq 'CI.1020')");
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/Company('" . $this->companyId . "')/SKU_Import?\$filter={$filter}";
        
        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer' => 'odata.maxpagesize=1000'
            ]
        ];
        $allData = [];
        do {
            $response = $this->client->get($url, $headers);
            $decoded = json_decode((string)$response->getBody(), true);

            if (isset($decoded['value'])) {
                $allData = array_merge($allData, $decoded['value']);
            }

            $url = $decoded['@odata.nextLink'] ?? null;
        } while ($url);
        return ['value' => $allData];
    }

    function getItemInvenLoc()
    {
        $filter = urlencode("(Location eq 'CI.1010' or Location eq 'CI.1020')");
        $url = "https://api.businesscentral.dynamics.com/v2.0/" . env('AZURE_TENANT_ID') . "/" . env('BC_ENVIRONMENT') .
            "/ODataV4/Company('" . $this->companyId . "')/ItemInvenLoc";
    
        
        $headers = [
            'headers' => [
                'Authorization' => "Bearer {$this->token}",
                'Accept'        => 'application/json',
                'Prefer' => 'odata.maxpagesize=1000'
            ]
        ];
        $allData = [];
        $response = $this->client->get($url, $headers);
        $decoded = json_decode((string)$response->getBody(), true);
        dd($decoded);
        return [$decoded];
    }
}