<?php

namespace App\Services;

use GuzzleHttp\Client;
use Illuminate\Support\Facades\Storage;

class OneDriveService
{
    public function __construct(private ?Client $http = null)
    {
        $this->http = $this->http ?: new Client([
            'base_uri' => 'https://graph.microsoft.com/v1.0/',
            'http_errors' => false,
            'timeout' => 120,
        ]);
    }

    private function withAuth(string $token): array
    {
        return ['headers' => ['Authorization' => "Bearer {$token}"]];
    }

    public function listByPath(string $accessToken, string $folderPath = ''): array
    {
        $path = $folderPath ? 'me/drive/root:/'.rawurlencode($folderPath).':/children'
                            : 'me/drive/root/children';
        $resp = $this->http->get($path, $this->withAuth($accessToken));
        $data = json_decode((string)$resp->getBody(), true);
        if (($resp->getStatusCode() !== 200) || !isset($data['value'])) {
            throw new \RuntimeException("List error {$resp->getStatusCode()}: ".(string)$resp->getBody());
        }
        return $data['value'];
    }

    private function encodePathSegments(string $path): string
    {
        $segments = array_map('rawurlencode', array_filter(explode('/', $path), 'strlen'));
        return implode('/', $segments);
    }

    public function downloadByPath(string $accessToken, string $filePath, string $saveAs): string
    {
        $encoded = $this->encodePathSegments($filePath); // e.g. Documents/Bali%20Canggu/history_forecast%28HIC%29.txt
        $resp = $this->http->get("me/drive/root:/{$encoded}:/content", array_merge(
            $this->withAuth($accessToken), ['stream' => true]
        ));
        if ($resp->getStatusCode() !== 200) {
            throw new \RuntimeException("Download error {$resp->getStatusCode()}: ".(string)$resp->getBody());
        }
        $dest = 'onedrive/'.ltrim($saveAs, '/');
        \Illuminate\Support\Facades\Storage::disk('local')->put($dest, $resp->getBody()->getContents());
        return storage_path('app/'.$dest);
    }

    public function downloadByShareUrl(string $accessToken, string $shareUrl, string $saveAs): string
    {
        $encoded = rtrim(strtr(base64_encode($shareUrl), ['+' => '-', '/' => '_']), '=');
    
        $resp = $this->http->get("shares/u!{$encoded}/driveItem", $this->withAuth($accessToken));
        \Log::info('Graph driveItem response', [
            'status' => $resp->getStatusCode(),
            'body'   => substr((string) $resp->getBody(), 0, 500), // log only first 500 chars
        ]);
    
        if ($resp->getStatusCode() !== 200) {
            \Log::error("Graph driveItem error {$resp->getStatusCode()}", [
                'body' => (string)$resp->getBody()
            ]);
            throw new \RuntimeException("Graph driveItem error {$resp->getStatusCode()}");
        }
    
        $json = json_decode((string)$resp->getBody(), true);
    
        if (! isset($json['@microsoft.graph.downloadUrl'])) {
            \Log::error('Graph response missing downloadUrl', $json);
            throw new \RuntimeException("Graph response did not include a downloadUrl");
        }
    
        $downloadUrl = $json['@microsoft.graph.downloadUrl'];
        \Log::info('Resolved downloadUrl', ['url' => $downloadUrl]);
    
        $bin = $this->http->get($downloadUrl, ['stream' => true]);
        \Log::info('Download attempt', ['status' => $bin->getStatusCode()]);
    
        if ($bin->getStatusCode() !== 200) {
            \Log::error("Download error {$bin->getStatusCode()}", [
                'body' => substr((string)$bin->getBody(), 0, 500),
            ]);
            throw new \RuntimeException("Download error {$bin->getStatusCode()}");
        }
    
        $dest = 'onedrive/' . ltrim($saveAs, '/');
        $bytes = $bin->getBody()->getContents();
        \Log::info('Saving file', ['dest' => $dest, 'size' => strlen($bytes)]);
    
        \Storage::disk('local')->put($dest, $bytes);
    
        $savedPath = storage_path('app/' . $dest);
        \Log::info('File saved', ['path' => $savedPath]);
    
        return $savedPath;
    }
    
   //  public function downloadFromUserPath(string $accessToken, string $userIdOrUpn, string $filePath, string $saveAs): string
//     {
//         $encode = function (string $p) {
//             return implode('/', array_map('rawurlencode', array_filter(explode('/', $p), 'strlen')));
//         };
//         $encodedPath = $encode($filePath);
    
//         $relative = 'onedrive/' . ltrim($saveAs, '/');
//         $absolute = storage_path('app/'.$relative);
//         @mkdir(dirname($absolute), 0775, true);
    
//         // 🔎 --- Diagnostics before download ---
//         try {
//             $probe = $this->http->get("users/{$userIdOrUpn}/drive", $this->withAuth($accessToken));
//             \Log::info('probe users/drive', [
//                 'status' => $probe->getStatusCode(),
//                 'body'   => substr((string)$probe->getBody(), 0, 300)
//             ]);
    
//             $listRoot = $this->http->get("users/{$userIdOrUpn}/drive/root/children", $this->withAuth($accessToken));
//             \Log::info('list root children', [
//                 'status' => $listRoot->getStatusCode(),
//                 'body'   => (string)$listRoot->getBody()
//             ]);
    
//             $listDocs = $this->http->get("users/{$userIdOrUpn}/drive/root:/Bali Canggu:/children", $this->withAuth($accessToken));
//             \Log::info('list Documents children', [
//                 'status' => $listDocs->getStatusCode(),
//                 'body'   => (string)$listDocs->getBody()
//             ]);
    
//             $search = $this->http->get("users/{$userIdOrUpn}/drive/root/search(q='history_forecast(HIC).txt')", $this->withAuth($accessToken));
//             \Log::info('search result', [
//                 'status' => $search->getStatusCode(),
//                 'body'   => substr((string)$search->getBody(), 0, 500)
//             ]);
//         } catch (\Throwable $e) {
//             \Log::error('Diagnostics failed', ['msg' => $e->getMessage()]);
//         }
//         // --- end diagnostics ---
    
//         // 📥 --- Actual download attempt ---
//         $res = $this->http->get("users/{$userIdOrUpn}/drive/root:/{$encodedPath}:/content", [
//             'headers'         => ['Authorization' => "Bearer {$accessToken}"],
//             'http_errors'     => false,
//             'allow_redirects' => true,
//             'stream'          => true,
//             'sink'            => $absolute,
//             'timeout'         => 300,
//             'base_uri'        => 'https://graph.microsoft.com/v1.0/',
//         ]);
    
//         if (!in_array($res->getStatusCode(), [200, 206], true)) {
//             if (is_file($absolute)) @unlink($absolute);
//             throw new \RuntimeException("UserPath download error {$res->getStatusCode()}: ".substr((string)$res->getBody(), 0, 500));
//         }
    
//         $head = @file_get_contents($absolute, false, null, 0, 200) ?: '';
//         if (str_starts_with($head, '<!DOCTYPE html') || str_starts_with($head, '<html') || str_starts_with($head, '{"error"')) {
//             @unlink($absolute);
//             throw new \RuntimeException('Downloaded content looks like HTML/JSON, not the file. Check perms/path.');
//         }
    
//         \Log::info('DownloadFromUserPath saved', ['path' => $absolute, 'size' => @filesize($absolute)]);
//         return $absolute;
//     }

    public function downloadFromUserPath(string $accessToken, string $userIdOrUpn, string $filePath, string $saveAs): string
    {
        $encodeSegs = function (string $p): string {
            $p = trim($p, "/ \t\n\r\0\x0B");
            $parts = array_filter(explode('/', $p), 'strlen');
            return implode('/', array_map('rawurlencode', $parts));
        };
    
        $relative = 'onedrive/' . ltrim($saveAs, '/');
        $absolute = storage_path('app/'.$relative);
        @mkdir(dirname($absolute), 0775, true);
    
        // ---- 1) Try ROOT path (this is where "Bali Canggu" really is) ----
        $encodedPath = $encodeSegs($filePath); // e.g. "Bali%20Canggu/history_forecast%28HIC%29.txt"
        $url = "users/{$userIdOrUpn}/drive/root:/{$encodedPath}:/content";
        \Log::info('download attempt (root)', ['url' => "https://graph.microsoft.com/v1.0/".$url, 'save' => $absolute]);
    
        $res = $this->http->get($url, [
            'headers'         => ['Authorization' => "Bearer {$accessToken}"],
            'http_errors'     => false,
            'allow_redirects' => true,
            'sink'            => $absolute,            // let Guzzle manage the file handle
            'timeout'         => 300,
            'base_uri'        => 'https://graph.microsoft.com/v1.0/',
            'decode_content'  => true,
        ]);
    
        $status = $res->getStatusCode();
        \Log::info('download response (root)', [
            'status'         => $status,
            'content_type'   => $res->getHeaderLine('Content-Type'),
            'content_length' => $res->getHeaderLine('Content-Length'),
        ]);
    
        if (in_array($status, [200, 206], true)) {
            // Wait briefly for FS to reflect the write
            clearstatcache(true, $absolute);
            $size = @filesize($absolute);
            if (!$size || $size <= 0) {
                for ($i = 0; $i < 5 && (!$size || $size <= 0); $i++) {
                    usleep(150_000); // 150 ms
                    clearstatcache(true, $absolute);
                    $size = @filesize($absolute);
                }
            }
            if (!$size || $size <= 0) {
                @unlink($absolute);
                throw new \RuntimeException('Downloaded zero bytes (stat/cache timing).');
            }
    
            // Final sniff to ensure not HTML/JSON
            $head = @file_get_contents($absolute, false, null, 0, 200) ?: '';
            if (str_starts_with($head, '<!DOCTYPE html') || str_starts_with($head, '<html') || str_starts_with($head, '{"error"')) {
                @unlink($absolute);
                throw new \RuntimeException('Downloaded content looks like HTML/JSON, not the file.');
            }
    
            \Log::info('Download saved', ['path' => $absolute, 'size' => $size]);
            return $absolute; // <<< IMPORTANT: stop here on success
        }
    
        // ---- 2) Only if root path is NOT found, try Documents prefix ----
        if ($status === 404) {
            $url2 = "users/{$userIdOrUpn}/drive/root:/".$encodeSegs("Documents/{$filePath}").":/content";
            \Log::info('download attempt (Documents)', ['url' => "https://graph.microsoft.com/v1.0/".$url2, 'save' => $absolute]);
    
            $res2 = $this->http->get($url2, [
                'headers'         => ['Authorization' => "Bearer {$accessToken}"],
                'http_errors'     => false,
                'allow_redirects' => true,
                'sink'            => $absolute,
                'timeout'         => 300,
                'base_uri'        => 'https://graph.microsoft.com/v1.0/',
                'decode_content'  => true,
            ]);
    
            $status2 = $res2->getStatusCode();
            \Log::info('download response (Documents)', [
                'status'         => $status2,
                'content_type'   => $res2->getHeaderLine('Content-Type'),
                'content_length' => $res2->getHeaderLine('Content-Length'),
            ]);
    
            if (!in_array($status2, [200, 206], true)) {
                if (is_file($absolute)) @unlink($absolute);
                throw new \RuntimeException("Download error {$status2} (Documents).");
            }
    
            clearstatcache(true, $absolute);
            $size2 = @filesize($absolute);
            if (!$size2 || $size2 <= 0) {
                for ($i = 0; $i < 5 && (!$size2 || $size2 <= 0); $i++) {
                    usleep(150_000);
                    clearstatcache(true, $absolute);
                    $size2 = @filesize($absolute);
                }
            }
            if (!$size2 || $size2 <= 0) {
                @unlink($absolute);
                throw new \RuntimeException('Downloaded zero bytes (Documents path).');
            }
    
            $head2 = @file_get_contents($absolute, false, null, 0, 200) ?: '';
            if (str_starts_with($head2, '<!DOCTYPE html') || str_starts_with($head2, '<html') || str_starts_with($head2, '{"error"')) {
                @unlink($absolute);
                throw new \RuntimeException('Downloaded content looks like HTML/JSON (Documents).');
            }
    
            \Log::info('Download saved (Documents)', ['path' => $absolute, 'size' => $size2]);
            return $absolute;
        }
    
        if (is_file($absolute)) @unlink($absolute);
        throw new \RuntimeException("Download error {$status} (root).");
    }

    
    public function findUserItemByName(string $accessToken, string $userIdOrUpn, string $fileName): ?array
    {
        $resp = $this->http->get(
            "users/{$userIdOrUpn}/drive/root/search(q='".addslashes($fileName)."')",
            ['headers' => ['Authorization' => "Bearer {$accessToken}"], 'http_errors' => false, 'timeout' => 60]
        );
    
        if ($resp->getStatusCode() !== 200) {
            \Log::error('search error', ['status' => $resp->getStatusCode(), 'body' => substr((string)$resp->getBody(), 0, 500)]);
            return null;
        }
    
        $data = json_decode((string)$resp->getBody(), true);
        if (empty($data['value'])) return null;
    
        // Prefer exact name match if present
        foreach ($data['value'] as $it) {
            if (isset($it['name']) && strcasecmp($it['name'], $fileName) === 0) {
                return [
                    'id'   => $it['id'] ?? null,
                    'name' => $it['name'] ?? null,
                    'path' => $it['parentReference']['path'] ?? null,
                    'webUrl' => $it['webUrl'] ?? null,
                ];
            }
        }
        $it = $data['value'][0];
        return [
            'id'   => $it['id'] ?? null,
            'name' => $it['name'] ?? null,
            'path' => $it['parentReference']['path'] ?? null,
            'webUrl' => $it['webUrl'] ?? null,
        ];
    }


}
