<?php
require_once "config/connection.php";

$type = $_POST['type'] ?? '';

switch ($type) {
    case 'ongoing':
        getOngoingData($conn);
        break;
    case 'dipinjam':
        getDipinjamData($conn);
        break;
    case 'overdue':
        getOverdueData($conn);
        break;
    case 'borrowed_detail':
        getBorrowedItemDetail($conn);
        break;
    case 'reservation_detail':
        getReservationDetail($conn);
        break;
    default:
        echo '<div class="text-red-500 text-center p-4">Invalid request type</div>';
}

function getOngoingData($conn) {
    $query = "
        SELECT 
            st.id,
            st.reservation_number,
            st.tanggal_transaksi,
            st.tanggal_pengembalian,
            bu.nama_borrower,
            bu.lokasi_posisi,
            COUNT(ti.id) as total_items,
            COUNT(CASE WHEN ti.status_waktu = 'overdue' THEN 1 END) as overdue_items,
            TIMESTAMPDIFF(MINUTE, st.tanggal_pengembalian, NOW()) as overdue_minutes
        FROM stock_transactions st
        INNER JOIN borrower_users bu ON st.borrower_id = bu.id
        LEFT JOIN transaction_items ti ON st.id = ti.transaction_id
        WHERE st.status_peminjaman = 'Dipinjam'
        GROUP BY st.id, st.reservation_number, st.tanggal_transaksi, st.tanggal_pengembalian, bu.nama_borrower, bu.lokasi_posisi
        ORDER BY st.tanggal_transaksi DESC
    ";
    
    $result = $conn->query($query);
    
    if ($result->num_rows === 0) {
        echo '<div class="text-center py-8">
                <i class="fas fa-clock text-gray-400 text-4xl mb-4"></i>
                <p class="text-gray-500 text-lg">Tidak ada transaksi On Going</p>
              </div>';
        return;
    }
    
    echo '<div class="overflow-x-auto">';
    echo '<table class="min-w-full divide-y divide-gray-200">';
    echo '<thead class="bg-gray-50">';
    echo '<tr>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Reservation Number</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Peminjam</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Lokasi Posisi</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Tanggal Pinjam</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Tanggal Kembali</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Total Items</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>';
    echo '</tr>';
    echo '</thead>';
    echo '<tbody class="bg-white divide-y divide-gray-200">';
    
    while ($row = $result->fetch_assoc()) {
        $is_overdue = $row['overdue_minutes'] > 15;
        $status_class = $is_overdue ? 'bg-red-100 text-red-800' : 'bg-blue-100 text-blue-800';
        $status_text = $is_overdue ? 'Overdue' : 'On Time';
        
        echo '<tr class="hover:bg-gray-50">';
        
        // Reservation Number - clickable
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm font-medium">';
        echo '<button onclick="showReservationDetail(\'' . htmlspecialchars($row['reservation_number']) . '\', \'ongoing\')" class="text-blue-600 hover:text-blue-800 hover:underline font-mono">';
        echo htmlspecialchars($row['reservation_number']);
        echo '</button>';
        echo '</td>';
        
        // Peminjam
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">';
        echo htmlspecialchars($row['nama_borrower']);
        echo '</td>';
        
        // Lokasi Posisi
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">';
        echo htmlspecialchars($row['lokasi_posisi']);
        echo '</td>';
        
        // Tanggal Pinjam
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">';
        echo $row['tanggal_transaksi'] ? date('d/m/Y H:i', strtotime($row['tanggal_transaksi'])) : '-';
        echo '</td>';
        
        // Tanggal Kembali
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">';
        echo $row['tanggal_pengembalian'] ? date('d/m/Y H:i', strtotime($row['tanggal_pengembalian'])) : '-';
        echo '</td>';
        
        // Total Items
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">';
        echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800">';
        echo $row['total_items'] . ' items';
        if ($row['overdue_items'] > 0) {
            echo ' <span class="ml-1 text-red-600">(' . $row['overdue_items'] . ' overdue)</span>';
        }
        echo '</span>';
        echo '</td>';
        
        // Status
        echo '<td class="px-6 py-4 whitespace-nowrap">';
        echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ' . $status_class . '">';
        echo $status_text;
        if ($is_overdue) {
            echo ' (' . $row['overdue_minutes'] . ' min)';
        }
        echo '</span>';
        echo '</td>';
        
        echo '</tr>';
    }
    
    echo '</tbody>';
    echo '</table>';
    echo '</div>';
}

function getDipinjamData($conn) {
    $query = "
        SELECT 
            csi.id,
            csi.kode_barang,
            csi.rack_no,
            csi.lokasi,
            csi.status,
            cc.component_name,
            mcp.code_color,
            b.nama_buyer,
            st.reservation_number,
            bu.nama_borrower,
            st.tanggal_transaksi,
            st.tanggal_pengembalian,
            TIMESTAMPDIFF(MINUTE, st.tanggal_pengembalian, NOW()) as overdue_minutes
        FROM color_stock_items csi
        INNER JOIN color_components cc ON csi.component_id = cc.id
        INNER JOIN master_color_panel mcp ON csi.color_id = mcp.id
        INNER JOIN buyers b ON mcp.buyer_id = b.id
        LEFT JOIN transaction_items ti ON csi.id = ti.stock_item_id
        LEFT JOIN stock_transactions st ON ti.transaction_id = st.id AND st.status_peminjaman = 'Dipinjam'
        LEFT JOIN borrower_users bu ON st.borrower_id = bu.id
        WHERE csi.status = 'Dipinjam'
        ORDER BY b.nama_buyer, mcp.code_color, cc.component_name, csi.kode_barang
    ";
    
    $result = $conn->query($query);
    
    if ($result->num_rows === 0) {
        echo '<div class="text-center py-8">
                <i class="fas fa-hand-holding text-gray-400 text-4xl mb-4"></i>
                <p class="text-gray-500 text-lg">Tidak ada code barang yang dipinjam</p>
              </div>';
        return;
    }
    
    echo '<div class="overflow-x-auto">';
    echo '<table class="min-w-full divide-y divide-gray-200">';
    echo '<thead class="bg-gray-50">';
    echo '<tr>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Kode Barang</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Buyer</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Color Code</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Component</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Peminjam</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Reservation</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status Waktu</th>';
    echo '</tr>';
    echo '</thead>';
    echo '<tbody class="bg-white divide-y divide-gray-200">';
    
    while ($row = $result->fetch_assoc()) {
        $is_overdue = $row['overdue_minutes'] > 15;
        
        echo '<tr class="hover:bg-gray-50">';
        
        // Kode Barang - clickable for detail
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm font-mono font-medium">';
        echo '<button onclick="showBorrowedItemDetail(\'' . htmlspecialchars($row['kode_barang']) . '\')" class="text-blue-600 hover:text-blue-800 hover:underline">';
        echo htmlspecialchars($row['kode_barang']);
        echo '</button>';
        echo '</td>';
        
        // Buyer
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">';
        echo htmlspecialchars($row['nama_buyer']);
        echo '</td>';
        
        // Color Code
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">';
        echo htmlspecialchars($row['code_color']);
        echo '</td>';
        
        // Component
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">';
        echo '<span class="bg-blue-50 px-2 py-1 rounded text-xs font-medium">';
        echo htmlspecialchars($row['component_name']);
        echo '</span>';
        echo '</td>';
        
        // Peminjam
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">';
        echo $row['nama_borrower'] ? htmlspecialchars($row['nama_borrower']) : '<span class="text-gray-400">-</span>';
        echo '</td>';
        
        // Reservation - clickable if available
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">';
        if ($row['reservation_number']) {
            echo '<button onclick="showReservationDetail(\'' . htmlspecialchars($row['reservation_number']) . '\', \'dipinjam\')" class="text-purple-600 hover:text-purple-800 hover:underline font-mono">';
            echo htmlspecialchars($row['reservation_number']);
            echo '</button>';
        } else {
            echo '<span class="text-gray-400">-</span>';
        }
        echo '</td>';
        
        // Status Waktu
        echo '<td class="px-6 py-4 whitespace-nowrap">';
        if ($row['reservation_number']) {
            if ($is_overdue) {
                $hours = floor($row['overdue_minutes'] / 60);
                $minutes = $row['overdue_minutes'] % 60;
                echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">';
                echo 'Overdue ';
                if ($hours > 0) {
                    echo '(' . $hours . 'h ' . $minutes . 'm)';
                } else {
                    echo '(' . $minutes . ' menit)';
                }
                echo '</span>';
            } else {
                echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">On Time</span>';
            }
        } else {
            echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800">No Data</span>';
        }
        echo '</td>';
        
        echo '</tr>';
    }
    
    echo '</tbody>';
    echo '</table>';
    echo '</div>';
}

function getOverdueData($conn) {
    $query = "
        SELECT 
            st.id,
            st.reservation_number,
            st.tanggal_transaksi,
            st.tanggal_pengembalian,
            bu.nama_borrower,
            bu.lokasi_posisi,
            COUNT(ti.id) as total_items,
            COUNT(CASE WHEN ti.status_waktu = 'overdue' THEN 1 END) as overdue_items,
            TIMESTAMPDIFF(MINUTE, st.tanggal_pengembalian, NOW()) as overdue_minutes
        FROM stock_transactions st
        INNER JOIN borrower_users bu ON st.borrower_id = bu.id
        INNER JOIN transaction_items ti ON st.id = ti.transaction_id
        WHERE st.status_peminjaman = 'Dipinjam' 
        AND ti.status_waktu = 'overdue'
        AND TIMESTAMPDIFF(MINUTE, st.tanggal_pengembalian, NOW()) > 15
        GROUP BY st.id, st.reservation_number, st.tanggal_transaksi, st.tanggal_pengembalian, bu.nama_borrower, bu.lokasi_posisi
        HAVING overdue_items > 0
        ORDER BY overdue_minutes DESC
    ";
    
    $result = $conn->query($query);
    
    if ($result->num_rows === 0) {
        echo '<div class="text-center py-8">
                <i class="fas fa-exclamation-triangle text-gray-400 text-4xl mb-4"></i>
                <p class="text-gray-500 text-lg">Tidak ada transaksi overdue</p>
              </div>';
        return;
    }
    
    echo '<div class="overflow-x-auto">';
    echo '<table class="min-w-full divide-y divide-gray-200">';
    echo '<thead class="bg-gray-50">';
    echo '<tr>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Reservation Number</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Peminjam</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Lokasi Posisi</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Tanggal Pinjam</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Tanggal Kembali</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Items Overdue</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Keterlambatan</th>';
    echo '</tr>';
    echo '</thead>';
    echo '<tbody class="bg-white divide-y divide-gray-200">';
    
    while ($row = $result->fetch_assoc()) {
        echo '<tr class="hover:bg-gray-50">';
        
        // Reservation Number - clickable
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm font-medium">';
        echo '<button onclick="showReservationDetail(\'' . htmlspecialchars($row['reservation_number']) . '\', \'overdue\')" class="text-red-600 hover:text-red-800 hover:underline font-mono">';
        echo htmlspecialchars($row['reservation_number']);
        echo '</button>';
        echo '</td>';
        
        // Peminjam
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">';
        echo htmlspecialchars($row['nama_borrower']);
        echo '</td>';
        
        // Lokasi Posisi
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">';
        echo htmlspecialchars($row['lokasi_posisi']);
        echo '</td>';
        
        // Tanggal Pinjam
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">';
        echo $row['tanggal_transaksi'] ? date('d/m/Y H:i', strtotime($row['tanggal_transaksi'])) : '-';
        echo '</td>';
        
        // Tanggal Kembali
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">';
        echo $row['tanggal_pengembalian'] ? date('d/m/Y H:i', strtotime($row['tanggal_pengembalian'])) : '-';
        echo '</td>';
        
        // Items Overdue
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">';
        echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">';
        echo $row['overdue_items'] . ' dari ' . $row['total_items'] . ' items';
        echo '</span>';
        echo '</td>';
        
        // Keterlambatan
        echo '<td class="px-6 py-4 whitespace-nowrap">';
        $hours = floor($row['overdue_minutes'] / 60);
        $minutes = $row['overdue_minutes'] % 60;
        echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">';
        if ($hours > 0) {
            echo $hours . 'h ' . $minutes . 'm';
        } else {
            echo $minutes . ' menit';
        }
        echo '</span>';
        echo '</td>';
        
        echo '</tr>';
    }
    
    echo '</tbody>';
    echo '</table>';
    echo '</div>';
}

function getBorrowedItemDetail($conn) {
    $kode_barang = $_POST['kode_barang'] ?? '';
    
    if (empty($kode_barang)) {
        echo '<div class="text-red-500 text-center p-4">Kode barang tidak valid</div>';
        return;
    }
    
    $query = "
        SELECT 
            csi.kode_barang,
            csi.rack_no,
            csi.lokasi,
            csi.status,
            cc.component_name,
            mcp.code_color,
            b.nama_buyer,
            st.reservation_number,
            st.tanggal_transaksi,
            st.tanggal_pengembalian,
            bu.nama_borrower,
            bu.lokasi_posisi,
            ti.status_waktu,
            ti.dikembalikan,
            TIMESTAMPDIFF(MINUTE, st.tanggal_pengembalian, NOW()) as overdue_minutes
        FROM color_stock_items csi
        INNER JOIN color_components cc ON csi.component_id = cc.id
        INNER JOIN master_color_panel mcp ON csi.color_id = mcp.id
        INNER JOIN buyers b ON mcp.buyer_id = b.id
        LEFT JOIN transaction_items ti ON csi.id = ti.stock_item_id
        LEFT JOIN stock_transactions st ON ti.transaction_id = st.id AND st.status_peminjaman = 'Dipinjam'
        LEFT JOIN borrower_users bu ON st.borrower_id = bu.id
        WHERE csi.kode_barang = ?
    ";
    
    $stmt = $conn->prepare($query);
    $stmt->bind_param("s", $kode_barang);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        echo '<div class="text-red-500 text-center p-4">Data tidak ditemukan</div>';
        return;
    }
    
    $row = $result->fetch_assoc();
    
    echo '<div class="space-y-4">';
    
    // Item Information
    echo '<div class="bg-blue-50 rounded-lg p-4">';
    echo '<h4 class="text-lg font-semibold text-blue-900 mb-3">Informasi Item</h4>';
    echo '<div class="grid grid-cols-2 gap-3 text-sm">';
    echo '<div><span class="font-medium text-gray-600">Kode Barang:</span> <span class="font-mono">' . htmlspecialchars($row['kode_barang']) . '</span></div>';
    echo '<div><span class="font-medium text-gray-600">Buyer:</span> ' . htmlspecialchars($row['nama_buyer']) . '</div>';
    echo '<div><span class="font-medium text-gray-600">Color Code:</span> ' . htmlspecialchars($row['code_color']) . '</div>';
    echo '<div><span class="font-medium text-gray-600">Component:</span> ' . htmlspecialchars($row['component_name']) . '</div>';
    echo '<div><span class="font-medium text-gray-600">Rack No:</span> <span class="font-mono">' . htmlspecialchars($row['rack_no']) . '</span></div>';
    echo '<div><span class="font-medium text-gray-600">Lokasi:</span> ' . htmlspecialchars($row['lokasi']) . '</div>';
    echo '</div>';
    echo '</div>';
    
    if ($row['reservation_number']) {
        // Transaction Information
        echo '<div class="bg-green-50 rounded-lg p-4">';
        echo '<h4 class="text-lg font-semibold text-green-900 mb-3">Informasi Peminjaman</h4>';
        echo '<div class="grid grid-cols-2 gap-3 text-sm">';
        echo '<div><span class="font-medium text-gray-600">Reservation:</span> ';
        echo '<button onclick="showReservationDetail(\'' . htmlspecialchars($row['reservation_number']) . '\', \'borrowed\')" class="text-purple-600 hover:text-purple-800 hover:underline font-mono">';
        echo htmlspecialchars($row['reservation_number']);
        echo '</button></div>';
        echo '<div><span class="font-medium text-gray-600">Peminjam:</span> ' . htmlspecialchars($row['nama_borrower']) . '</div>';
        echo '<div><span class="font-medium text-gray-600">Lokasi Posisi:</span> ' . htmlspecialchars($row['lokasi_posisi']) . '</div>';
        echo '<div><span class="font-medium text-gray-600">Tanggal Pinjam:</span> ' . ($row['tanggal_transaksi'] ? date('d/m/Y H:i', strtotime($row['tanggal_transaksi'])) : '-') . '</div>';
        echo '<div><span class="font-medium text-gray-600">Tanggal Kembali:</span> ' . ($row['tanggal_pengembalian'] ? date('d/m/Y H:i', strtotime($row['tanggal_pengembalian'])) : '-') . '</div>';
        
        // Status waktu
        echo '<div class="col-span-2">';
        echo '<span class="font-medium text-gray-600">Status Waktu:</span> ';
        if ($row['status_waktu'] === 'overdue') {
            $hours = floor($row['overdue_minutes'] / 60);
            $minutes = $row['overdue_minutes'] % 60;
            echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">';
            echo 'Overdue ';
            if ($hours > 0) {
                echo '(' . $hours . 'h ' . $minutes . 'm)';
            } else {
                echo '(' . $minutes . ' menit)';
            }
            echo '</span>';
        } else {
            echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">On Time</span>';
        }
        echo '</div>';
        
        echo '</div>';
        echo '</div>';
    } else {
        echo '<div class="bg-yellow-50 rounded-lg p-4">';
        echo '<h4 class="text-lg font-semibold text-yellow-900 mb-2">Informasi Peminjaman</h4>';
        echo '<p class="text-yellow-800">Item ini dipinjam tetapi tidak ditemukan data transaksi aktif.</p>';
        echo '</div>';
    }
    
    echo '</div>';
}

function getReservationDetail($conn) {
    $reservation_number = $_POST['reservation_number'] ?? '';
    $filter_type = $_POST['filter_type'] ?? 'all';
    
    if (empty($reservation_number)) {
        echo '<div class="text-red-500 text-center p-4">Reservation number tidak valid</div>';
        return;
    }
    
    // Additional WHERE condition based on filter type
    $additional_where = '';
    if ($filter_type === 'overdue') {
        $additional_where = 'AND ti.status_waktu = "overdue"';
    }
    
    $query = "
        SELECT 
            st.id,
            st.reservation_number,
            st.tanggal_transaksi,
            st.tanggal_pengembalian,
            st.status_peminjaman,
            bu.nama_borrower,
            bu.lokasi_posisi,
            csi.kode_barang,
            csi.rack_no,
            csi.lokasi,
            csi.status as item_status,
            cc.component_name,
            mcp.code_color,
            b.nama_buyer,
            ti.status_waktu,
            ti.dikembalikan,
            TIMESTAMPDIFF(MINUTE, st.tanggal_pengembalian, NOW()) as overdue_minutes
        FROM stock_transactions st
        INNER JOIN borrower_users bu ON st.borrower_id = bu.id
        INNER JOIN transaction_items ti ON st.id = ti.transaction_id
        INNER JOIN color_stock_items csi ON ti.stock_item_id = csi.id
        INNER JOIN color_components cc ON csi.component_id = cc.id
        INNER JOIN master_color_panel mcp ON csi.color_id = mcp.id
        INNER JOIN buyers b ON mcp.buyer_id = b.id
        WHERE st.reservation_number = ? $additional_where
        ORDER BY cc.component_name, csi.kode_barang
    ";
    
    $stmt = $conn->prepare($query);
    $stmt->bind_param("s", $reservation_number);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        echo '<div class="text-center py-8">
                <i class="fas fa-search text-gray-400 text-4xl mb-4"></i>
                <p class="text-gray-500 text-lg">Tidak ada data untuk reservation ini</p>
              </div>';
        return;
    }
    
    $first_row = $result->fetch_assoc();
    $result->data_seek(0); // Reset pointer
    
    // Header Information
    echo '<div class="bg-blue-50 rounded-lg p-6 mb-6">';
    echo '<h4 class="text-xl font-semibold text-blue-900 mb-4">Detail Reservation: ' . htmlspecialchars($reservation_number) . '</h4>';
    echo '<div class="grid grid-cols-2 md:grid-cols-4 gap-4 text-sm">';
    echo '<div><span class="font-medium text-gray-600">Peminjam:</span><br>' . htmlspecialchars($first_row['nama_borrower']) . '</div>';
    echo '<div><span class="font-medium text-gray-600">Lokasi Posisi:</span><br>' . htmlspecialchars($first_row['lokasi_posisi']) . '</div>';
    echo '<div><span class="font-medium text-gray-600">Tanggal Pinjam:</span><br>' . ($first_row['tanggal_transaksi'] ? date('d/m/Y H:i', strtotime($first_row['tanggal_transaksi'])) : '-') . '</div>';
    echo '<div><span class="font-medium text-gray-600">Tanggal Kembali:</span><br>' . ($first_row['tanggal_pengembalian'] ? date('d/m/Y H:i', strtotime($first_row['tanggal_pengembalian'])) : '-') . '</div>';
    echo '</div>';
    echo '</div>';
    
    // Items Table
    echo '<div class="overflow-x-auto">';
    echo '<table class="min-w-full divide-y divide-gray-200">';
    echo '<thead class="bg-gray-50">';
    echo '<tr>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Kode Barang</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Buyer</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Color Code</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Component</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Rack No</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status Item</th>';
    echo '<th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status Waktu</th>';
    echo '</tr>';
    echo '</thead>';
    echo '<tbody class="bg-white divide-y divide-gray-200">';
    
    while ($row = $result->fetch_assoc()) {
        echo '<tr class="hover:bg-gray-50">';
        
        // Kode Barang - clickable for detail
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm font-mono font-medium text-gray-900">';
        echo '<button onclick="showBorrowedItemDetail(\'' . htmlspecialchars($row['kode_barang']) . '\')" class="text-blue-600 hover:text-blue-800 hover:underline">';
        echo htmlspecialchars($row['kode_barang']);
        echo '</button>';
        echo '</td>';
        
        // Buyer
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">';
        echo htmlspecialchars($row['nama_buyer']);
        echo '</td>';
        
        // Color Code
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">';
        echo htmlspecialchars($row['code_color']);
        echo '</td>';
        
        // Component
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm text-gray-900">';
        echo '<span class="bg-blue-50 px-2 py-1 rounded text-xs font-medium">';
        echo htmlspecialchars($row['component_name']);
        echo '</span>';
        echo '</td>';
        
        // Rack No
        echo '<td class="px-6 py-4 whitespace-nowrap text-sm font-mono text-gray-900">';
        echo htmlspecialchars($row['rack_no']);
        echo '</td>';
        
        // Status Item
        echo '<td class="px-6 py-4 whitespace-nowrap">';
        $status_class = $row['item_status'] === 'Dipinjam' ? 'bg-blue-100 text-blue-800' : 'bg-green-100 text-green-800';
        echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium ' . $status_class . '">';
        echo htmlspecialchars($row['item_status']);
        echo '</span>';
        echo '</td>';
        
        // Status Waktu
        echo '<td class="px-6 py-4 whitespace-nowrap">';
        if ($row['status_waktu'] === 'overdue') {
            $hours = floor($row['overdue_minutes'] / 60);
            $minutes = $row['overdue_minutes'] % 60;
            echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">';
            echo 'Overdue ';
            if ($hours > 0) {
                echo '(' . $hours . 'h ' . $minutes . 'm)';
            } else {
                echo '(' . $minutes . ' menit)';
            }
            echo '</span>';
        } else {
            echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">On Time</span>';
        }
        echo '</td>';
        
        echo '</tr>';
    }
    
    echo '</tbody>';
    echo '</table>';
    echo '</div>';
}
?>