<?php
// Start session and include database connection
session_start();
require_once "../config/connection.php";

// Check if user is admin
if (!is_logged_in() || !is_user_type('admin')) {
    http_response_code(403);
    echo '<div class="empty-state"><i class="fas fa-lock"></i><p>Access denied</p></div>';
    exit();
}

// Get the type of data requested
$type = isset($_GET['type']) ? $_GET['type'] : '';

switch ($type) {
    case 'ongoing':
        renderOngoingTransactions($conn);
        break;
    case 'dipinjam':
        renderDipinjamItems($conn);
        break;
    case 'overdue':
        renderOverdueTransactions($conn);
        break;
    case 'renew':
        renderRenewItems($conn);
        break;
    default:
        echo '<div class="empty-state"><i class="fas fa-question-circle"></i><p>Invalid request</p></div>';
        break;
}

function renderOngoingTransactions($conn) {
    $query = "
        SELECT 
            st.id,
            st.reservation_number,
            st.tanggal_transaksi,
            st.tanggal_pengembalian,
            st.status,
            st.status_peminjaman,
            bu.username as borrower_name,
            bu.nama_borrower,
            bu.nik,
            bu.lokasi_posisi,
            COUNT(ti.id) as total_items
        FROM stock_transactions_surabaya st
        JOIN borrower_users_surabaya bu ON st.borrower_id = bu.id
        LEFT JOIN transaction_items_surabaya ti ON st.id = ti.transaction_id
        WHERE st.status_peminjaman = 'Dipinjam'
        GROUP BY st.id
        ORDER BY st.tanggal_transaksi DESC
    ";
    
    $result = $conn->query($query);
    
    if ($result->num_rows > 0) {
        echo '<div class="overflow-x-auto">';
        echo '<table class="min-w-full bg-white border border-gray-200 rounded-lg">';
        echo '<thead class="bg-gray-50">';
        echo '<tr>';
        echo '<th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider border-b">No. Reservasi</th>';
        echo '<th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider border-b">Peminjam</th>';
        echo '<th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider border-b">Tanggal Pinjam</th>';
        echo '<th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider border-b">Batas Kembali</th>';
        echo '<th class="px-4 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider border-b">Total Item</th>';
        echo '<th class="px-4 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider border-b">Status</th>';
        echo '</tr>';
        echo '</thead>';
        echo '<tbody class="bg-white divide-y divide-gray-200">';
        
        while ($row = $result->fetch_assoc()) {
            $isOverdue = strtotime($row['tanggal_pengembalian']) + (15 * 60) < time();
            $rowClass = $isOverdue ? 'bg-red-50 hover:bg-red-100' : 'hover:bg-gray-50';
            
            echo '<tr class="' . $rowClass . ' cursor-pointer transition-colors duration-200" onclick="redirectToReservation(\'' . htmlspecialchars($row['reservation_number']) . '\')">';
            
            // Reservation Number
            echo '<td class="px-4 py-4 border-b">';
            echo '<div class="flex items-center">';
            if ($isOverdue) {
                echo '<span class="w-2 h-2 bg-red-500 rounded-full mr-2 animate-pulse"></span>';
            } else {
                echo '<span class="w-2 h-2 bg-green-500 rounded-full mr-2"></span>';
            }
            echo '<span class="font-medium text-blue-600 hover:text-blue-800">' . htmlspecialchars($row['reservation_number']) . '</span>';
            echo '</div>';
            echo '</td>';
            
            // Borrower Info
            echo '<td class="px-4 py-4 border-b">';
            echo '<div class="space-y-1">';
            echo '<div class="font-medium text-gray-900">' . htmlspecialchars($row['borrower_name']) . '</div>';
            echo '<div class="text-sm text-gray-600">' . htmlspecialchars($row['nama_borrower']) . '</div>';
            echo '<div class="text-xs text-gray-500">NIK: ' . htmlspecialchars($row['nik']) . '</div>';
            echo '<div class="text-xs bg-gray-100 text-gray-700 px-2 py-1 rounded inline-block">' . htmlspecialchars($row['lokasi_posisi']) . '</div>';
            echo '</div>';
            echo '</td>';
            
            // Transaction Date
            echo '<td class="px-4 py-4 border-b">';
            echo '<div class="space-y-1">';
            echo '<div class="font-medium text-gray-900">' . date('d M Y', strtotime($row['tanggal_transaksi'])) . '</div>';
            echo '<div class="text-sm text-gray-600">' . date('H:i', strtotime($row['tanggal_transaksi'])) . ' WIB</div>';
            echo '</div>';
            echo '</td>';
            
            // Return Date
            echo '<td class="px-4 py-4 border-b">';
            echo '<div class="space-y-1">';
            $dateClass = $isOverdue ? 'text-red-600 font-semibold' : 'text-gray-900';
            echo '<div class="font-medium ' . $dateClass . '">' . date('d M Y', strtotime($row['tanggal_pengembalian'])) . '</div>';
            echo '<div class="text-sm ' . ($isOverdue ? 'text-red-600' : 'text-gray-600') . '">' . date('H:i', strtotime($row['tanggal_pengembalian'])) . ' WIB</div>';
            if ($isOverdue) {
                $overdueMinutes = floor((time() - (strtotime($row['tanggal_pengembalian']) + (15 * 60))) / 60);
                echo '<div class="text-xs text-red-600 font-medium">Terlambat ' . $overdueMinutes . ' menit</div>';
            }
            echo '</div>';
            echo '</td>';
            
            // Total Items
            echo '<td class="px-4 py-4 border-b text-center">';
            echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800">';
            echo $row['total_items'] . ' item' . ($row['total_items'] > 1 ? 's' : '');
            echo '</span>';
            echo '</td>';
            
            // Status
            echo '<td class="px-4 py-4 border-b text-center">';
            if ($isOverdue) {
                echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">';
                echo '<i class="fas fa-exclamation-triangle mr-1"></i>Overdue';
            } else {
                echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">';
                echo '<i class="fas fa-clock mr-1"></i>On Going';
            }
            echo '</span>';
            echo '</td>';
            
            echo '</tr>';
        }
        
        echo '</tbody>';
        echo '</table>';
        echo '</div>';
    } else {
        echo '<div class="empty-state">';
        echo '<i class="fas fa-clock text-4xl text-gray-300 mb-4"></i>';
        echo '<p class="text-lg font-medium text-gray-600">Tidak ada transaksi ongoing</p>';
        echo '<p class="text-sm text-gray-500">Semua transaksi sudah diselesaikan</p>';
        echo '</div>';
    }
}

function renderDipinjamItems($conn) {
    $query = "
        SELECT 
            csi.id,
            csi.kode_barang,
            csi.stock_number,
            csi.status,
            csi.lokasi,
            csi.rack_no,
            csi.keterangan,
            mcp.code_color,
            mcp.deskripsi as color_description,
            cc.component_name,
            b.nama_buyer,
            b.kode_buyer,
            st.reservation_number,
            st.tanggal_pengembalian,
            bu.username as borrower_name,
            bu.nama_borrower,
            bu.nik,
            bu.lokasi_posisi
        FROM color_stock_items_surabaya csi
        JOIN master_color_panel_surabaya mcp ON csi.color_id = mcp.id
        JOIN color_components_surabaya cc ON csi.component_id = cc.id
        JOIN buyers_surabaya b ON mcp.buyer_id = b.id
        LEFT JOIN transaction_items_surabaya ti ON csi.id = ti.stock_item_id AND ti.dikembalikan = 0
        LEFT JOIN stock_transactions_surabaya st ON ti.transaction_id = st.id
        LEFT JOIN borrower_users_surabaya bu ON st.borrower_id = bu.id
        WHERE csi.status = 'Dipinjam'
        ORDER BY st.tanggal_pengembalian ASC, csi.kode_barang ASC
    ";
    
    $result = $conn->query($query);
    
    if ($result->num_rows > 0) {
        echo '<div class="overflow-x-auto">';
        echo '<table class="min-w-full bg-white border border-gray-200 rounded-lg">';
        echo '<thead class="bg-gray-50">';
        echo '<tr>';
        echo '<th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider border-b">Kode Barang</th>';
        echo '<th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider border-b">Color & Component</th>';
        echo '<th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider border-b">Peminjam</th>';
        echo '<th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider border-b">No. Reservasi</th>';
        echo '<th class="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider border-b">Batas Kembali</th>';
        echo '<th class="px-4 py-3 text-center text-xs font-medium text-gray-500 uppercase tracking-wider border-b">Status</th>';
        echo '</tr>';
        echo '</thead>';
        echo '<tbody class="bg-white divide-y divide-gray-200">';
        
        while ($row = $result->fetch_assoc()) {
            $isOverdue = $row['tanggal_pengembalian'] && strtotime($row['tanggal_pengembalian']) + (15 * 60) < time();
            $rowClass = $isOverdue ? 'bg-red-50' : '';
            
            echo '<tr class="' . $rowClass . ' hover:bg-gray-50 transition-colors duration-200">';
            
            // Kode Barang
            echo '<td class="px-4 py-4 border-b">';
            echo '<div class="space-y-1">';
            echo '<div class="font-mono font-bold text-gray-900">' . htmlspecialchars($row['kode_barang']) . '</div>';
            echo '<div class="text-xs text-gray-500">Stock #' . $row['stock_number'] . '</div>';
            if ($row['rack_no']) {
                echo '<div class="text-xs bg-gray-100 text-gray-700 px-2 py-1 rounded inline-block">Rack: ' . htmlspecialchars($row['rack_no']) . '</div>';
            }
            echo '</div>';
            echo '</td>';
            
            // Color & Component
            echo '<td class="px-4 py-4 border-b">';
            echo '<div class="space-y-1">';
            echo '<div class="font-medium text-gray-900">' . htmlspecialchars($row['kode_buyer']) . ' ' . htmlspecialchars($row['code_color']) . '</div>';
            echo '<div class="text-sm text-gray-600">' . htmlspecialchars($row['component_name']) . '</div>';
            echo '<div class="text-xs text-gray-500">' . htmlspecialchars($row['nama_buyer']) . '</div>';
            echo '<div class="text-xs bg-blue-100 text-blue-700 px-2 py-1 rounded inline-block">' . htmlspecialchars($row['lokasi']) . '</div>';
            echo '</div>';
            echo '</td>';
            
            // Borrower Info
            echo '<td class="px-4 py-4 border-b">';
            if ($row['borrower_name']) {
                echo '<div class="space-y-1">';
                echo '<div class="font-medium text-gray-900">' . htmlspecialchars($row['borrower_name']) . '</div>';
                echo '<div class="text-sm text-gray-600">' . htmlspecialchars($row['nama_borrower']) . '</div>';
                echo '<div class="text-xs text-gray-500">NIK: ' . htmlspecialchars($row['nik']) . '</div>';
                echo '<div class="text-xs bg-gray-100 text-gray-700 px-2 py-1 rounded inline-block">' . htmlspecialchars($row['lokasi_posisi']) . '</div>';
                echo '</div>';
            } else {
                echo '<span class="text-gray-400 italic">Data tidak tersedia</span>';
            }
            echo '</td>';
            
            // Reservation Number
            echo '<td class="px-4 py-4 border-b">';
            if ($row['reservation_number']) {
                echo '<span class="font-medium text-blue-600 hover:text-blue-800 cursor-pointer" onclick="redirectToReservation(\'' . htmlspecialchars($row['reservation_number']) . '\')">';
                echo htmlspecialchars($row['reservation_number']);
                echo '</span>';
            } else {
                echo '<span class="text-gray-400 italic">-</span>';
            }
            echo '</td>';
            
            // Return Date
            echo '<td class="px-4 py-4 border-b">';
            if ($row['tanggal_pengembalian']) {
                echo '<div class="space-y-1">';
                $dateClass = $isOverdue ? 'text-red-600 font-semibold' : 'text-gray-900';
                echo '<div class="font-medium ' . $dateClass . '">' . date('d M Y', strtotime($row['tanggal_pengembalian'])) . '</div>';
                echo '<div class="text-sm ' . ($isOverdue ? 'text-red-600' : 'text-gray-600') . '">' . date('H:i', strtotime($row['tanggal_pengembalian'])) . ' WIB</div>';
                if ($isOverdue) {
                    $overdueMinutes = floor((time() - (strtotime($row['tanggal_pengembalian']) + (15 * 60))) / 60);
                    echo '<div class="text-xs text-red-600 font-medium">Terlambat ' . $overdueMinutes . ' menit</div>';
                }
                echo '</div>';
            } else {
                echo '<span class="text-gray-400 italic">-</span>';
            }
            echo '</td>';
            
            // Status
            echo '<td class="px-4 py-4 border-b text-center">';
            if ($isOverdue) {
                echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">';
                echo '<i class="fas fa-exclamation-triangle mr-1"></i>Overdue';
            } else {
                echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800">';
                echo '<i class="fas fa-hand-holding mr-1"></i>Dipinjam';
            }
            echo '</span>';
            echo '</td>';
            
            echo '</tr>';
        }
        
        echo '</tbody>';
        echo '</table>';
        echo '</div>';
    } else {
        echo '<div class="empty-state">';
        echo '<i class="fas fa-hand-holding text-4xl text-gray-300 mb-4"></i>';
        echo '<p class="text-lg font-medium text-gray-600">Tidak ada code barang yang sedang dipinjam</p>';
        echo '<p class="text-sm text-gray-500">Semua barang sudah dikembalikan</p>';
        echo '</div>';
    }
}

function renderOverdueTransactions($conn) {
    $query = "
        SELECT 
            st.id,
            st.reservation_number,
            st.tanggal_transaksi,
            st.tanggal_pengembalian,
            st.status,
            st.status_peminjaman,
            bu.username as borrower_name,
            bu.nama_borrower,
            bu.nik,
            bu.lokasi_posisi,
            COUNT(ti.id) as total_items,
            TIMESTAMPDIFF(MINUTE, st.tanggal_pengembalian, NOW()) as overdue_minutes
        FROM stock_transactions_surabaya st
        JOIN borrower_users_surabaya bu ON st.borrower_id = bu.id
        LEFT JOIN transaction_items_surabaya ti ON st.id = ti.transaction_id
        WHERE st.status_peminjaman = 'Dipinjam' 
        AND TIMESTAMPDIFF(MINUTE, st.tanggal_pengembalian, NOW()) > 15
        GROUP BY st.id
        ORDER BY overdue_minutes DESC
    ";
    
    $result = $conn->query($query);
    
    if ($result->num_rows > 0) {
        echo '<div class="overflow-x-auto">';
        echo '<table class="min-w-full bg-white border border-gray-200 rounded-lg">';
        echo '<thead class="bg-red-50">';
        echo '<tr>';
        echo '<th class="px-4 py-3 text-left text-xs font-medium text-red-700 uppercase tracking-wider border-b">No. Reservasi</th>';
        echo '<th class="px-4 py-3 text-left text-xs font-medium text-red-700 uppercase tracking-wider border-b">Peminjam</th>';
        echo '<th class="px-4 py-3 text-left text-xs font-medium text-red-700 uppercase tracking-wider border-b">Tanggal Pinjam</th>';
        echo '<th class="px-4 py-3 text-left text-xs font-medium text-red-700 uppercase tracking-wider border-b">Batas Kembali</th>';
        echo '<th class="px-4 py-3 text-center text-xs font-medium text-red-700 uppercase tracking-wider border-b">Total Item</th>';
        echo '<th class="px-4 py-3 text-center text-xs font-medium text-red-700 uppercase tracking-wider border-b">Terlambat</th>';
        echo '</tr>';
        echo '</thead>';
        echo '<tbody class="bg-white divide-y divide-gray-200">';
        
        while ($row = $result->fetch_assoc()) {
            echo '<tr class="bg-red-50 hover:bg-red-100 cursor-pointer transition-colors duration-200" onclick="redirectToReservation(\'' . htmlspecialchars($row['reservation_number']) . '\')">';
            
            // Reservation Number
            echo '<td class="px-4 py-4 border-b">';
            echo '<div class="flex items-center">';
            echo '<span class="w-2 h-2 bg-red-500 rounded-full mr-2 animate-pulse"></span>';
            echo '<span class="font-medium text-red-600 hover:text-red-800">' . htmlspecialchars($row['reservation_number']) . '</span>';
            echo '</div>';
            echo '</td>';
            
            // Borrower Info
            echo '<td class="px-4 py-4 border-b">';
            echo '<div class="space-y-1">';
            echo '<div class="font-medium text-gray-900">' . htmlspecialchars($row['borrower_name']) . '</div>';
            echo '<div class="text-sm text-gray-600">' . htmlspecialchars($row['nama_borrower']) . '</div>';
            echo '<div class="text-xs text-gray-500">NIK: ' . htmlspecialchars($row['nik']) . '</div>';
            echo '<div class="text-xs bg-gray-100 text-gray-700 px-2 py-1 rounded inline-block">' . htmlspecialchars($row['lokasi_posisi']) . '</div>';
            echo '</div>';
            echo '</td>';
            
            // Transaction Date
            echo '<td class="px-4 py-4 border-b">';
            echo '<div class="space-y-1">';
            echo '<div class="font-medium text-gray-900">' . date('d M Y', strtotime($row['tanggal_transaksi'])) . '</div>';
            echo '<div class="text-sm text-gray-600">' . date('H:i', strtotime($row['tanggal_transaksi'])) . ' WIB</div>';
            echo '</div>';
            echo '</td>';
            
            // Return Date
            echo '<td class="px-4 py-4 border-b">';
            echo '<div class="space-y-1">';
            echo '<div class="font-medium text-red-600">' . date('d M Y', strtotime($row['tanggal_pengembalian'])) . '</div>';
            echo '<div class="text-sm text-red-600">' . date('H:i', strtotime($row['tanggal_pengembalian'])) . ' WIB</div>';
            echo '<div class="text-xs text-red-600 font-medium">Seharusnya sudah kembali</div>';
            echo '</div>';
            echo '</td>';
            
            // Total Items
            echo '<td class="px-4 py-4 border-b text-center">';
            echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">';
            echo $row['total_items'] . ' item' . ($row['total_items'] > 1 ? 's' : '');
            echo '</span>';
            echo '</td>';
            
            // Overdue Time
            echo '<td class="px-4 py-4 border-b text-center">';
            $overdueMinutes = $row['overdue_minutes'] - 15; // Subtract the 15-minute tolerance
            echo '<div class="space-y-1">';
            if ($overdueMinutes >= 1440) { // More than 24 hours
                $days = floor($overdueMinutes / 1440);
                $hours = floor(($overdueMinutes % 1440) / 60);
                echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">';
                echo '<i class="fas fa-clock mr-1"></i>' . $days . ' hari ' . $hours . ' jam';
                echo '</span>';
            } elseif ($overdueMinutes >= 60) { // More than 1 hour
                $hours = floor($overdueMinutes / 60);
                $minutes = $overdueMinutes % 60;
                echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">';
                echo '<i class="fas fa-clock mr-1"></i>' . $hours . ' jam ' . $minutes . ' menit';
                echo '</span>';
            } else {
                echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">';
                echo '<i class="fas fa-clock mr-1"></i>' . $overdueMinutes . ' menit';
                echo '</span>';
            }
            echo '<div class="text-xs text-red-600 mt-1">Dari batas waktu</div>';
            echo '</div>';
            echo '</td>';
            
            echo '</tr>';
        }
        
        echo '</tbody>';
        echo '</table>';
        echo '</div>';
    } else {
        echo '<div class="empty-state">';
        echo '<i class="fas fa-check-circle text-4xl text-green-300 mb-4"></i>';
        echo '<p class="text-lg font-medium text-gray-600">Tidak ada transaksi overdue</p>';
        echo '<p class="text-sm text-gray-500">Semua peminjaman masih dalam batas waktu atau sudah dikembalikan</p>';
        echo '</div>';
    }
}

function renderRenewItems($conn) {
    $query = "
        SELECT 
            csi.id,
            csi.kode_barang,
            csi.stock_number,
            csi.status,
            csi.lokasi,
            csi.rack_no,
            csi.tanggal_pembuatan,
            csi.tanggal_kedaluwarsa,
            csi.keterangan,
            mcp.code_color,
            mcp.deskripsi as color_description,
            cc.component_name,
            b.nama_buyer,
            b.kode_buyer
        FROM color_stock_items_surabaya csi
        JOIN master_color_panel_surabaya mcp ON csi.color_id = mcp.id
        JOIN color_components_surabaya cc ON csi.component_id = cc.id
        JOIN buyers_surabaya b ON mcp.buyer_id = b.id
        WHERE csi.status = 'Renew'
        ORDER BY csi.tanggal_kedaluwarsa ASC, csi.kode_barang ASC
    ";
    
    $result = $conn->query($query);
    
    if ($result->num_rows > 0) {
        echo '<div class="overflow-x-auto">';
        echo '<table class="min-w-full bg-white border border-gray-200 rounded-lg">';
        echo '<thead class="bg-yellow-50">';
        echo '<tr>';
        echo '<th class="px-4 py-3 text-left text-xs font-medium text-yellow-700 uppercase tracking-wider border-b">Kode Barang</th>';
        echo '<th class="px-4 py-3 text-left text-xs font-medium text-yellow-700 uppercase tracking-wider border-b">Color & Component</th>';
        echo '<th class="px-4 py-3 text-left text-xs font-medium text-yellow-700 uppercase tracking-wider border-b">Lokasi</th>';
        echo '<th class="px-4 py-3 text-left text-xs font-medium text-yellow-700 uppercase tracking-wider border-b">Tanggal Dibuat</th>';
        echo '<th class="px-4 py-3 text-left text-xs font-medium text-yellow-700 uppercase tracking-wider border-b">Tanggal Kadaluarsa</th>';
        echo '<th class="px-4 py-3 text-center text-xs font-medium text-yellow-700 uppercase tracking-wider border-b">Status</th>';
        echo '</tr>';
        echo '</thead>';
        echo '<tbody class="bg-white divide-y divide-gray-200">';
        
        while ($row = $result->fetch_assoc()) {
            $isExpired = $row['tanggal_kedaluwarsa'] && strtotime($row['tanggal_kedaluwarsa']) < time();
            $rowClass = $isExpired ? 'bg-red-50' : 'bg-yellow-50';
            
            echo '<tr class="' . $rowClass . ' hover:bg-yellow-100 transition-colors duration-200">';
            
            // Kode Barang
            echo '<td class="px-4 py-4 border-b">';
            echo '<div class="space-y-1">';
            echo '<div class="font-mono font-bold text-gray-900">' . htmlspecialchars($row['kode_barang']) . '</div>';
            echo '<div class="text-xs text-gray-500">Stock #' . $row['stock_number'] . '</div>';
            if ($row['keterangan']) {
                echo '<div class="text-xs text-gray-600 italic">' . htmlspecialchars(substr($row['keterangan'], 0, 30)) . '...</div>';
            }
            echo '</div>';
            echo '</td>';
            
            // Color & Component
            echo '<td class="px-4 py-4 border-b">';
            echo '<div class="space-y-1">';
            echo '<div class="font-medium text-gray-900">' . htmlspecialchars($row['kode_buyer']) . ' ' . htmlspecialchars($row['code_color']) . '</div>';
            echo '<div class="text-sm text-gray-600">' . htmlspecialchars($row['component_name']) . '</div>';
            echo '<div class="text-xs text-gray-500">' . htmlspecialchars($row['nama_buyer']) . '</div>';
            echo '</div>';
            echo '</td>';
            
            // Lokasi
            echo '<td class="px-4 py-4 border-b">';
            echo '<div class="space-y-1">';
            echo '<div class="font-medium text-gray-900">' . htmlspecialchars($row['lokasi']) . '</div>';
            if ($row['rack_no']) {
                echo '<div class="text-sm bg-blue-100 text-blue-700 px-2 py-1 rounded inline-block">Rack: ' . htmlspecialchars($row['rack_no']) . '</div>';
            }
            echo '</div>';
            echo '</td>';
            
            // Tanggal Pembuatan
            echo '<td class="px-4 py-4 border-b">';
            if ($row['tanggal_pembuatan']) {
                echo '<div class="space-y-1">';
                echo '<div class="font-medium text-gray-900">' . date('d M Y', strtotime($row['tanggal_pembuatan'])) . '</div>';
                $daysSinceCreated = floor((time() - strtotime($row['tanggal_pembuatan'])) / (60 * 60 * 24));
                echo '<div class="text-xs text-gray-500">' . $daysSinceCreated . ' hari yang lalu</div>';
                echo '</div>';
            } else {
                echo '<span class="text-gray-400 italic">Tidak tersedia</span>';
            }
            echo '</td>';
            
            // Tanggal Kedaluwarsa
            echo '<td class="px-4 py-4 border-b">';
            if ($row['tanggal_kedaluwarsa']) {
                echo '<div class="space-y-1">';
                $dateClass = $isExpired ? 'text-red-600 font-semibold' : 'text-gray-900';
                echo '<div class="font-medium ' . $dateClass . '">' . date('d M Y', strtotime($row['tanggal_kedaluwarsa'])) . '</div>';
                if ($isExpired) {
                    $daysExpired = floor((time() - strtotime($row['tanggal_kedaluwarsa'])) / (60 * 60 * 24));
                    echo '<div class="text-xs text-red-600 font-medium">Expired ' . $daysExpired . ' hari</div>';
                } else {
                    $daysToExpiry = floor((strtotime($row['tanggal_kedaluwarsa']) - time()) / (60 * 60 * 24));
                    if ($daysToExpiry <= 7) {
                        echo '<div class="text-xs text-yellow-600 font-medium">Akan expired ' . $daysToExpiry . ' hari</div>';
                    } else {
                        echo '<div class="text-xs text-gray-500">' . $daysToExpiry . ' hari lagi</div>';
                    }
                }
                echo '</div>';
            } else {
                echo '<span class="text-gray-400 italic">Tidak tersedia</span>';
            }
            echo '</td>';
            
            // Status
            echo '<td class="px-4 py-4 border-b text-center">';
            if ($isExpired) {
                echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800">';
                echo '<i class="fas fa-times-circle mr-1"></i>Expired';
            } else {
                echo '<span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800">';
                echo '<i class="fas fa-redo mr-1"></i>Perlu Renew';
            }
            echo '</span>';
            echo '</td>';
            
            echo '</tr>';
        }
        
        echo '</tbody>';
        echo '</table>';
        echo '</div>';
    } else {
        echo '<div class="empty-state">';
        echo '<i class="fas fa-redo text-4xl text-gray-300 mb-4"></i>';
        echo '<p class="text-lg font-medium text-gray-600">Tidak ada code barang yang perlu diperbaharui</p>';
        echo '<p class="text-sm text-gray-500">Semua barang dalam kondisi baik</p>';
        echo '</div>';
    }
}
?>