@extends('layouts.app')

@section('title', 'Create Invoice BOM - Cost Furniture')

@section('page-title', 'Create Invoice BOM')

@section('content')

<!-- Step Indicator -->
<div class="step-indicator mb-4">
    <div class="step active" id="step-indicator-1">
        <div class="step-number">1</div>
        <div class="step-text">Initial Form</div>
    </div>
    <div class="step-line"></div>
    <div class="step" id="step-indicator-2">
        <div class="step-number">2</div>
        <div class="step-text">Calculator</div>
    </div>
</div>

<!-- STEP 1: Initial Form -->
<div class="card modern-card" id="step-1-card">
    <div class="card-header">
        <h5><i class="fas fa-file-alt me-2"></i>Step 1: Informasi Product & Material Locking</h5>
    </div>
    <div class="card-body">
        <form id="initialForm">
            <div class="row">
                <!-- Left Column -->
                <div class="col-md-6">
                    <h6 class="section-title">INFORMASI PRODUCT</h6>
                    
                    <div class="mb-3">
                        <label class="form-label">Product Name (Buyer) <span class="text-danger">*</span></label>
                        <select name="product_name" id="product_name" class="form-select" required>
                            <option value="">-- Pilih Buyer --</option>
                            <option value="Ethan Allen">Ethan Allen</option>
                            <option value="Uttermost">Uttermost</option>
                            <option value="Crate&Barrel">Crate&Barrel</option>
                            <option value="Century">Century</option>
                            <option value="Vanguard">Vanguard</option>
                            <option value="Hickory">Hickory</option>
                            <option value="Brunswick">Brunswick</option>
                        </select>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Product <span class="text-danger">*</span></label>
                        <select name="product_category" id="product_category" class="form-select" required>
                            <option value="">-- Pilih Product --</option>
                            <option value="Casegoods">Casegoods</option>
                            <option value="Occasional Table">Occasional Table</option>
                            <option value="Dining Table">Dining Table</option>
                            <option value="Bed">Bed</option>
                            <option value="Chair">Chair</option>
                        </select>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Product Type <span class="text-danger">*</span></label>
                        <input type="text" name="product_type" id="product_type" class="form-control" required placeholder="Misal: Cabinet, Table, Chair">
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Product Code <span class="text-danger">*</span></label>
                        <input type="text" name="product_code" id="product_code" class="form-control" required placeholder="Misal: EA-001">
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Size <span class="text-danger">*</span></label>
                        <input type="text" name="size" id="size" class="form-control" required placeholder="Misal: 120x60x75 cm">
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Date <span class="text-danger">*</span></label>
                        <input type="date" name="invoice_date" id="invoice_date" class="form-control" required>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Order (Pcs) <span class="text-danger">*</span></label>
                        <input type="number" name="order_qty" id="order_qty" class="form-control" min="1" required placeholder="Jumlah Order">
                    </div>

                    <!-- ✅ IMAGE OPTIONAL -->
                    <div class="mb-3">
                        <label class="form-label">Upload Gambar Product <span class="text-muted">(Opsional)</span></label>
                        <input type="file" name="image" id="image" class="form-control" accept="image/*">
                        <small class="text-muted">Format: JPG, PNG, WEBP - Max 11MB. Bisa diupload nanti saat edit.</small>
                        
                        <!-- Image Preview -->
                        <div id="imagePreview" style="display: none; margin-top: 10px;">
                            <img id="previewImg" src="" alt="Preview" style="max-width: 200px; max-height: 200px; border-radius: 8px; border: 2px solid #ddd;">
                            <button type="button" class="btn btn-sm btn-danger mt-2" onclick="clearImagePreview()">
                                <i class="fas fa-times"></i> Hapus
                            </button>
                        </div>
                    </div>
                </div>

                <!-- Right Column -->
                <div class="col-md-6">
                    <h6 class="section-title">MATERIAL LOCKING</h6>
                    <p class="text-muted small">Material yang dipilih akan mengunci semua komponen di calculator</p>

                    <div class="mb-3">
                        <label class="form-label">Panel <span class="text-danger">*</span></label>
                        <select name="panel_type" id="panel_type" class="form-select" required>
                            <option value="">-- Pilih Panel Type --</option>
                            @foreach($panelTypes as $type)
                            <option value="{{ $type }}">{{ $type }}</option>
                            @endforeach
                        </select>
                        <small class="text-muted">Akan digunakan untuk: Top, Side, Bottom, Back Panel</small>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Solid <span class="text-danger">*</span></label>
                        <select name="solid_type" id="solid_type" class="form-select" required>
                            <option value="">-- Pilih Solid Type --</option>
                            @foreach($solidTypes as $type)
                            <option value="{{ $type }}">{{ $type }}</option>
                            @endforeach
                        </select>
                        <small class="text-muted">Akan digunakan untuk: Patching Solid, Frame</small>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Veneer Face <span class="text-danger">*</span></label>
                        <select name="veneer_face_type" id="veneer_face_type" class="form-select" required>
                            <option value="">-- Pilih Veneer Face Type --</option>
                            @foreach($veneerTypes as $type)
                            <option value="{{ $type }}">{{ $type }}</option>
                            @endforeach
                        </select>
                        <small class="text-muted">Veneer untuk sisi depan (Face)</small>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Veneer Back <span class="text-danger">*</span></label>
                        <select name="veneer_back_type" id="veneer_back_type" class="form-select" required>
                            <option value="">-- Pilih Veneer Back Type --</option>
                            @foreach($veneerTypes as $type)
                            <option value="{{ $type }}">{{ $type }}</option>
                            @endforeach
                        </select>
                        <small class="text-muted">Veneer untuk sisi belakang (Back)</small>
                    </div>

                    <div class="mb-3">
                        <label class="form-label">Box Drawer <span class="text-danger">*</span></label>
                        <select name="box_drawer_type" id="box_drawer_type" class="form-select" required>
                            <option value="">-- Pilih Box Drawer Material --</option>
                            <option value="Mahoni">Mahoni</option>
                            <option value="White Oak">White Oak</option>
                        </select>
                        <small class="text-muted">Material untuk semua drawer</small>
                    </div>

                    <div class="alert alert-info">
                        <i class="fas fa-info-circle me-2"></i>
                        <strong>Penting:</strong> Material yang dipilih tidak bisa diubah setelah masuk ke calculator. Pastikan pilihan sudah benar!
                    </div>
                </div>
            </div>

            <hr>

            <div class="d-flex gap-2 justify-content-end">
                <button type="button" class="btn btn-warning" onclick="saveStep1Draft()">
                    <i class="fas fa-save me-2"></i>Simpan Draft
                </button>
                <button type="button" class="btn btn-primary" onclick="goToCalculator()">
                    Lanjut ke Kalkulator <i class="fas fa-arrow-right ms-2"></i>
                </button>
            </div>
        </form>
    </div>
</div>
<!-- STEP 2: Calculator -->
<div class="card modern-card" id="step-2-card" style="display: none;">
    <div class="card-header d-flex justify-content-between align-items-center">
        <h5 class="mb-0"><i class="fas fa-calculator me-2"></i>Step 2: Calculator Components</h5>
        <div>
            <button type="button" class="btn btn-sm btn-danger me-2" onclick="clearAllDrafts()">
                <i class="fas fa-trash me-2"></i>Clear All Drafts
            </button>
            <button type="button" class="btn btn-sm btn-secondary" onclick="backToStep1()">
                <i class="fas fa-arrow-left me-2"></i>Kembali ke Step 1
            </button>
        </div>
    </div>
    <div class="card-body">
        <!-- Summary Info Bar -->
        <div class="summary-info-bar mb-4">
            <div class="summary-item">
                <div class="summary-label">Product</div>
                <div class="summary-value" id="summary-product">-</div>
            </div>
            <div class="summary-item">
                <div class="summary-label">Type</div>
                <div class="summary-value" id="summary-type">-</div>
            </div>
            <div class="summary-item">
                <div class="summary-label">Code</div>
                <div class="summary-value" id="summary-code">-</div>
            </div>
            <div class="summary-item">
                <div class="summary-label">Grand Total</div>
                <div class="summary-value grand-total" id="summary-grand-total">Rp 0</div>
            </div>
        </div>

        <!-- ✅ Components Grid Layout (Card-based) WITH M³ DISPLAY -->
        <div class="components-grid">
            @php
                $components = [
                    ['index' => 0, 'name' => 'Top Panel', 'icon' => 'fa-layer-group', 'color' => '#3498db'],
                    ['index' => 1, 'name' => 'Side Panel L', 'icon' => 'fa-layer-group', 'color' => '#9b59b6'],
                    ['index' => 2, 'name' => 'Side Panel R', 'icon' => 'fa-layer-group', 'color' => '#9b59b6'],
                    ['index' => 3, 'name' => 'Bottom Panel', 'icon' => 'fa-layer-group', 'color' => '#e74c3c'],
                    ['index' => 4, 'name' => 'Back Panel', 'icon' => 'fa-layer-group', 'color' => '#f39c12'],
                    ['index' => 5, 'name' => 'Front Frame', 'icon' => 'fa-border-all', 'color' => '#1abc9c'],
                    ['index' => 6, 'name' => 'Back Frame', 'icon' => 'fa-border-all', 'color' => '#16a085'],
                    ['index' => 7, 'name' => 'Face Drawer', 'icon' => 'fa-box-open', 'color' => '#2ecc71'],
                    ['index' => 8, 'name' => 'Front Drawer', 'icon' => 'fa-box', 'color' => '#27ae60'],
                    ['index' => 9, 'name' => 'Side Drawer L', 'icon' => 'fa-box', 'color' => '#8e44ad'],
                    ['index' => 10, 'name' => 'Side Drawer R', 'icon' => 'fa-box', 'color' => '#8e44ad'],
                    ['index' => 11, 'name' => 'Back Drawer', 'icon' => 'fa-box', 'color' => '#c0392b'],
                    ['index' => 12, 'name' => 'Base Drawer', 'icon' => 'fa-box', 'color' => '#d35400'],
                    ['index' => 13, 'name' => 'Support Drawer', 'icon' => 'fa-tools', 'color' => '#7f8c8d'],
                    ['index' => 14, 'name' => 'Cleat Drawer', 'icon' => 'fa-tools', 'color' => '#95a5a6'],
                ];
            @endphp

            @foreach($components as $component)
            <!-- ✅ Component Card (Clickable) WITH M³ -->
            <div class="component-grid-card" onclick="openComponentModal({{ $component['index'] }})" style="border-left-color: {{ $component['color'] }}">
                <div class="component-card-icon" style="background: {{ $component['color'] }}20; color: {{ $component['color'] }}">
                    <i class="fas {{ $component['icon'] }} fa-2x"></i>
                </div>
                <div class="component-card-content">
                    <h6 class="component-card-title">{{ $component['name'] }}</h6>
                    <span class="component-card-status" id="card-status-{{ $component['index'] }}">
                        <i class="fas fa-circle text-secondary"></i> Belum diisi
                    </span>
                    <div class="component-card-total" id="card-total-{{ $component['index'] }}">Rp 0</div>
                    <!-- ✅ NEW: M³ DISPLAY PER COMPONENT -->
                    <div class="component-card-m3" id="card-m3-{{ $component['index'] }}" style="font-size: 11px; color: #666; margin-top: 3px;">
                        <i class="fas fa-cube" style="font-size: 10px;"></i> 0.000000 m³
                    </div>
                </div>
                <div class="component-card-arrow">
                    <i class="fas fa-chevron-right"></i>
                </div>
            </div>
            @endforeach
        </div>

        <hr class="my-4">

        <!-- ✅ NEW: BIAYA PRODUKSI M³ SECTION -->
        <div class="card border-success mb-4">
            <div class="card-header bg-success text-white">
                <h5 class="mb-0"><i class="fas fa-cube me-2"></i>Biaya Produksi (M³)</h5>
            </div>
            <div class="card-body">
                <div class="alert alert-info mb-3">
                    <i class="fas fa-info-circle me-2"></i>
                    <strong>Formula:</strong> Total M³ × Harga per M³ = Biaya Produksi
                </div>

                <div class="row g-3 mb-3">
                    <div class="col-md-4">
                        <label class="form-label fw-bold">Total M³ (Auto-calculated)</label>
                        <div class="input-group">
                            <input type="text" class="form-control form-control-lg bg-light fw-bold text-primary" 
                                   id="total-m3-display" value="0.000000" readonly>
                            <span class="input-group-text">m³</span>
                        </div>
                        <small class="text-muted">Sum dari semua komponen</small>
                    </div>

                    <div class="col-md-4">
                        <label class="form-label fw-bold">Harga per M³</label>
                        <select class="form-select form-select-lg" id="harga-per-m3-select" onchange="onHargaPerM3Change()">
                            <option value="3500000">Rp 3.500.000</option>
                            <option value="4000000">Rp 4.000.000</option>
                            <option value="4500000">Rp 4.500.000</option>
                            <option value="5000000">Rp 5.000.000</option>
                            <option value="5500000">Rp 5.500.000</option>
                            <option value="6000000" selected>Rp 6.000.000</option>
                            <option value="6500000">Rp 6.500.000</option>
                            <option value="7000000">Rp 7.000.000</option>
                            <option value="7500000">Rp 7.500.000</option>
                            <option value="8000000">Rp 8.000.000</option>
                        </select>
                        <small class="text-muted">Pilih harga per M³</small>
                    </div>

                    <div class="col-md-4">
                        <label class="form-label fw-bold">Biaya Produksi</label>
                        <div class="form-control form-control-lg bg-success text-white fw-bold" id="biaya-produksi-display">
                            Rp 0
                        </div>
                        <small class="text-muted">Total M³ × Harga per M³</small>
                    </div>
                </div>

                <hr>

                <!-- GRAND TOTAL SUMMARY -->
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label class="form-label text-muted">Total Material (Components):</label>
                            <h4 class="mb-0 text-primary" id="total-components-display">Rp 0</h4>
                        </div>
                    </div>
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label class="form-label text-muted">Biaya Produksi (M³):</label>
                            <h4 class="mb-0 text-success" id="biaya-produksi-summary">Rp 0</h4>
                        </div>
                    </div>
                </div>

                <hr>

                <div class="row">
                    <div class="col-md-12">
                        <div class="text-center p-3 bg-light rounded">
                            <label class="form-label text-muted mb-2">GRAND TOTAL (Components + Biaya Produksi):</label>
                            <h2 class="mb-0 text-success fw-bold" id="grand-total-final-display">Rp 0</h2>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Action Buttons -->
        <div class="d-flex gap-2 justify-content-end">
            <button type="button" class="btn btn-secondary" onclick="backToStep1()">
                <i class="fas fa-arrow-left me-2"></i>Kembali ke Step 1
            </button>
            <button type="button" class="btn btn-success btn-lg" onclick="finalizeInvoice()">
                <i class="fas fa-check me-2"></i>Simpan Invoice
            </button>
        </div>
    </div>
</div>

<!-- ✅ MODAL untuk Input Component (Reusable) -->
<div class="modal fade" id="componentModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-xl modal-dialog-scrollable">
        <div class="modal-content">
            <div class="modal-header" id="modalHeader">
                <h5 class="modal-title" id="modalTitle">
                    <i class="fas fa-layer-group me-2"></i>Component Name
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body" id="modalBody">
                <!-- Content will be loaded dynamically -->
                <div class="text-center p-5">
                    <i class="fas fa-spinner fa-spin fa-3x text-primary mb-3"></i>
                    <p>Loading component...</p>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                    <i class="fas fa-times me-2"></i>Tutup
                </button>
                <button type="button" class="btn btn-warning" id="saveDraftModalBtn">
                    <i class="fas fa-save me-2"></i>Simpan Draft
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Loading Overlay -->
<div class="loading-overlay" id="loadingOverlay" style="display: none;">
    <div class="spinner-border text-light" role="status">
        <span class="visually-hidden">Loading...</span>
    </div>
    <p class="text-light mt-3">Processing invoice...</p>
</div>

@endsection
@push('styles')
<style>
/* ============================================
   MODERN CARD STYLES
   ============================================ */
.modern-card {
    border: none;
    border-radius: 15px;
    box-shadow: 0 5px 20px rgba(0,0,0,0.05);
    overflow: hidden;
}

.modern-card .card-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border-radius: 15px 15px 0 0;
    padding: 20px 25px;
    border: none;
}

.modern-card .card-header h5 {
    margin: 0;
    font-weight: 700;
}

.modern-card .card-body {
    padding: 30px;
}

/* Section Title */
.section-title {
    font-weight: 700;
    color: #667eea;
    margin-bottom: 20px;
    padding-bottom: 10px;
    border-bottom: 2px solid #f0f0f0;
    font-size: 16px;
}

/* Form Controls */
.form-label {
    font-weight: 600;
    color: #2c3e50;
    margin-bottom: 8px;
    font-size: 14px;
}

.form-control, .form-select {
    padding: 12px 15px;
    border: 2px solid #e0e0e0;
    border-radius: 10px;
    font-size: 14px;
    transition: all 0.3s ease;
}

.form-control:focus, .form-select:focus {
    border-color: #667eea;
    box-shadow: 0 0 0 0.2rem rgba(102, 126, 234, 0.1);
    outline: none;
}

/* Step Indicator */
.step-indicator {
    display: flex;
    align-items: center;
    justify-content: center;
    background: white;
    padding: 25px;
    border-radius: 15px;
    box-shadow: 0 2px 10px rgba(0,0,0,0.05);
}

.step {
    display: flex;
    flex-direction: column;
    align-items: center;
    position: relative;
}

.step-number {
    width: 50px;
    height: 50px;
    border-radius: 50%;
    background: #e0e0e0;
    color: #999;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    font-size: 20px;
    margin-bottom: 10px;
    transition: all 0.3s ease;
}

.step.active .step-number {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    box-shadow: 0 4px 15px rgba(102, 126, 234, 0.3);
}

.step-text {
    font-weight: 600;
    color: #999;
    transition: all 0.3s ease;
    font-size: 14px;
}

.step.active .step-text {
    color: #667eea;
}

.step-line {
    width: 100px;
    height: 3px;
    background: #e0e0e0;
    margin: 0 20px 30px 20px;
}

/* Summary Info Bar */
.summary-info-bar {
    display: flex;
    gap: 20px;
    padding: 20px;
    background: linear-gradient(135deg, rgba(102, 126, 234, 0.1) 0%, rgba(118, 75, 162, 0.1) 100%);
    border-radius: 12px;
    border: 2px solid rgba(102, 126, 234, 0.2);
}

.summary-item {
    flex: 1;
}

.summary-label {
    font-size: 12px;
    color: #666;
    margin-bottom: 5px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.summary-value {
    font-size: 16px;
    font-weight: bold;
    color: #2c3e50;
}

.summary-value.grand-total {
    font-size: 24px;
    color: #667eea;
}

/* ============================================
   COMPONENT GRID CARDS
   ============================================ */
.components-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(280px, 1fr));
    gap: 20px;
    margin-bottom: 30px;
}

.component-grid-card {
    background: white;
    border: 2px solid #e9ecef;
    border-left: 5px solid #667eea;
    border-radius: 12px;
    padding: 20px;
    display: flex;
    align-items: center;
    gap: 15px;
    cursor: pointer;
    transition: all 0.3s ease;
    position: relative;
}

.component-grid-card:hover {
    transform: translateY(-5px);
    box-shadow: 0 10px 30px rgba(0,0,0,0.1);
    border-color: #667eea;
}

.component-card-icon {
    width: 60px;
    height: 60px;
    border-radius: 12px;
    display: flex;
    align-items: center;
    justify-content: center;
    flex-shrink: 0;
}

.component-card-content {
    flex: 1;
}

.component-card-title {
    font-size: 15px;
    font-weight: 700;
    color: #2c3e50;
    margin: 0 0 5px 0;
}

.component-card-status {
    font-size: 12px;
    color: #666;
    display: block;
    margin-bottom: 8px;
}

.component-card-status i {
    font-size: 8px;
    margin-right: 5px;
}

.component-card-total {
    font-size: 16px;
    font-weight: bold;
    color: #27ae60;
}

/* ✅ NEW: M³ Display in Card */
.component-card-m3 {
    font-size: 11px;
    color: #666;
    margin-top: 3px;
    font-weight: 500;
}

.component-card-arrow {
    color: #ccc;
    font-size: 20px;
    transition: all 0.3s ease;
}

.component-grid-card:hover .component-card-arrow {
    color: #667eea;
    transform: translateX(5px);
}

/* ============================================
   MODAL STYLES
   ============================================ */
.modal-xl {
    max-width: 1200px;
}

.modal-header {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    color: white;
    border: none;
    padding: 20px 25px;
}

.modal-header .modal-title {
    font-weight: 700;
    color: white;
}

.modal-header .btn-close {
    filter: brightness(0) invert(1);
}

.modal-body {
    padding: 30px;
    max-height: 70vh;
    overflow-y: auto;
}

.modal-footer {
    border-top: 2px solid #f0f0f0;
    padding: 20px 25px;
}

/* Sub Component Card (Inside Modal) */
.sub-component-card {
    background: white;
    border: 2px solid #e9ecef;
    border-left: 4px solid #667eea;
    border-radius: 8px;
    padding: 15px;
    margin-bottom: 15px;
    transition: all 0.3s ease;
}

.sub-component-card:hover {
    box-shadow: 0 2px 8px rgba(0,0,0,0.05);
}

.sub-component-title {
    font-size: 14px;
    font-weight: 700;
    color: #495057;
    margin-bottom: 15px;
}

.calculated-info {
    padding: 10px 15px;
    background: #f8f9fa;
    border-radius: 6px;
    font-size: 13px;
    border-left: 3px solid #667eea;
}

.total-component {
    background: linear-gradient(135deg, rgba(102, 126, 234, 0.1) 0%, rgba(118, 75, 162, 0.1) 100%);
    padding: 15px 20px;
    border-radius: 8px;
    border: 2px solid rgba(102, 126, 234, 0.2);
}

/* Alert Styles */
.alert-sm {
    padding: 8px 12px;
    font-size: 13px;
    margin-bottom: 15px;
}

/* Loading Overlay */
.loading-overlay {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.7);
    display: flex;
    flex-direction: column;
    justify-content: center;
    align-items: center;
    z-index: 9999;
}

/* Buttons */
.btn {
    padding: 12px 24px;
    border-radius: 8px;
    font-weight: 600;
    transition: all 0.3s ease;
}

.btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
}

.btn-primary {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border: none;
}

.btn-primary:hover {
    background: linear-gradient(135deg, #764ba2 0%, #667eea 100%);
}

.btn-sm {
    padding: 6px 12px;
    font-size: 13px;
}

/* ============================================
   RESPONSIVE
   ============================================ */
@media (max-width: 768px) {
    .components-grid {
        grid-template-columns: 1fr;
    }
    
    .summary-info-bar {
        flex-direction: column;
    }
    
    .modern-card .card-body {
        padding: 20px;
    }
}

/* ============================================
   SCROLLBAR CUSTOMIZATION
   ============================================ */
.modal-body::-webkit-scrollbar {
    width: 8px;
}

.modal-body::-webkit-scrollbar-track {
    background: #f1f1f1;
    border-radius: 10px;
}

.modal-body::-webkit-scrollbar-thumb {
    background: #667eea;
    border-radius: 10px;
}

.modal-body::-webkit-scrollbar-thumb:hover {
    background: #764ba2;
}
</style>
@endpush
@push('scripts')
<script>
// ============================================
// GLOBAL VARIABLES
// ============================================

const productData = {
    panels: @json($panels),
    solids: @json($solids),
    veneers: @json($veneers),
    drawers: @json($drawers),
    materials: @json($materials ?? [])
};

console.log('Product Data:', productData);

let initialFormData = {};
let calculationData = {};
let grandTotal = 0;
let componentDrafts = {};
let faceDrawerQty = 1;
let sidePanelLData = null;
let sideDrawerLData = null;
let currentStep = 1;
let currentComponentIndex = null;

// ✅ NEW: M³ VARIABLES
let totalM3 = 0;
let hargaPerM3 = 6000000; // Default Rp 6.000.000
let biayaProduksi = 0;

// ============================================
// HELPER FUNCTIONS
// ============================================

function calculateM2(width, length, qty = 1) {
    return ((width * length) / 1000000) * qty;
}

// ✅ NEW: M³ CALCULATION FUNCTION
function calculateM3(width, length, thickness, qty = 1) {
    // Formula: P × L × T / 1,000,000,000
    return ((width * length * thickness) / 1000000000) * qty;
}

function formatRupiah(amount) {
    return 'Rp ' + Math.round(amount).toLocaleString('id-ID');
}

function getPanelsByType(type) {
    return productData.panels.filter(p => p.panel_type === type);
}

function getSolidsByType(type) {
    return productData.solids.filter(s => s.solid_type === type);
}

function getVeneersByType(type) {
    return productData.veneers.filter(v => v.veneer_type === type);
}

function getAllVeneers() {
    return productData.veneers;
}

function getFrontBackDrawers(material) {
    return productData.drawers.filter(d => 
        d.material_type === material && 
        d.drawer_type === 'front drawer'
    );
}

function getSideDrawers(material) {
    return productData.drawers.filter(d => 
        d.material_type === material && 
        d.drawer_type === 'side drawer'
    );
}

function getBaseDrawers(material) {
    return productData.drawers.filter(d => 
        d.material_type === material && 
        d.drawer_type === 'base drawer'
    );
}

function getCoreThickness(totalThickness) {
    const thicknessMap = {
        18: 15, 21: 18, 28: 25,
        16: 15, 19: 18, 26: 25,
        6: 5
    };
    return thicknessMap[totalThickness] || 0;
}

function getComponentName(index) {
    const names = [
        'Top Panel', 'Side Panel L', 'Side Panel R', 'Bottom Panel', 'Back Panel',
        'Front Frame', 'Back Frame', 'Face Drawer', 'Front Drawer',
        'Side Drawer L', 'Side Drawer R', 'Back Drawer', 'Base Drawer',
        'Support Drawer', 'Cleat Drawer'
    ];
    return names[index] || 'Unknown Component';
}

// ✅ FIXED: getCalculationDataKey - Face Drawer key fixed
function getCalculationDataKey(index) {
    const keys = [
        'top-panel-0', 'sidel-panel-1', 'sider-panel-2', 'bottom-panel-3', 'back-panel-4',
        'front-frame-5', 'back-frame-6', 'facedrawer-7', 'frontdrawer-8', // ✅ FIXED: facedrawer-7
        'sidedrawerl-9', 'sidedrawerr-10', 'backdrawer-11', 'basedrawer-12',
        'support-drawer-13', 'cleat-drawer-14'
    ];
    return keys[index];
}

// ============================================
// ✅ NEW: M³ & BIAYA PRODUKSI UPDATE FUNCTIONS
// ============================================

function updateTotalM3AndBiayaProduksi() {
    console.log('=== UPDATE TOTAL M³ & BIAYA PRODUKSI ===');
    
    // Calculate total M³ from all components
    totalM3 = 0;
    Object.keys(calculationData).forEach(key => {
        if (calculationData[key].m3) {
            totalM3 += calculationData[key].m3;
        }
    });
    
    // Calculate biaya produksi
    biayaProduksi = totalM3 * hargaPerM3;
    
    // Update displays
    document.getElementById('total-m3-display').value = totalM3.toFixed(6);
    document.getElementById('biaya-produksi-display').textContent = formatRupiah(biayaProduksi);
    document.getElementById('biaya-produksi-summary').textContent = formatRupiah(biayaProduksi);
    
    // Save to sessionStorage
    sessionStorage.setItem('bom_total_m3', totalM3);
    sessionStorage.setItem('bom_harga_per_m3', hargaPerM3);
    sessionStorage.setItem('bom_biaya_produksi', biayaProduksi);
    
    console.log('Total M³:', totalM3);
    console.log('Harga per M³:', formatRupiah(hargaPerM3));
    console.log('Biaya Produksi:', formatRupiah(biayaProduksi));
    
    // Trigger grand total update
    updateGrandTotal();
}

function onHargaPerM3Change() {
    const select = document.getElementById('harga-per-m3-select');
    hargaPerM3 = parseFloat(select.value);
    
    console.log('Harga per M³ changed to:', formatRupiah(hargaPerM3));
    
    // Recalculate biaya produksi
    updateTotalM3AndBiayaProduksi();
}

// ============================================
// ✅ UPDATED: GRAND TOTAL WITH BIAYA PRODUKSI
// ============================================

function updateGrandTotal() {
    // Calculate components total
    grandTotal = 0;
    Object.keys(calculationData).forEach(key => {
        if (calculationData[key].componentTotal) {
            grandTotal += calculationData[key].componentTotal;
        }
    });
    
    // Update components total display
    document.getElementById('total-components-display').textContent = formatRupiah(grandTotal);
    
    // Calculate final grand total (components + biaya produksi)
    const finalGrandTotal = grandTotal + biayaProduksi;
    
    // Update all grand total displays
    const summaryGrandTotal = document.getElementById('summary-grand-total');
    if (summaryGrandTotal) {
        summaryGrandTotal.textContent = formatRupiah(finalGrandTotal);
    }
    
    const finalGrandTotalDisplay = document.getElementById('grand-total-final-display');
    if (finalGrandTotalDisplay) {
        finalGrandTotalDisplay.textContent = formatRupiah(finalGrandTotal);
    }
    
    // Save to sessionStorage
    sessionStorage.setItem('bom_grand_total', grandTotal);
    sessionStorage.setItem('bom_final_grand_total', finalGrandTotal);
    
    console.log('Components Total:', formatRupiah(grandTotal));
    console.log('Biaya Produksi:', formatRupiah(biayaProduksi));
    console.log('Final Grand Total:', formatRupiah(finalGrandTotal));
}

// ============================================
// ✅ UPDATED: COMPONENT CARD UPDATE WITH M³
// ============================================

function updateComponentCard(index, status, total) {
    const statusElement = document.getElementById(`card-status-${index}`);
    const totalElement = document.getElementById(`card-total-${index}`);
    const m3Element = document.getElementById(`card-m3-${index}`);
    
    if (statusElement) {
        let icon = '<i class="fas fa-circle text-secondary"></i>';
        let text = 'Belum diisi';
        
        if (status === 'filled') {
            icon = '<i class="fas fa-check-circle text-success"></i>';
            text = 'Sudah diisi';
        } else if (status === 'draft') {
            icon = '<i class="fas fa-save text-warning"></i>';
            text = 'Draft tersimpan';
        } else if (status === 'copied') {
            icon = '<i class="fas fa-copy text-info"></i>';
            text = 'Copy dari komponen lain';
        }
        
        statusElement.innerHTML = icon + ' ' + text;
    }
    
    if (totalElement) {
        totalElement.textContent = formatRupiah(total || 0);
    }
    
    // ✅ NEW: UPDATE M³ DISPLAY
    if (m3Element) {
        const key = getCalculationDataKey(index);
        if (calculationData[key] && calculationData[key].m3 !== undefined) {
            const m3Value = calculationData[key].m3;
            m3Element.innerHTML = `<i class="fas fa-cube" style="font-size: 10px;"></i> ${m3Value.toFixed(6)} m³`;
        } else {
            m3Element.innerHTML = `<i class="fas fa-cube" style="font-size: 10px;"></i> 0.000000 m³`;
        }
    }
}

// ============================================
// AUTO SAVE & RESTORE COMPONENT VALUES
// ============================================

function saveComponentValuesToDraft(index, componentType) {
    const draftKey = `bom_component_values_${index}`;
    const values = {};
    
    const modalBody = document.getElementById('modalBody');
    if (!modalBody) return;
    
    const inputs = modalBody.querySelectorAll('input, select');
    inputs.forEach(input => {
        if (input.id) {
            if (input.type === 'checkbox') {
                values[input.id] = input.checked;
            } else if (input.type === 'radio') {
                if (input.checked) {
                    values[input.id] = input.value;
                }
            } else {
                values[input.id] = input.value;
            }
        }
    });
    
    sessionStorage.setItem(draftKey, JSON.stringify(values));
    console.log(`✅ Component ${index} values auto-saved:`, values);
}

function restoreComponentValuesFromDraft(index) {
    const draftKey = `bom_component_values_${index}`;
    const savedValues = sessionStorage.getItem(draftKey);
    
    if (!savedValues) {
        console.log(`No draft found for component ${index}`);
        return;
    }
    
    const values = JSON.parse(savedValues);
    
    setTimeout(() => {
        Object.keys(values).forEach(inputId => {
            const element = document.getElementById(inputId);
            if (element) {
                if (element.type === 'checkbox') {
                    element.checked = values[inputId];
                } else if (element.type === 'radio') {
                    if (element.value === values[inputId]) {
                        element.checked = true;
                    }
                } else {
                    element.value = values[inputId];
                }
            }
        });
        
        console.log(`✅ Component ${index} values restored from draft`);
        
        // ✅ AUTO TRIGGER CALCULATION AFTER RESTORE
        setTimeout(() => {
            triggerAutoCalculate(index);
        }, 300);
        
    }, 500);
}

function triggerAutoCalculate(index) {
    console.log(`🔄 Auto-calculating component ${index}...`);
    
    switch(index) {
        case 0:
            calculateTopPanel(index);
            break;
        case 1:
            calculateSidePanel(index, 'sidel');
            break;
        case 2:
            const siderMode = document.getElementById(`sider-mode-${index}`)?.value;
            if (siderMode === 'manual') {
                calculateSidePanel(index, 'sider');
            }
            break;
        case 3:
            calculateBottomPanel(index);
            break;
        case 4:
            calculateBackPanel(index);
            break;
        case 5:
            calculateFrame(index, 'front');
            break;
        case 6:
            calculateFrame(index, 'back');
            break;
        case 7:
            calculateFaceDrawer(index);
            break;
        case 8:
            calculateDrawer(index, 'frontdrawer');
            break;
        case 9:
            calculateDrawer(index, 'sidedrawerl');
            break;
        case 10:
            const sidedrawerrMode = document.getElementById(`sidedrawerr-mode-${index}`)?.value;
            if (sidedrawerrMode === 'manual') {
                calculateDrawer(index, 'sidedrawerr');
            }
            break;
        case 11:
            calculateDrawer(index, 'backdrawer');
            break;
        case 12:
            calculateDrawer(index, 'basedrawer');
            break;
        case 13:
            calculateSupportDrawer(index);
            break;
        case 14:
            calculateCleatDrawer(index);
            break;
    }
}
// ============================================
// MODAL SYSTEM
// ============================================

let componentModal = null;

document.addEventListener('DOMContentLoaded', function() {
    const modalElement = document.getElementById('componentModal');
    if (modalElement) {
        componentModal = new bootstrap.Modal(modalElement);
    }
});

function openComponentModal(index) {
    currentComponentIndex = index;
    const componentName = getComponentName(index);
    
    console.log(`Opening modal for: ${componentName} (index: ${index})`);
    
    const modalHeader = document.getElementById('modalHeader');
    const modalTitle = document.getElementById('modalTitle');
    
    const colors = ['#3498db', '#9b59b6', '#9b59b6', '#e74c3c', '#f39c12', 
                   '#1abc9c', '#16a085', '#2ecc71', '#27ae60', '#8e44ad', 
                   '#8e44ad', '#c0392b', '#d35400', '#7f8c8d', '#95a5a6'];
    
    const icons = ['fa-layer-group', 'fa-layer-group', 'fa-layer-group', 'fa-layer-group', 
                  'fa-layer-group', 'fa-border-all', 'fa-border-all', 'fa-box-open', 
                  'fa-box', 'fa-box', 'fa-box', 'fa-box', 'fa-box', 'fa-tools', 'fa-tools'];
    
    modalHeader.style.background = `linear-gradient(135deg, ${colors[index]} 0%, ${colors[index]}dd 100%)`;
    modalTitle.innerHTML = `<i class="fas ${icons[index]} me-2"></i>${componentName}`;
    
    loadComponentContentToModal(index);
    
    const saveDraftBtn = document.getElementById('saveDraftModalBtn');
    saveDraftBtn.onclick = function() {
        saveComponentDraft(index, componentName);
    };
    
    if (componentModal) {
        componentModal.show();
    }
}

function loadComponentContentToModal(index) {
    const modalBody = document.getElementById('modalBody');
    
    modalBody.innerHTML = `
        <div class="text-center p-5">
            <i class="fas fa-spinner fa-spin fa-3x text-primary mb-3"></i>
            <p>Loading component...</p>
        </div>
    `;
    
    setTimeout(() => {
        let content = '';
        
        switch(index) {
            case 0:
                content = getTopPanelContent(index);
                break;
            case 1:
                content = getSidePanelContent(index, 'Side Panel L');
                break;
            case 2:
                content = getSidePanelRContent(index);
                break;
            case 3:
                content = getBottomPanelContent(index);
                break;
            case 4:
                content = getBackPanelContent(index);
                break;
            case 5:
                content = getFrameContent(index, 'Front Frame');
                break;
            case 6:
                content = getFrameContent(index, 'Back Frame');
                break;
            case 7:
                content = getFaceDrawerContent(index);
                break;
            case 8:
                content = getDrawerContent(index, 'Front Drawer');
                break;
            case 9:
                content = getDrawerContent(index, 'Side Drawer L');
                break;
            case 10:
                content = getSideDrawerRContent(index);
                break;
            case 11:
                content = getDrawerContent(index, 'Back Drawer');
                break;
            case 12:
                content = getDrawerContent(index, 'Base Drawer');
                break;
            case 13:
                content = getSupportDrawerContent(index);
                break;
            case 14:
                content = getCleatDrawerContent(index);
                break;
            default:
                content = '<div class="p-4"><p class="text-danger">Component not found</p></div>';
        }
        
        modalBody.innerHTML = content;
        
        restoreComponentValuesFromDraft(index);
        
    }, 300);
}

// ============================================
// ✅ SAVE COMPONENT DRAFT (WITH M³) - FIXED ALERT LOGIC
// ============================================

function saveComponentDraft(index, componentName) {
    console.log('=== SAVE COMPONENT DRAFT ===');
    console.log('Index:', index);
    console.log('Component Name:', componentName);
    
    // ✅ SAVE INPUT VALUES FIRST
    saveComponentValuesToDraft(index, componentName);
    
    const key = getCalculationDataKey(index);
    
    console.log('Key:', key);
    console.log('Calculation Data:', calculationData[key]);
    
    // ✅ FIXED: Cek apakah data ada DAN componentTotal > 0
    if (!calculationData[key] || calculationData[key].componentTotal === 0 || calculationData[key].componentTotal === undefined) {
        alert('⚠️ Belum ada data yang dihitung. Silakan isi dan hitung komponen terlebih dahulu.');
        return;
    }
    
    componentDrafts[index] = calculationData[key];
    
    if (componentName === 'Side Panel L') {
        sidePanelLData = calculationData[key];
        sessionStorage.setItem('bom_side_panel_l_data', JSON.stringify(sidePanelLData));
        console.log('Side Panel L Data Saved:', sidePanelLData);
    }
    
    if (componentName === 'Side Drawer L') {
        sideDrawerLData = calculationData[key];
        sessionStorage.setItem('bom_side_drawer_l_data', JSON.stringify(sideDrawerLData));
        console.log('Side Drawer L Data Saved:', sideDrawerLData);
    }
    
    sessionStorage.setItem('bom_component_drafts', JSON.stringify(componentDrafts));
    sessionStorage.setItem('bom_calculation_data', JSON.stringify(calculationData));
    sessionStorage.setItem('bom_grand_total', grandTotal);
    sessionStorage.setItem('bom_face_drawer_qty', faceDrawerQty);
    
    // ✅ SAVE M³ DATA
    sessionStorage.setItem('bom_total_m3', totalM3);
    sessionStorage.setItem('bom_harga_per_m3', hargaPerM3);
    sessionStorage.setItem('bom_biaya_produksi', biayaProduksi);
    
    updateComponentCard(index, 'draft', calculationData[key].componentTotal);
    
    // ✅ IMPROVED ALERT WITH M³ INFO
    alert(`✅ Draft ${componentName} berhasil disimpan!\n\n💰 Total: ${formatRupiah(calculationData[key].componentTotal)}\n🧊 M³: ${calculationData[key].m3 ? calculationData[key].m3.toFixed(6) : '0.000000'}`);
    console.log('Component Draft Saved:', componentName, componentDrafts[index]);
}

// ============================================
// STEP NAVIGATION
// ============================================

function goToCalculator() {
    const form = document.getElementById('initialForm');
    if (!form.checkValidity()) {
        form.reportValidity();
        return;
    }
    
    const formData = new FormData(form);
    initialFormData = {};
    for (let [key, value] of formData.entries()) {
        if (key !== 'image') {
            initialFormData[key] = value;
        }
    }
    
    console.log('Initial Form Data:', initialFormData);
    
    sessionStorage.setItem('bom_initial_form', JSON.stringify(initialFormData));
    
    document.getElementById('summary-product').textContent = initialFormData.product_name || '-';
    document.getElementById('summary-type').textContent = initialFormData.product_type || '-';
    document.getElementById('summary-code').textContent = initialFormData.product_code || '-';
    
    // Update grand total displays
    updateGrandTotal();
    
    document.getElementById('step-indicator-1').classList.remove('active');
    document.getElementById('step-indicator-2').classList.add('active');
    
    document.getElementById('step-1-card').style.display = 'none';
    document.getElementById('step-2-card').style.display = 'block';
    
    currentStep = 2;
    sessionStorage.setItem('bom_step2_visible', 'true');
    
    window.scrollTo({ top: 0, behavior: 'smooth' });
    
    loadAllDrafts();
}

function backToStep1() {
    document.getElementById('step-indicator-2').classList.remove('active');
    document.getElementById('step-indicator-1').classList.add('active');
    
    document.getElementById('step-2-card').style.display = 'none';
    document.getElementById('step-1-card').style.display = 'block';
    
    currentStep = 1;
    sessionStorage.setItem('bom_step2_visible', 'false');
    
    window.scrollTo({ top: 0, behavior: 'smooth' });
}

function saveStep1Draft() {
    const form = document.getElementById('initialForm');
    if (!form.checkValidity()) {
        form.reportValidity();
        return;
    }
    
    const formData = new FormData(form);
    const draftData = {};
    for (let [key, value] of formData.entries()) {
        if (key !== 'image') {
            draftData[key] = value;
        }
    }
    
    sessionStorage.setItem('bom_initial_form', JSON.stringify(draftData));
    alert('✅ Draft Step 1 berhasil disimpan!');
    console.log('Step 1 Draft Saved:', draftData);
}

function clearImagePreview() {
    document.getElementById('image').value = '';
    document.getElementById('imagePreview').style.display = 'none';
}

// ============================================
// IMAGE PREVIEW HANDLER
// ============================================

document.addEventListener('DOMContentLoaded', function() {
    const imageInput = document.getElementById('image');
    if (imageInput) {
        imageInput.addEventListener('change', function(e) {
            const file = e.target.files[0];
            if (file) {
                const maxSize = 11 * 1024 * 1024;
                if (file.size > maxSize) {
                    alert('⚠️ Ukuran file terlalu besar! Maximum 11MB');
                    this.value = '';
                    document.getElementById('imagePreview').style.display = 'none';
                    return;
                }
                
                const reader = new FileReader();
                reader.onload = function(e) {
                    document.getElementById('previewImg').src = e.target.result;
                    document.getElementById('imagePreview').style.display = 'block';
                };
                reader.readAsDataURL(file);
            } else {
                document.getElementById('imagePreview').style.display = 'none';
            }
        });
    }
    
    const today = new Date().toISOString().split('T')[0];
    document.getElementById('invoice_date').value = today;
});

// ============================================
// ✅ DRAFT SYSTEM (WITH M³ DATA)
// ============================================

function loadAllDrafts() {
    const initialDraft = sessionStorage.getItem('bom_initial_form');
    if (initialDraft) {
        initialFormData = JSON.parse(initialDraft);
        console.log('Initial Form Loaded from Draft:', initialFormData);
    }
    
    const calcDraft = sessionStorage.getItem('bom_calculation_data');
    if (calcDraft) {
        calculationData = JSON.parse(calcDraft);
        console.log('Calculation Data Loaded from Draft:', calculationData);
    }
    
    const compDraft = sessionStorage.getItem('bom_component_drafts');
    if (compDraft) {
        componentDrafts = JSON.parse(compDraft);
        console.log('Component Drafts Loaded:', componentDrafts);
    }
    
    const gtDraft = sessionStorage.getItem('bom_grand_total');
    if (gtDraft) {
        grandTotal = parseFloat(gtDraft);
    }
    
    const faceQtyDraft = sessionStorage.getItem('bom_face_drawer_qty');
    if (faceQtyDraft) {
        faceDrawerQty = parseInt(faceQtyDraft);
    }
    
    const sidePanelLDraft = sessionStorage.getItem('bom_side_panel_l_data');
    if (sidePanelLDraft) {
        sidePanelLData = JSON.parse(sidePanelLDraft);
    }
    
    const sideDrawerLDraft = sessionStorage.getItem('bom_side_drawer_l_data');
    if (sideDrawerLDraft) {
        sideDrawerLData = JSON.parse(sideDrawerLDraft);
    }
    
    // ✅ LOAD M³ DATA
    const m3Draft = sessionStorage.getItem('bom_total_m3');
    if (m3Draft) {
        totalM3 = parseFloat(m3Draft);
        document.getElementById('total-m3-display').value = totalM3.toFixed(6);
    }
    
    const hargaM3Draft = sessionStorage.getItem('bom_harga_per_m3');
    if (hargaM3Draft) {
        hargaPerM3 = parseFloat(hargaM3Draft);
        document.getElementById('harga-per-m3-select').value = hargaPerM3;
    }
    
    const biayaDraft = sessionStorage.getItem('bom_biaya_produksi');
    if (biayaDraft) {
        biayaProduksi = parseFloat(biayaDraft);
        document.getElementById('biaya-produksi-display').textContent = formatRupiah(biayaProduksi);
        document.getElementById('biaya-produksi-summary').textContent = formatRupiah(biayaProduksi);
    }
    
    // Update all displays
    updateGrandTotal();
    
    Object.keys(componentDrafts).forEach(index => {
        updateComponentCard(parseInt(index), 'draft', componentDrafts[index].componentTotal);
    });
    
    console.log('=== ALL DRAFTS LOADED ===');
}

function clearAllDrafts() {
    if (!confirm('⚠️ Yakin ingin menghapus SEMUA draft? Data yang belum disimpan akan hilang!')) {
        return;
    }
    
    sessionStorage.removeItem('bom_initial_form');
    sessionStorage.removeItem('bom_component_drafts');
    sessionStorage.removeItem('bom_calculation_data');
    sessionStorage.removeItem('bom_grand_total');
    sessionStorage.removeItem('bom_face_drawer_qty');
    sessionStorage.removeItem('bom_step2_visible');
    sessionStorage.removeItem('bom_side_panel_l_data');
    sessionStorage.removeItem('bom_side_drawer_l_data');
    
    // ✅ CLEAR M³ DATA
    sessionStorage.removeItem('bom_total_m3');
    sessionStorage.removeItem('bom_harga_per_m3');
    sessionStorage.removeItem('bom_biaya_produksi');
    sessionStorage.removeItem('bom_final_grand_total');
    
    // Clear component values
    for (let i = 0; i < 15; i++) {
        sessionStorage.removeItem(`bom_component_values_${i}`);
    }
    
    calculationData = {};
    componentDrafts = {};
    grandTotal = 0;
    faceDrawerQty = 1;
    sidePanelLData = null;
    sideDrawerLData = null;
    totalM3 = 0;
    hargaPerM3 = 6000000;
    biayaProduksi = 0;
    
    alert('✅ Semua draft berhasil dihapus!');
    location.reload();
}
// ============================================
// ✅ FINALIZE INVOICE (WITH M³ DATA)
// ============================================

function finalizeInvoice() {
    const hasComponents = Object.keys(calculationData).some(key => 
        calculationData[key].componentTotal && calculationData[key].componentTotal > 0
    );
    
    if (!hasComponents) {
        if (!confirm('⚠️ Belum ada komponen yang diisi. Yakin ingin melanjutkan?')) {
            return;
        }
    }
    
    // Calculate final grand total
    const finalGrandTotal = grandTotal + biayaProduksi;
    
    const confirmText = `Apakah Anda yakin ingin menyimpan invoice ini?

📦 Total Material (Components): ${formatRupiah(grandTotal)}
🏭 Biaya Produksi (M³): ${formatRupiah(biayaProduksi)}
💰 GRAND TOTAL: ${formatRupiah(finalGrandTotal)}

Setelah disimpan, Anda akan diarahkan ke halaman detail invoice.`;
    
    if (!confirm(confirmText)) {
        return;
    }
    
    document.getElementById('loadingOverlay').style.display = 'flex';
    
    const formData = new FormData();
    
    Object.keys(initialFormData).forEach(key => {
        formData.append(key, initialFormData[key]);
    });
    
    const imageFile = document.getElementById('image').files[0];
    if (imageFile) {
        formData.append('image', imageFile);
    }
    
    const calculationDataJson = JSON.stringify(calculationData);
    formData.append('calculation_data', calculationDataJson);
    formData.append('grand_total', grandTotal);
    
    // ✅ ADD M³ DATA TO FORM
    formData.append('total_m3', totalM3);
    formData.append('harga_per_m3', hargaPerM3);
    formData.append('biaya_produksi', biayaProduksi);
    formData.append('final_grand_total', finalGrandTotal);
    
    console.log('=== FINALIZE INVOICE ===');
    console.log('Calculation Data:', calculationData);
    console.log('Components Total:', formatRupiah(grandTotal));
    console.log('Total M³:', totalM3);
    console.log('Harga per M³:', formatRupiah(hargaPerM3));
    console.log('Biaya Produksi:', formatRupiah(biayaProduksi));
    console.log('Final Grand Total:', formatRupiah(finalGrandTotal));
    console.log('Has Image:', !!imageFile);
    
    fetch('{{ route("invoice.bom.store") }}', {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': '{{ csrf_token() }}'
        },
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        console.log('Server Response:', data);
        
        if (data.success) {
            sessionStorage.clear();
            
            alert('✅ Invoice berhasil dibuat!');
            window.location.href = `{{ url('invoice/bom') }}/${data.invoice.id}`;
        } else {
            throw new Error(data.message || 'Failed to create invoice');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('❌ Terjadi kesalahan saat membuat invoice: ' + error.message);
    })
    .finally(() => {
        document.getElementById('loadingOverlay').style.display = 'none';
    });
}

// ============================================
// PAGE LOAD INITIALIZATION
// ============================================

document.addEventListener('DOMContentLoaded', function() {
    console.log('BOM Create Page Loaded');
    
    const mainDraft = sessionStorage.getItem('bom_initial_form');
    if (mainDraft) {
        const draftData = JSON.parse(mainDraft);
        Object.keys(draftData).forEach(key => {
            const element = document.getElementById(key);
            if (element) {
                element.value = draftData[key];
            }
        });
    }
    
    const step2Visible = sessionStorage.getItem('bom_step2_visible');
    if (step2Visible === 'true') {
        const initialDraft = sessionStorage.getItem('bom_initial_form');
        if (initialDraft) {
            initialFormData = JSON.parse(initialDraft);
            
            Object.keys(initialFormData).forEach(key => {
                const element = document.getElementById(key);
                if (element) {
                    element.value = initialFormData[key];
                }
            });
            
            goToCalculator();
        }
    }
    
    console.log('✅ Page initialized');
});

window.addEventListener('beforeunload', function(e) {
    const step2Visible = document.getElementById('step-2-card').style.display !== 'none';
    sessionStorage.setItem('bom_step2_visible', step2Visible ? 'true' : 'false');
    
    if (Object.keys(calculationData).length > 0 || Object.keys(initialFormData).length > 0) {
        e.preventDefault();
        e.returnValue = '';
    }
});

// ============================================
// ✅ FACE DRAWER QTY CHANGE HANDLER
// ============================================

function onFaceDrawerQtyChange(index) {
    const qtyInput = document.getElementById(`facedrawer-qty-${index}`);
    if (qtyInput) {
        const newQty = parseInt(qtyInput.value) || 1;
        const oldQty = faceDrawerQty;
        
        faceDrawerQty = newQty;
        sessionStorage.setItem('bom_face_drawer_qty', faceDrawerQty);
        
        console.log('Face Drawer Qty Updated:', oldQty, '→', faceDrawerQty);
        
        // Update qty displays in other drawers
        const drawerPrefixes = ['frontdrawer', 'sidedrawerl', 'sidedrawerr', 'backdrawer', 'basedrawer'];
        drawerPrefixes.forEach(prefix => {
            const qtyDisplay = document.querySelector(`input[value="${oldQty}"][readonly]`);
            if (qtyDisplay) {
                qtyDisplay.value = faceDrawerQty;
            }
        });
    }
}

function recalculateAllDrawers() {
    console.log('=== RECALCULATE ALL DRAWERS ===');
    console.log('New Face Drawer Qty:', faceDrawerQty);
    
    const drawerComponents = [
        { index: 8, prefix: 'frontdrawer', name: 'Front Drawer' },
        { index: 9, prefix: 'sidedrawerl', name: 'Side Drawer L' },
        { index: 10, prefix: 'sidedrawerr', name: 'Side Drawer R' },
        { index: 11, prefix: 'backdrawer', name: 'Back Drawer' },
        { index: 12, prefix: 'basedrawer', name: 'Base Drawer' }
    ];
    
    drawerComponents.forEach(drawer => {
        const dimensionSelect = document.getElementById(`${drawer.prefix}-dimension-${drawer.index}`);
        
        if (dimensionSelect && dimensionSelect.value) {
            console.log(`♻️ Recalculating ${drawer.name}...`);
            
            if (drawer.prefix === 'sidedrawerr') {
                const mode = document.getElementById(`sidedrawerr-mode-${drawer.index}`)?.value;
                
                if (mode === 'copy') {
                    const sideLDimension = document.getElementById(`sidedrawerl-dimension-9`);
                    if (sideLDimension && sideLDimension.value) {
                        calculateDrawer(9, 'sidedrawerl');
                    }
                    
                    copySideDrawerLToR(drawer.index);
                } else if (mode === 'manual') {
                    calculateDrawer(drawer.index, drawer.prefix);
                }
            } else {
                calculateDrawer(drawer.index, drawer.prefix);
            }
        }
    });
    
    console.log('✅ All drawers recalculated!');
}

// ============================================
// ✅ SIDE PANEL R - MODE CHANGE
// ============================================

function onSidePanelRModeChange(index) {
    const mode = document.getElementById(`sider-mode-${index}`).value;
    const manualInputs = document.getElementById(`sider-manual-inputs-${index}`);
    
    if (mode === 'copy') {
        manualInputs.style.display = 'none';
        
        if (!sidePanelLData) {
            alert('❌ Side Panel L belum diisi! Silakan isi Side Panel L terlebih dahulu.');
            document.getElementById(`sider-mode-${index}`).value = '';
            return;
        }
        
        copySidePanelLToR(index);
        
    } else if (mode === 'manual') {
        manualInputs.style.display = 'block';
        
        document.getElementById(`sider-width-${index}`).value = '';
        document.getElementById(`sider-length-${index}`).value = '';
        document.getElementById(`sider-thickness-${index}`).value = '';
        document.getElementById(`sider-qty-${index}`).value = '1';
        
        document.getElementById(`total-sider-panel-${index}`).textContent = 'Rp 0';
        
        delete calculationData[`sider-panel-${index}`];
        updateComponentCard(index, 'empty', 0);
        updateTotalM3AndBiayaProduksi();
        
    } else {
        manualInputs.style.display = 'none';
    }
}

function copySidePanelLToR(index) {
    console.log('=== COPY SIDE PANEL L TO R ===');
    
    if (!sidePanelLData) {
        alert('❌ Data Side Panel L tidak ditemukan!');
        return;
    }
    
    calculationData[`sider-panel-${index}`] = {
        componentName: 'SIDE PANEL R',
        componentTotal: sidePanelLData.componentTotal,
        m3: sidePanelLData.m3, // ✅ COPY M³
        details: { ...sidePanelLData.details },
        subComponents: { ...sidePanelLData.subComponents },
        copiedFrom: 'Side Panel L'
    };
    
    document.getElementById(`total-sider-panel-${index}`).textContent = formatRupiah(sidePanelLData.componentTotal);
    
    updateComponentCard(index, 'copied', sidePanelLData.componentTotal);
    
    updateTotalM3AndBiayaProduksi();
    
    saveComponentValuesToDraft(index, 'Side Panel R');
    
    alert('✅ Side Panel R berhasil di-copy dari Side Panel L!');
    console.log('Side Panel R (copied):', calculationData[`sider-panel-${index}`]);
}

// ============================================
// ✅ SIDE DRAWER R - MODE CHANGE
// ============================================

function onSideDrawerRModeChange(index) {
    const mode = document.getElementById(`sidedrawerr-mode-${index}`).value;
    const manualInputs = document.getElementById(`sidedrawerr-manual-inputs-${index}`);
    
    if (mode === 'copy') {
        manualInputs.style.display = 'none';
        
        if (!sideDrawerLData) {
            alert('❌ Side Drawer L belum diisi! Silakan isi Side Drawer L terlebih dahulu.');
            document.getElementById(`sidedrawerr-mode-${index}`).value = '';
            return;
        }
        
        copySideDrawerLToR(index);
        
    } else if (mode === 'manual') {
        manualInputs.style.display = 'block';
        
        document.getElementById(`sidedrawerr-dimension-${index}`).value = '';
        document.getElementById(`total-sidedrawerr-${index}`).textContent = 'Rp 0';
        
        delete calculationData[`sidedrawerr-${index}`];
        updateComponentCard(index, 'empty', 0);
        updateTotalM3AndBiayaProduksi();
        
    } else {
        manualInputs.style.display = 'none';
    }
}

function copySideDrawerLToR(index) {
    console.log('=== COPY SIDE DRAWER L TO R ===');
    
    if (!sideDrawerLData) {
        alert('❌ Data Side Drawer L tidak ditemukan!');
        return;
    }
    
    calculationData[`sidedrawerr-${index}`] = {
        componentName: 'SIDE DRAWER R',
        componentTotal: sideDrawerLData.componentTotal,
        m3: sideDrawerLData.m3, // ✅ COPY M³
        details: { ...sideDrawerLData.details },
        copiedFrom: 'Side Drawer L'
    };
    
    document.getElementById(`total-sidedrawerr-${index}`).textContent = formatRupiah(sideDrawerLData.componentTotal);
    
    updateComponentCard(index, 'copied', sideDrawerLData.componentTotal);
    
    updateTotalM3AndBiayaProduksi();
    
    saveComponentValuesToDraft(index, 'Side Drawer R');
    
    alert('✅ Side Drawer R berhasil di-copy dari Side Drawer L!');
    console.log('Side Drawer R (copied):', calculationData[`sidedrawerr-${index}`]);
}
// ============================================
// ✅ CALCULATE TOP PANEL - FLEXIBLE INPUT (AUTO CALCULATE) + M³ DISPLAY
// ============================================

function calculateTopPanel(index) {
    console.log('=== CALCULATE TOP PANEL ===');
    
    const width = parseFloat(document.getElementById(`top-width-${index}`).value) || 0;
    const length = parseFloat(document.getElementById(`top-length-${index}`).value) || 0;
    const thicknessTotal = parseFloat(document.getElementById(`top-thickness-${index}`).value) || 0;
    const qty = parseFloat(document.getElementById(`top-qty-${index}`).value) || 1;
    
    // ✅ FLEXIBLE: Hanya cek width, length, thickness (qty default 1)
    if (!width || !length || !thicknessTotal) {
        console.log('❌ Dimensi belum lengkap, skip calculation');
        return;
    }
    
    if (![18, 21, 28].includes(thicknessTotal)) {
        console.log('❌ Thickness tidak valid');
        return;
    }
    
    const m2 = calculateM2(width, length, qty);
    document.getElementById(`top-m2-${index}`).value = m2.toFixed(4);
    
    // ✅ CALCULATE M³
    const m3 = calculateM3(width, length, thicknessTotal, qty);
    
    // ✅ UPDATE M³ DISPLAY IN MODAL
    const m3Display = document.getElementById(`top-m3-${index}`);
    if (m3Display) {
        m3Display.value = m3.toFixed(6);
    }
    
    const coreThickness = getCoreThickness(thicknessTotal);
    
    document.getElementById(`top-core-thickness-${index}`).value = coreThickness + 'mm';
    document.getElementById(`top-core-type-${index}`).value = initialFormData.panel_type;
    
    let componentTotal = 0;
    const subComponents = {};
    
    // 1. Patching Panel 1 (Core Panel - ALWAYS CALCULATED)
    const corePanel = getPanelsByType(initialFormData.panel_type)
        .find(p => parseFloat(p.thickness) === parseFloat(coreThickness));
    
    if (corePanel) {
        const patchingPanelQty = parseFloat(document.getElementById(`top-patching-panel-qty-${index}`).value) || 1;
        const patchingM2 = calculateM2(width, length, qty * patchingPanelQty);
        const patchingTotal = patchingM2 * corePanel.price;
        
        document.getElementById(`top-patching-panel-m2-${index}`).value = patchingM2.toFixed(4);
        document.getElementById(`top-patching-panel-price-${index}`).textContent = formatRupiah(corePanel.price);
        document.getElementById(`top-patching-panel-total-${index}`).textContent = formatRupiah(patchingTotal);
        
        componentTotal += patchingTotal;
        
        subComponents.patchingPanel = {
            name: 'Patching Panel 1',
            thickness: coreThickness,
            width: width,
            length: length,
            qty: qty * patchingPanelQty,
            m2: patchingM2,
            price: corePanel.price,
            total: patchingTotal,
            material: initialFormData.panel_type
        };
    }
    
    // 2. Patching Solid 2 (PANJANG) - ✅ OPTIONAL
    const solid2Width = parseFloat(document.getElementById(`top-patching-solid2-width-${index}`).value) || 0;
    const solid2Select = document.getElementById(`top-patching-solid2-thickness-${index}`);
    const solid2Qty = parseFloat(document.getElementById(`top-patching-solid2-qty-${index}`).value) || 2;
    
    if (solid2Select && solid2Select.value && solid2Width > 0) {
        const selectedOption = solid2Select.options[solid2Select.selectedIndex];
        const thickness = parseFloat(selectedOption.dataset.thickness);
        const price = parseFloat(selectedOption.dataset.price);
        
        const solid2M2 = calculateM2(solid2Width, length, qty * solid2Qty);
        const total = solid2M2 * price;
        
        document.getElementById(`top-patching-solid2-m2-${index}`).value = solid2M2.toFixed(4);
        document.getElementById(`top-patching-solid2-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`top-patching-solid2-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.patchingSolid2 = {
            name: 'Patching Solid 2 (PANJANG)',
            thickness: thickness,
            width: solid2Width,
            length: length,
            qty: qty * solid2Qty,
            m2: solid2M2,
            price: price,
            total: total,
            material: `SL.${initialFormData.solid_type}`
        };
    }
    
    // 3. Patching Solid 3 (LEBAR) - ✅ OPTIONAL
    const solid3Width = parseFloat(document.getElementById(`top-patching-solid3-width-${index}`).value) || 0;
    const solid3Select = document.getElementById(`top-patching-solid3-thickness-${index}`);
    const solid3Qty = parseFloat(document.getElementById(`top-patching-solid3-qty-${index}`).value) || 2;
    
    if (solid3Select && solid3Select.value && solid3Width > 0) {
        const selectedOption = solid3Select.options[solid3Select.selectedIndex];
        const thickness = parseFloat(selectedOption.dataset.thickness);
        const price = parseFloat(selectedOption.dataset.price);
        
        const solid3M2 = calculateM2(solid3Width, width, qty * solid3Qty);
        const total = solid3M2 * price;
        
        document.getElementById(`top-patching-solid3-m2-${index}`).value = solid3M2.toFixed(4);
        document.getElementById(`top-patching-solid3-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`top-patching-solid3-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.patchingSolid3 = {
            name: 'Patching Solid 3 (LEBAR)',
            thickness: thickness,
            width: solid3Width,
            length: width,
            qty: qty * solid3Qty,
            m2: solid3M2,
            price: price,
            total: total,
            material: `SL.${initialFormData.solid_type}`
        };
    }
    
    // 4. Crossgrain (MDF) - ✅ OPTIONAL
    const crossgrainSelect = document.getElementById(`top-crossgrain-thickness-${index}`);
    const crossgrainQty = parseFloat(document.getElementById(`top-crossgrain-qty-${index}`).value) || 2;
    
    if (crossgrainSelect && crossgrainSelect.value) {
        const selectedOption = crossgrainSelect.options[crossgrainSelect.selectedIndex];
        const thickness = parseFloat(selectedOption.dataset.thickness);
        const price = parseFloat(selectedOption.dataset.price);
        
        const crossgrainM2 = calculateM2(width, length, qty * crossgrainQty);
        const total = crossgrainM2 * price;
        
        document.getElementById(`top-crossgrain-m2-${index}`).value = crossgrainM2.toFixed(4);
        document.getElementById(`top-crossgrain-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`top-crossgrain-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.crossgrain = {
            name: 'Crossgrain',
            thickness: thickness,
            width: width,
            length: length,
            qty: qty * crossgrainQty,
            m2: crossgrainM2,
            price: price,
            total: total,
            material: 'MDF'
        };
    }
    
    // 5. Veneer Face - ✅ OPTIONAL
    const veneerFacePattern = document.getElementById(`top-veneer-face-pattern-${index}`).value;
    const veneerFaceSelect = document.getElementById(`top-veneer-face-type-${index}`);
    const veneerFaceQty = parseFloat(document.getElementById(`top-veneer-face-qty-${index}`).value) || 1;
    
    if (veneerFaceSelect && veneerFaceSelect.value && veneerFaceQty > 0) {
        const selectedOption = veneerFaceSelect.options[veneerFaceSelect.selectedIndex];
        const veneerType = selectedOption.dataset.type;
        const priceLurus = parseFloat(selectedOption.dataset.priceLurus);
        const priceMotif = parseFloat(selectedOption.dataset.priceMotif);
        const price = veneerFacePattern === 'LURUS' ? priceLurus : priceMotif;
        
        const veneerFaceM2 = calculateM2(width, length, qty * veneerFaceQty);
        const total = veneerFaceM2 * price;
        
        document.getElementById(`top-veneer-face-m2-${index}`).value = veneerFaceM2.toFixed(4);
        document.getElementById(`top-veneer-face-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`top-veneer-face-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.veneerFace = {
            name: 'TV. FACE',
            thickness: 0.5,
            width: width,
            length: length,
            qty: qty * veneerFaceQty,
            m2: veneerFaceM2,
            price: price,
            total: total,
            material: `V.${veneerType}`,
            pattern: veneerFacePattern,
            veneerType: veneerType
        };
    }
    
    // 6. Veneer Back - ✅ OPTIONAL
    const veneerBackPattern = document.getElementById(`top-veneer-back-pattern-${index}`).value;
    const veneerBackSelect = document.getElementById(`top-veneer-back-type-${index}`);
    const veneerBackQty = parseFloat(document.getElementById(`top-veneer-back-qty-${index}`).value) || 1;
    
    if (veneerBackSelect && veneerBackSelect.value && veneerBackQty > 0) {
        const selectedOption = veneerBackSelect.options[veneerBackSelect.selectedIndex];
        const veneerType = selectedOption.dataset.type;
        const priceLurus = parseFloat(selectedOption.dataset.priceLurus);
        const priceMotif = parseFloat(selectedOption.dataset.priceMotif);
        const price = veneerBackPattern === 'LURUS' ? priceLurus : priceMotif;
        
        const veneerBackM2 = calculateM2(width, length, qty * veneerBackQty);
        const total = veneerBackM2 * price;
        
        document.getElementById(`top-veneer-back-m2-${index}`).value = veneerBackM2.toFixed(4);
        document.getElementById(`top-veneer-back-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`top-veneer-back-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.veneerBack = {
            name: 'TV. BACK',
            thickness: 0.5,
            width: width,
            length: length,
            qty: qty * veneerBackQty,
            m2: veneerBackM2,
            price: price,
            total: total,
            material: `V.${veneerType}`,
            pattern: veneerBackPattern,
            veneerType: veneerType
        };
    }
    
    // 7. Cleat - ✅ OPTIONAL
    const cleatQty = parseFloat(document.getElementById(`top-cleat-qty-${index}`).value) || 0;
    
    if (cleatQty > 0) {
        const cleatPricePerUnit = 1000;
        const cleatTotal = cleatQty * cleatPricePerUnit;
        
        document.getElementById(`top-cleat-total-${index}`).textContent = formatRupiah(cleatTotal);
        
        componentTotal += cleatTotal;
        
        subComponents.cleat = {
            name: 'Cleat',
            thickness: 11,
            width: 11,
            length: 60,
            qty: cleatQty,
            pricePerUnit: cleatPricePerUnit,
            total: cleatTotal,
            material: 'Mix'
        };
    }
    
    document.getElementById(`total-top-panel-${index}`).textContent = formatRupiah(componentTotal);
    
    // ✅ SAVE WITH M³
    calculationData[`top-panel-${index}`] = {
        componentName: 'TOP PANEL',
        componentTotal: componentTotal,
        m3: m3,
        details: {
            width: width,
            length: length,
            qty: qty,
            m2: m2,
            thicknessTotal: thicknessTotal,
            coreThickness: coreThickness
        },
        subComponents: subComponents
    };
    
    console.log('✅ Top Panel Calculation (Flexible):', calculationData[`top-panel-${index}`]);
    
    updateComponentCard(index, 'filled', componentTotal);
    updateTotalM3AndBiayaProduksi();
    saveComponentValuesToDraft(index, 'Top Panel');
}
// ============================================
// ✅ CALCULATE SIDE PANEL L/R - FLEXIBLE INPUT (AUTO CALCULATE) + M³ DISPLAY
// ============================================

function calculateSidePanel(index, prefix) {
    console.log(`=== CALCULATE SIDE PANEL (${prefix.toUpperCase()}) - FLEXIBLE ===`);
    
    const width = parseFloat(document.getElementById(`${prefix}-width-${index}`).value) || 0;
    const length = parseFloat(document.getElementById(`${prefix}-length-${index}`).value) || 0;
    const thicknessTotal = parseFloat(document.getElementById(`${prefix}-thickness-${index}`).value) || 0;
    const qty = parseFloat(document.getElementById(`${prefix}-qty-${index}`).value) || 1;
    
    // ✅ FLEXIBLE: Hanya cek width, length, thickness
    if (!width || !length || !thicknessTotal) {
        console.log('❌ Dimensi belum lengkap, skip calculation');
        return;
    }
    
    if (![18, 21, 28].includes(thicknessTotal)) {
        console.log('❌ Thickness tidak valid');
        return;
    }
    
    const m2 = calculateM2(width, length, qty);
    document.getElementById(`${prefix}-m2-${index}`).value = m2.toFixed(4);
    
    // ✅ CALCULATE M³
    const m3 = calculateM3(width, length, thicknessTotal, qty);
    
    // ✅ UPDATE M³ DISPLAY IN MODAL
    const m3Display = document.getElementById(`${prefix}-m3-${index}`);
    if (m3Display) {
        m3Display.value = m3.toFixed(6);
    }
    
    const coreThickness = getCoreThickness(thicknessTotal);
    
    document.getElementById(`${prefix}-core-thickness-${index}`).value = coreThickness + 'mm';
    document.getElementById(`${prefix}-core-type-${index}`).value = initialFormData.panel_type;
    
    let componentTotal = 0;
    const subComponents = {};
    
    // 1. Patching Panel 1 (Core Panel - ALWAYS CALCULATED)
    const corePanel = getPanelsByType(initialFormData.panel_type)
        .find(p => parseFloat(p.thickness) === parseFloat(coreThickness));
    
    if (corePanel) {
        const patchingPanelQty = parseFloat(document.getElementById(`${prefix}-patching-panel-qty-${index}`).value) || 1;
        const patchingM2 = calculateM2(width, length, qty * patchingPanelQty);
        const patchingTotal = patchingM2 * corePanel.price;
        
        document.getElementById(`${prefix}-patching-panel-m2-${index}`).value = patchingM2.toFixed(4);
        document.getElementById(`${prefix}-patching-panel-price-${index}`).textContent = formatRupiah(corePanel.price);
        document.getElementById(`${prefix}-patching-panel-total-${index}`).textContent = formatRupiah(patchingTotal);
        
        componentTotal += patchingTotal;
        
        subComponents.patchingPanel = {
            name: 'Patching Panel 1',
            thickness: coreThickness,
            width: width,
            length: length,
            qty: qty * patchingPanelQty,
            m2: patchingM2,
            price: corePanel.price,
            total: patchingTotal,
            material: initialFormData.panel_type
        };
    }
    
    // 2. Patching Solid 2 (PANJANG) - ✅ OPTIONAL
    const solid2Width = parseFloat(document.getElementById(`${prefix}-patching-solid2-width-${index}`).value) || 0;
    const solid2Select = document.getElementById(`${prefix}-patching-solid2-thickness-${index}`);
    const solid2Qty = parseFloat(document.getElementById(`${prefix}-patching-solid2-qty-${index}`).value) || 2;
    
    if (solid2Select && solid2Select.value && solid2Width > 0) {
        const selectedOption = solid2Select.options[solid2Select.selectedIndex];
        const thickness = parseFloat(selectedOption.dataset.thickness);
        const price = parseFloat(selectedOption.dataset.price);
        
        const solid2M2 = calculateM2(solid2Width, length, qty * solid2Qty);
        const total = solid2M2 * price;
        
        document.getElementById(`${prefix}-patching-solid2-m2-${index}`).value = solid2M2.toFixed(4);
        document.getElementById(`${prefix}-patching-solid2-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`${prefix}-patching-solid2-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.patchingSolid2 = {
            name: 'Patching Solid 2 (PANJANG)',
            thickness: thickness,
            width: solid2Width,
            length: length,
            qty: qty * solid2Qty,
            m2: solid2M2,
            price: price,
            total: total,
            material: `SL.${initialFormData.solid_type}`
        };
    }
    
    // 3. Patching Solid 3 (LEBAR) - ✅ OPTIONAL
    const solid3Width = parseFloat(document.getElementById(`${prefix}-patching-solid3-width-${index}`).value) || 0;
    const solid3Select = document.getElementById(`${prefix}-patching-solid3-thickness-${index}`);
    const solid3Qty = parseFloat(document.getElementById(`${prefix}-patching-solid3-qty-${index}`).value) || 2;
    
    if (solid3Select && solid3Select.value && solid3Width > 0) {
        const selectedOption = solid3Select.options[solid3Select.selectedIndex];
        const thickness = parseFloat(selectedOption.dataset.thickness);
        const price = parseFloat(selectedOption.dataset.price);
        
        const solid3M2 = calculateM2(solid3Width, width, qty * solid3Qty);
        const total = solid3M2 * price;
        
        document.getElementById(`${prefix}-patching-solid3-m2-${index}`).value = solid3M2.toFixed(4);
        document.getElementById(`${prefix}-patching-solid3-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`${prefix}-patching-solid3-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.patchingSolid3 = {
            name: 'Patching Solid 3 (LEBAR)',
            thickness: thickness,
            width: solid3Width,
            length: width,
            qty: qty * solid3Qty,
            m2: solid3M2,
            price: price,
            total: total,
            material: `SL.${initialFormData.solid_type}`
        };
    }
    
    // 4. Crossgrain (MDF) - ✅ OPTIONAL
    const crossgrainSelect = document.getElementById(`${prefix}-crossgrain-thickness-${index}`);
    const crossgrainQty = parseFloat(document.getElementById(`${prefix}-crossgrain-qty-${index}`).value) || 2;
    
    if (crossgrainSelect && crossgrainSelect.value) {
        const selectedOption = crossgrainSelect.options[crossgrainSelect.selectedIndex];
        const thickness = parseFloat(selectedOption.dataset.thickness);
        const price = parseFloat(selectedOption.dataset.price);
        
        const crossgrainM2 = calculateM2(width, length, qty * crossgrainQty);
        const total = crossgrainM2 * price;
        
        document.getElementById(`${prefix}-crossgrain-m2-${index}`).value = crossgrainM2.toFixed(4);
        document.getElementById(`${prefix}-crossgrain-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`${prefix}-crossgrain-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.crossgrain = {
            name: 'Crossgrain',
            thickness: thickness,
            width: width,
            length: length,
            qty: qty * crossgrainQty,
            m2: crossgrainM2,
            price: price,
            total: total,
            material: 'MDF'
        };
    }
    
    // 5. Veneer Face - ✅ OPTIONAL
    const veneerFacePattern = document.getElementById(`${prefix}-veneer-face-pattern-${index}`).value;
    const veneerFaceSelect = document.getElementById(`${prefix}-veneer-face-type-${index}`);
    const veneerFaceQty = parseFloat(document.getElementById(`${prefix}-veneer-face-qty-${index}`).value) || 1;
    
    if (veneerFaceSelect && veneerFaceSelect.value && veneerFaceQty > 0) {
        const selectedOption = veneerFaceSelect.options[veneerFaceSelect.selectedIndex];
        const veneerType = selectedOption.dataset.type;
        const priceLurus = parseFloat(selectedOption.dataset.priceLurus);
        const priceMotif = parseFloat(selectedOption.dataset.priceMotif);
        const price = veneerFacePattern === 'LURUS' ? priceLurus : priceMotif;
        
        const veneerFaceM2 = calculateM2(width, length, qty * veneerFaceQty);
        const total = veneerFaceM2 * price;
        
        document.getElementById(`${prefix}-veneer-face-m2-${index}`).value = veneerFaceM2.toFixed(4);
        document.getElementById(`${prefix}-veneer-face-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`${prefix}-veneer-face-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.veneerFace = {
            name: 'TV. FACE',
            thickness: 0.5,
            width: width,
            length: length,
            qty: qty * veneerFaceQty,
            m2: veneerFaceM2,
            price: price,
            total: total,
            material: `V.${veneerType}`,
            pattern: veneerFacePattern,
            veneerType: veneerType
        };
    }
    
    // 6. Veneer Back - ✅ OPTIONAL
    const veneerBackPattern = document.getElementById(`${prefix}-veneer-back-pattern-${index}`).value;
    const veneerBackSelect = document.getElementById(`${prefix}-veneer-back-type-${index}`);
    const veneerBackQty = parseFloat(document.getElementById(`${prefix}-veneer-back-qty-${index}`).value) || 1;
    
    if (veneerBackSelect && veneerBackSelect.value && veneerBackQty > 0) {
        const selectedOption = veneerBackSelect.options[veneerBackSelect.selectedIndex];
        const veneerType = selectedOption.dataset.type;
        const priceLurus = parseFloat(selectedOption.dataset.priceLurus);
        const priceMotif = parseFloat(selectedOption.dataset.priceMotif);
        const price = veneerBackPattern === 'LURUS' ? priceLurus : priceMotif;
        
        const veneerBackM2 = calculateM2(width, length, qty * veneerBackQty);
        const total = veneerBackM2 * price;
        
        document.getElementById(`${prefix}-veneer-back-m2-${index}`).value = veneerBackM2.toFixed(4);
        document.getElementById(`${prefix}-veneer-back-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`${prefix}-veneer-back-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.veneerBack = {
            name: 'TV. BACK',
            thickness: 0.5,
            width: width,
            length: length,
            qty: qty * veneerBackQty,
            m2: veneerBackM2,
            price: price,
            total: total,
            material: `V.${veneerType}`,
            pattern: veneerBackPattern,
            veneerType: veneerType
        };
    }
    
    document.getElementById(`total-${prefix}-panel-${index}`).textContent = formatRupiah(componentTotal);
    
    const componentName = prefix === 'sidel' ? 'SIDE PANEL L' : 'SIDE PANEL R';
    
    // ✅ SAVE WITH M³
    calculationData[`${prefix}-panel-${index}`] = {
        componentName: componentName,
        componentTotal: componentTotal,
        m3: m3,
        details: {
            width: width,
            length: length,
            qty: qty,
            m2: m2,
            thicknessTotal: thicknessTotal,
            coreThickness: coreThickness
        },
        subComponents: subComponents
    };
    
    if (prefix === 'sidel') {
        sidePanelLData = calculationData[`${prefix}-panel-${index}`];
        sessionStorage.setItem('bom_side_panel_l_data', JSON.stringify(sidePanelLData));
        console.log('✅ Side Panel L Data Auto-Saved:', sidePanelLData);
    }
    
    console.log(`✅ ${componentName} Calculation (Flexible):`, calculationData[`${prefix}-panel-${index}`]);
    
    updateComponentCard(index, 'filled', componentTotal);
    updateTotalM3AndBiayaProduksi();
    saveComponentValuesToDraft(index, componentName);
}
// ============================================
// ✅ CALCULATE BOTTOM PANEL - CORRECT THICKNESS MAPPING
// ============================================

function calculateBottomPanel(index) {
    console.log('=== CALCULATE BOTTOM PANEL - 4 COMPONENTS ===');
    
    const width = parseFloat(document.getElementById(`bottom-width-${index}`).value) || 0;
    const length = parseFloat(document.getElementById(`bottom-length-${index}`).value) || 0;
    const thicknessTotal = parseFloat(document.getElementById(`bottom-thickness-${index}`).value) || 0;
    const qty = parseFloat(document.getElementById(`bottom-qty-${index}`).value) || 1;
    
    // ✅ FLEXIBLE: Hanya cek width, length, thickness
    if (!width || !length || !thicknessTotal) {
        console.log('❌ Dimensi belum lengkap, skip calculation');
        return;
    }
    
    if (![16, 19, 26].includes(thicknessTotal)) {
        console.log('❌ Thickness tidak valid');
        return;
    }
    
    const m2 = calculateM2(width, length, qty);
    document.getElementById(`bottom-m2-${index}`).value = m2.toFixed(4);
    
    // ✅ CALCULATE M³
    const m3 = calculateM3(width, length, thicknessTotal, qty);
    
    // ✅ UPDATE M³ DISPLAY IN MODAL
    const m3Display = document.getElementById(`bottom-m3-${index}`);
    if (m3Display) {
        m3Display.value = m3.toFixed(6);
    }
    
    // ✅ BOTTOM PANEL CORE THICKNESS MAPPING:
    // 16mm → Core 15mm + Veneer Face 0.5mm + Veneer Back 0.5mm = 16mm
    // 19mm → Core 18mm + Veneer Face 0.5mm + Veneer Back 0.5mm = 19mm
    // 26mm → Core 25mm + Veneer Face 0.5mm + Veneer Back 0.5mm = 26mm
    const coreThickness = getCoreThickness(thicknessTotal);
    
    document.getElementById(`bottom-core-thickness-${index}`).value = coreThickness + 'mm';
    document.getElementById(`bottom-core-type-${index}`).value = initialFormData.panel_type;
    
    // ✅ DISPLAY THICKNESS BREAKDOWN
    console.log(`📊 Bottom Panel Thickness Breakdown:`);
    console.log(`   Total: ${thicknessTotal}mm`);
    console.log(`   Core: ${coreThickness}mm`);
    console.log(`   Veneer Face: 0.5mm`);
    console.log(`   Veneer Back: 0.5mm`);
    console.log(`   Formula: ${coreThickness} + 0.5 + 0.5 = ${coreThickness + 1}mm`);
    
    let componentTotal = 0;
    const subComponents = {};
    
    // ============================================
    // 1. CORE PANEL (ALWAYS CALCULATED)
    // ============================================
    const corePanel = getPanelsByType(initialFormData.panel_type)
        .find(p => parseFloat(p.thickness) === parseFloat(coreThickness));
    
    if (corePanel) {
        const coreQty = parseFloat(document.getElementById(`bottom-core-qty-${index}`).value) || 1;
        const coreM2 = calculateM2(width, length, qty * coreQty);
        const coreTotal = coreM2 * corePanel.price;
        
        document.getElementById(`bottom-core-m2-${index}`).value = coreM2.toFixed(4);
        document.getElementById(`bottom-core-price-${index}`).textContent = formatRupiah(corePanel.price);
        document.getElementById(`bottom-core-total-${index}`).textContent = formatRupiah(coreTotal);
        
        componentTotal += coreTotal;
        
        subComponents.corePanel = {
            name: 'Core Panel',
            thickness: coreThickness,
            width: width,
            length: length,
            qty: qty * coreQty,
            m2: coreM2,
            price: corePanel.price,
            total: coreTotal,
            material: initialFormData.panel_type
        };
        
        console.log(`✅ Core Panel (${coreThickness}mm):`, subComponents.corePanel);
    }
    
    // ============================================
    // 2. VENEER FACE (OPTIONAL - 0.5mm)
    // ============================================
    const veneerFacePattern = document.getElementById(`bottom-veneer-face-pattern-${index}`).value;
    const veneerFaceSelect = document.getElementById(`bottom-veneer-face-type-${index}`);
    const veneerFaceQty = parseFloat(document.getElementById(`bottom-veneer-face-qty-${index}`).value) || 0;
    
    if (veneerFaceSelect && veneerFaceSelect.value && veneerFaceQty > 0) {
        const selectedOption = veneerFaceSelect.options[veneerFaceSelect.selectedIndex];
        const veneerType = selectedOption.dataset.type;
        const priceLurus = parseFloat(selectedOption.dataset.priceLurus);
        const priceMotif = parseFloat(selectedOption.dataset.priceMotif);
        const price = veneerFacePattern === 'LURUS' ? priceLurus : priceMotif;
        
        const veneerFaceM2 = calculateM2(width, length, qty * veneerFaceQty);
        const total = veneerFaceM2 * price;
        
        document.getElementById(`bottom-veneer-face-m2-${index}`).value = veneerFaceM2.toFixed(4);
        document.getElementById(`bottom-veneer-face-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`bottom-veneer-face-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.veneerFace = {
            name: 'TV. FACE',
            thickness: 0.5,
            width: width,
            length: length,
            qty: qty * veneerFaceQty,
            m2: veneerFaceM2,
            price: price,
            total: total,
            material: `V.${veneerType}`,
            pattern: veneerFacePattern,
            veneerType: veneerType
        };
        
        console.log('✅ Veneer Face (0.5mm):', subComponents.veneerFace);
    } else {
        // Reset displays if qty = 0
        document.getElementById(`bottom-veneer-face-m2-${index}`).value = '';
        document.getElementById(`bottom-veneer-face-price-${index}`).textContent = 'Rp 0';
        document.getElementById(`bottom-veneer-face-total-${index}`).textContent = 'Rp 0';
    }
    
    // ============================================
    // 3. VENEER BACK (OPTIONAL - 0.5mm)
    // ============================================
    const veneerBackPattern = document.getElementById(`bottom-veneer-back-pattern-${index}`).value;
    const veneerBackSelect = document.getElementById(`bottom-veneer-back-type-${index}`);
    const veneerBackQty = parseFloat(document.getElementById(`bottom-veneer-back-qty-${index}`).value) || 0;
    
    if (veneerBackSelect && veneerBackSelect.value && veneerBackQty > 0) {
        const selectedOption = veneerBackSelect.options[veneerBackSelect.selectedIndex];
        const veneerType = selectedOption.dataset.type;
        const priceLurus = parseFloat(selectedOption.dataset.priceLurus);
        const priceMotif = parseFloat(selectedOption.dataset.priceMotif);
        const price = veneerBackPattern === 'LURUS' ? priceLurus : priceMotif;
        
        const veneerBackM2 = calculateM2(width, length, qty * veneerBackQty);
        const total = veneerBackM2 * price;
        
        document.getElementById(`bottom-veneer-back-m2-${index}`).value = veneerBackM2.toFixed(4);
        document.getElementById(`bottom-veneer-back-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`bottom-veneer-back-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.veneerBack = {
            name: 'TV. BACK',
            thickness: 0.5,
            width: width,
            length: length,
            qty: qty * veneerBackQty,
            m2: veneerBackM2,
            price: price,
            total: total,
            material: `V.${veneerType}`,
            pattern: veneerBackPattern,
            veneerType: veneerType
        };
        
        console.log('✅ Veneer Back (0.5mm):', subComponents.veneerBack);
    } else {
        // Reset displays if qty = 0
        document.getElementById(`bottom-veneer-back-m2-${index}`).value = '';
        document.getElementById(`bottom-veneer-back-price-${index}`).textContent = 'Rp 0';
        document.getElementById(`bottom-veneer-back-total-${index}`).textContent = 'Rp 0';
    }
    
    // ============================================
    // 4. E.BAND (OPTIONAL)
    // ============================================
    const ebandQty = parseFloat(document.getElementById(`bottom-eband-qty-${index}`).value) || 0;
    
    if (ebandQty > 0) {
        const ebandPricePerUnit = 3000; // Rp 3.000 per piece
        const ebandM2 = calculateM2(width, length, ebandQty); // M² based on panel dimensions × qty
        const ebandTotal = ebandM2 * ebandPricePerUnit;
        
        document.getElementById(`bottom-eband-m2-${index}`).value = ebandM2.toFixed(4);
        document.getElementById(`bottom-eband-price-${index}`).textContent = formatRupiah(ebandPricePerUnit);
        document.getElementById(`bottom-eband-total-${index}`).textContent = formatRupiah(ebandTotal);
        
        componentTotal += ebandTotal;
        
        subComponents.eband = {
            name: 'E.Band',
            thickness: 0, // E.Band tidak punya thickness
            width: width,
            length: length,
            qty: ebandQty,
            m2: ebandM2,
            pricePerUnit: ebandPricePerUnit,
            total: ebandTotal,
            material: 'E.Band'
        };
        
        console.log('✅ E.Band:', subComponents.eband);
    } else {
        // Reset displays if qty = 0
        document.getElementById(`bottom-eband-m2-${index}`).value = '';
        document.getElementById(`bottom-eband-price-${index}`).textContent = 'Rp 3.000';
        document.getElementById(`bottom-eband-total-${index}`).textContent = 'Rp 0';
    }
    
    // ============================================
    // TOTAL BOTTOM PANEL
    // ============================================
    document.getElementById(`total-bottom-panel-${index}`).textContent = formatRupiah(componentTotal);
    
    // ✅ SAVE WITH M³
    calculationData[`bottom-panel-${index}`] = {
        componentName: 'BOTTOM PANEL',
        componentTotal: componentTotal,
        m3: m3,
        details: {
            width: width,
            length: length,
            qty: qty,
            m2: m2,
            thicknessTotal: thicknessTotal,
            coreThickness: coreThickness
        },
        subComponents: subComponents
    };
    
    console.log('✅ Bottom Panel Calculation Complete:', calculationData[`bottom-panel-${index}`]);
    console.log(`💰 Total: ${formatRupiah(componentTotal)}`);
    console.log(`🧊 M³: ${m3.toFixed(6)}`);
    
    updateComponentCard(index, 'filled', componentTotal);
    updateTotalM3AndBiayaProduksi();
    saveComponentValuesToDraft(index, 'Bottom Panel');
}

// ============================================
// ✅ CALCULATE BACK PANEL - ONLY 3 SUB-COMPONENTS
// ============================================

function calculateBackPanel(index) {
    console.log('=== CALCULATE BACK PANEL - 3 COMPONENTS ===');
    
    const width = parseFloat(document.getElementById(`back-width-${index}`).value) || 0;
    const length = parseFloat(document.getElementById(`back-length-${index}`).value) || 0;
    const thicknessTotal = parseFloat(document.getElementById(`back-thickness-${index}`).value) || 0;
    const qty = parseFloat(document.getElementById(`back-qty-${index}`).value) || 1;
    
    // ✅ FLEXIBLE: Hanya cek width, length, thickness
    if (!width || !length || !thicknessTotal) {
        console.log('❌ Dimensi belum lengkap, skip calculation');
        return;
    }
    
    if (![6, 19, 26].includes(thicknessTotal)) {
        console.log('❌ Thickness tidak valid');
        return;
    }
    
    const m2 = calculateM2(width, length, qty);
    document.getElementById(`back-m2-${index}`).value = m2.toFixed(4);
    
    // ✅ CALCULATE M³
    const m3 = calculateM3(width, length, thicknessTotal, qty);
    
    // ✅ UPDATE M³ DISPLAY IN MODAL
    const m3Display = document.getElementById(`back-m3-${index}`);
    if (m3Display) {
        m3Display.value = m3.toFixed(6);
    }
    
    // ✅ BACK PANEL CORE THICKNESS MAPPING:
    // 6mm  → Core 5mm  + Veneer Face 0.5mm + Veneer Back 0.5mm = 6mm
    // 19mm → Core 18mm + Veneer Face 0.5mm + Veneer Back 0.5mm = 19mm
    // 26mm → Core 25mm + Veneer Face 0.5mm + Veneer Back 0.5mm = 26mm
    const coreThickness = getCoreThickness(thicknessTotal);
    
    // ✅ BACK PANEL: MDF or Plywood
    const backPanelType = document.getElementById(`back-panel-type-${index}`).value;
    
    if (!backPanelType) {
        console.log('❌ Back Panel Type belum dipilih');
        return;
    }
    
    document.getElementById(`back-core-thickness-${index}`).value = coreThickness + 'mm';
    document.getElementById(`back-core-type-${index}`).value = backPanelType;
    
    // ✅ DISPLAY THICKNESS BREAKDOWN
    console.log(`📊 Back Panel Thickness Breakdown:`);
    console.log(`   Total: ${thicknessTotal}mm`);
    console.log(`   Core: ${coreThickness}mm`);
    console.log(`   Veneer Face: 0.5mm`);
    console.log(`   Veneer Back: 0.5mm`);
    console.log(`   Formula: ${coreThickness} + 0.5 + 0.5 = ${coreThickness + 1}mm`);
    
    let componentTotal = 0;
    const subComponents = {};
    
    // ============================================
    // 1. CORE PANEL (MDF/Plywood - ALWAYS CALCULATED)
    // ============================================
    const corePanel = getPanelsByType(backPanelType)
        .find(p => parseFloat(p.thickness) === parseFloat(coreThickness));
    
    if (corePanel) {
        const coreQty = parseFloat(document.getElementById(`back-core-qty-${index}`).value) || 1;
        const coreM2 = calculateM2(width, length, qty * coreQty);
        const coreTotal = coreM2 * corePanel.price;
        
        document.getElementById(`back-core-m2-${index}`).value = coreM2.toFixed(4);
        document.getElementById(`back-core-price-${index}`).textContent = formatRupiah(corePanel.price);
        document.getElementById(`back-core-total-${index}`).textContent = formatRupiah(coreTotal);
        
        componentTotal += coreTotal;
        
        subComponents.corePanel = {
            name: 'Core Panel',
            thickness: coreThickness,
            width: width,
            length: length,
            qty: qty * coreQty,
            m2: coreM2,
            price: corePanel.price,
            total: coreTotal,
            material: backPanelType
        };
        
        console.log(`✅ Core Panel (${backPanelType} ${coreThickness}mm):`, subComponents.corePanel);
    }
    
    // ============================================
    // 2. VENEER FACE (0.5mm)
    // ============================================
    const veneerFacePattern = document.getElementById(`back-veneer-face-pattern-${index}`).value;
    const veneerFaceSelect = document.getElementById(`back-veneer-face-type-${index}`);
    const veneerFaceQty = parseFloat(document.getElementById(`back-veneer-face-qty-${index}`).value) || 0;
    
    if (veneerFaceSelect && veneerFaceSelect.value && veneerFaceQty > 0) {
        const selectedOption = veneerFaceSelect.options[veneerFaceSelect.selectedIndex];
        const veneerType = selectedOption.dataset.type;
        const priceLurus = parseFloat(selectedOption.dataset.priceLurus);
        const priceMotif = parseFloat(selectedOption.dataset.priceMotif);
        const price = veneerFacePattern === 'LURUS' ? priceLurus : priceMotif;
        
        const veneerFaceM2 = calculateM2(width, length, qty * veneerFaceQty);
        const total = veneerFaceM2 * price;
        
        document.getElementById(`back-veneer-face-m2-${index}`).value = veneerFaceM2.toFixed(4);
        document.getElementById(`back-veneer-face-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`back-veneer-face-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.veneerFace = {
            name: 'TV. FACE',
            thickness: 0.5,
            width: width,
            length: length,
            qty: qty * veneerFaceQty,
            m2: veneerFaceM2,
            price: price,
            total: total,
            material: `V.${veneerType}`,
            pattern: veneerFacePattern,
            veneerType: veneerType
        };
        
        console.log('✅ Veneer Face (0.5mm):', subComponents.veneerFace);
    } else {
        // Reset displays if qty = 0
        document.getElementById(`back-veneer-face-m2-${index}`).value = '';
        document.getElementById(`back-veneer-face-price-${index}`).textContent = 'Rp 0';
        document.getElementById(`back-veneer-face-total-${index}`).textContent = 'Rp 0';
    }
    
    // ============================================
    // 3. VENEER BACK (0.5mm)
    // ============================================
    const veneerBackPattern = document.getElementById(`back-veneer-back-pattern-${index}`).value;
    const veneerBackSelect = document.getElementById(`back-veneer-back-type-${index}`);
    const veneerBackQty = parseFloat(document.getElementById(`back-veneer-back-qty-${index}`).value) || 0;
    
    if (veneerBackSelect && veneerBackSelect.value && veneerBackQty > 0) {
        const selectedOption = veneerBackSelect.options[veneerBackSelect.selectedIndex];
        const veneerType = selectedOption.dataset.type;
        const priceLurus = parseFloat(selectedOption.dataset.priceLurus);
        const priceMotif = parseFloat(selectedOption.dataset.priceMotif);
        const price = veneerBackPattern === 'LURUS' ? priceLurus : priceMotif;
        
        const veneerBackM2 = calculateM2(width, length, qty * veneerBackQty);
        const total = veneerBackM2 * price;
        
        document.getElementById(`back-veneer-back-m2-${index}`).value = veneerBackM2.toFixed(4);
        document.getElementById(`back-veneer-back-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`back-veneer-back-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.veneerBack = {
            name: 'TV. BACK',
            thickness: 0.5,
            width: width,
            length: length,
            qty: qty * veneerBackQty,
            m2: veneerBackM2,
            price: price,
            total: total,
            material: `V.${veneerType}`,
            pattern: veneerBackPattern,
            veneerType: veneerType
        };
        
        console.log('✅ Veneer Back (0.5mm):', subComponents.veneerBack);
    } else {
        // Reset displays if qty = 0
        document.getElementById(`back-veneer-back-m2-${index}`).value = '';
        document.getElementById(`back-veneer-back-price-${index}`).textContent = 'Rp 0';
        document.getElementById(`back-veneer-back-total-${index}`).textContent = 'Rp 0';
    }
    
    // ============================================
    // TOTAL BACK PANEL
    // ============================================
    document.getElementById(`total-back-panel-${index}`).textContent = formatRupiah(componentTotal);
    
    // ✅ SAVE WITH M³
    calculationData[`back-panel-${index}`] = {
        componentName: 'BACK PANEL',
        componentTotal: componentTotal,
        m3: m3,
        details: {
            width: width,
            length: length,
            qty: qty,
            m2: m2,
            thicknessTotal: thicknessTotal,
            coreThickness: coreThickness,
            backPanelType: backPanelType
        },
        subComponents: subComponents
    };
    
    console.log('✅ Back Panel Calculation Complete:', calculationData[`back-panel-${index}`]);
    console.log(`💰 Total: ${formatRupiah(componentTotal)}`);
    console.log(`🧊 M³: ${m3.toFixed(6)}`);
    
    updateComponentCard(index, 'filled', componentTotal);
    updateTotalM3AndBiayaProduksi();
    saveComponentValuesToDraft(index, 'Back Panel');
}
// ============================================
// ✅ CALCULATE FRAME (FRONT/BACK) - ONLY SOLID
// ============================================

function calculateFrame(index, frameType) {
    console.log(`=== CALCULATE ${frameType.toUpperCase()} FRAME - ONLY SOLID ===`);
    
    const prefix = frameType === 'front' ? 'front' : 'back';
    
    const width = parseFloat(document.getElementById(`${prefix}-width-${index}`).value) || 0;
    const length = parseFloat(document.getElementById(`${prefix}-length-${index}`).value) || 0;
    const solidSelect = document.getElementById(`${prefix}-solid-thickness-${index}`);
    const solidQty = parseFloat(document.getElementById(`${prefix}-solid-qty-${index}`).value) || 1;
    
    // ✅ FLEXIBLE: Hanya cek width, length, solid thickness
    if (!width || !length || !solidSelect || !solidSelect.value) {
        console.log('❌ Dimensi atau solid belum lengkap, skip calculation');
        return;
    }
    
    const selectedOption = solidSelect.options[solidSelect.selectedIndex];
    const solidThickness = parseFloat(selectedOption.dataset.thickness);
    const solidPrice = parseFloat(selectedOption.dataset.price);
    
    // ✅ CALCULATE M²
    const m2 = calculateM2(width, length, solidQty);
    document.getElementById(`${prefix}-m2-${index}`).value = m2.toFixed(4);
    
    // ✅ CALCULATE M³
    const m3 = calculateM3(width, length, solidThickness, solidQty);
    
    // ✅ UPDATE M³ DISPLAY IN MODAL
    const m3Display = document.getElementById(`${prefix}-m3-${index}`);
    if (m3Display) {
        m3Display.value = m3.toFixed(6);
    }
    
    // ✅ DISPLAY CALCULATION INFO
    console.log(`📊 ${frameType.toUpperCase()} Frame Calculation:`);
    console.log(`   Width: ${width}mm`);
    console.log(`   Length: ${length}mm`);
    console.log(`   Thickness: ${solidThickness}mm`);
    console.log(`   Qty: ${solidQty}`);
    console.log(`   M²: ${m2.toFixed(4)}`);
    console.log(`   M³: ${m3.toFixed(6)}`);
    
    let componentTotal = 0;
    const subComponents = {};
    
    // ============================================
    // CORE SOLID (ONLY COMPONENT)
    // ============================================
    const solidTotal = m2 * solidPrice;
    
    document.getElementById(`${prefix}-solid-m2-${index}`).value = m2.toFixed(4);
    document.getElementById(`${prefix}-solid-price-${index}`).textContent = formatRupiah(solidPrice);
    document.getElementById(`${prefix}-solid-total-${index}`).textContent = formatRupiah(solidTotal);
    
    componentTotal = solidTotal;
    
    subComponents.solid = {
        name: 'Core Solid',
        thickness: solidThickness,
        width: width,
        length: length,
        qty: solidQty,
        m2: m2,
        price: solidPrice,
        total: solidTotal,
        material: `SL.${initialFormData.solid_type}`
    };
    
    console.log('✅ Core Solid:', subComponents.solid);
    console.log(`💰 Total: ${formatRupiah(componentTotal)}`);
    
    document.getElementById(`total-${prefix}-frame-${index}`).textContent = formatRupiah(componentTotal);
    
    const componentName = frameType === 'front' ? 'FRONT FRAME' : 'BACK FRAME';
    
    // ✅ SAVE WITH M³
    calculationData[`${prefix}-frame-${index}`] = {
        componentName: componentName,
        componentTotal: componentTotal,
        m3: m3,
        details: {
            width: width,
            length: length,
            solidThickness: solidThickness,
            qty: solidQty,
            m2: m2
        },
        subComponents: subComponents
    };
    
    console.log(`✅ ${componentName} Calculation Complete:`, calculationData[`${prefix}-frame-${index}`]);
    
    updateComponentCard(index, 'filled', componentTotal);
    updateTotalM3AndBiayaProduksi();
    saveComponentValuesToDraft(index, componentName);
}

// ============================================
// ✅ CALCULATE FACE DRAWER - FLEXIBLE INPUT (AUTO CALCULATE) + M³ DISPLAY
// ============================================

function calculateFaceDrawer(index) {
    console.log('=== CALCULATE FACE DRAWER - FLEXIBLE ===');
    
    const width = parseFloat(document.getElementById(`facedrawer-width-${index}`).value) || 0;
    const length = parseFloat(document.getElementById(`facedrawer-length-${index}`).value) || 0;
    const thicknessTotal = parseFloat(document.getElementById(`facedrawer-thickness-${index}`).value) || 0;
    const qty = parseFloat(document.getElementById(`facedrawer-qty-${index}`).value) || 1;
    
    // ✅ FLEXIBLE: Hanya cek width, length, thickness
    if (!width || !length || !thicknessTotal) {
        console.log('❌ Dimensi belum lengkap, skip calculation');
        return;
    }
    
    if (![18, 21, 28].includes(thicknessTotal)) {
        console.log('❌ Thickness tidak valid');
        return;
    }
    
    faceDrawerQty = qty;
    sessionStorage.setItem('bom_face_drawer_qty', faceDrawerQty);
    
    const m2 = calculateM2(width, length, qty);
    document.getElementById(`facedrawer-m2-${index}`).value = m2.toFixed(4);
    
    // ✅ CALCULATE M³
    const m3 = calculateM3(width, length, thicknessTotal, qty);
    
    // ✅ UPDATE M³ DISPLAY IN MODAL
    const m3Display = document.getElementById(`facedrawer-m3-${index}`);
    if (m3Display) {
        m3Display.value = m3.toFixed(6);
    }
    
    const coreThickness = getCoreThickness(thicknessTotal);
    
    document.getElementById(`facedrawer-core-thickness-${index}`).value = coreThickness + 'mm';
    document.getElementById(`facedrawer-core-type-${index}`).value = initialFormData.panel_type;
    
    let componentTotal = 0;
    const subComponents = {};
    
    // 1. Patching Panel 1 (Core Panel - ALWAYS CALCULATED)
    const corePanel = getPanelsByType(initialFormData.panel_type)
        .find(p => parseFloat(p.thickness) === parseFloat(coreThickness));
    
    if (corePanel) {
        const patchingPanelQty = parseFloat(document.getElementById(`facedrawer-patching-panel-qty-${index}`).value) || 1;
        const patchingM2 = calculateM2(width, length, qty * patchingPanelQty);
        const patchingTotal = patchingM2 * corePanel.price;
        
        document.getElementById(`facedrawer-patching-panel-m2-${index}`).value = patchingM2.toFixed(4);
        document.getElementById(`facedrawer-patching-panel-price-${index}`).textContent = formatRupiah(corePanel.price);
        document.getElementById(`facedrawer-patching-panel-total-${index}`).textContent = formatRupiah(patchingTotal);
        
        componentTotal += patchingTotal;
        
        subComponents.patchingPanel = {
            name: 'Patching Panel 1',
            thickness: coreThickness,
            width: width,
            length: length,
            qty: qty * patchingPanelQty,
            m2: patchingM2,
            price: corePanel.price,
            total: patchingTotal,
            material: initialFormData.panel_type
        };
    }
    
    // 2. Patching Solid 2 (PANJANG) - ✅ OPTIONAL
    const solid2Width = parseFloat(document.getElementById(`facedrawer-patching-solid2-width-${index}`).value) || 0;
    const solid2Select = document.getElementById(`facedrawer-patching-solid2-thickness-${index}`);
    const solid2Qty = parseFloat(document.getElementById(`facedrawer-patching-solid2-qty-${index}`).value) || 2;
    
    if (solid2Select && solid2Select.value && solid2Width > 0) {
        const selectedOption = solid2Select.options[solid2Select.selectedIndex];
        const thickness = parseFloat(selectedOption.dataset.thickness);
        const price = parseFloat(selectedOption.dataset.price);
        
        const solid2M2 = calculateM2(solid2Width, length, qty * solid2Qty);
        const total = solid2M2 * price;
        
        document.getElementById(`facedrawer-patching-solid2-m2-${index}`).value = solid2M2.toFixed(4);
        document.getElementById(`facedrawer-patching-solid2-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`facedrawer-patching-solid2-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.patchingSolid2 = {
            name: 'Patching Solid 2 (PANJANG)',
            thickness: thickness,
            width: solid2Width,
            length: length,
            qty: qty * solid2Qty,
            m2: solid2M2,
            price: price,
            total: total,
            material: `SL.${initialFormData.solid_type}`
        };
    }
    
    // 3. Patching Solid 3 (LEBAR) - ✅ OPTIONAL
    const solid3Width = parseFloat(document.getElementById(`facedrawer-patching-solid3-width-${index}`).value) || 0;
    const solid3Select = document.getElementById(`facedrawer-patching-solid3-thickness-${index}`);
    const solid3Qty = parseFloat(document.getElementById(`facedrawer-patching-solid3-qty-${index}`).value) || 2;
    
    if (solid3Select && solid3Select.value && solid3Width > 0) {
        const selectedOption = solid3Select.options[solid3Select.selectedIndex];
        const thickness = parseFloat(selectedOption.dataset.thickness);
        const price = parseFloat(selectedOption.dataset.price);
        
        const solid3M2 = calculateM2(solid3Width, width, qty * solid3Qty);
        const total = solid3M2 * price;
        
        document.getElementById(`facedrawer-patching-solid3-m2-${index}`).value = solid3M2.toFixed(4);
        document.getElementById(`facedrawer-patching-solid3-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`facedrawer-patching-solid3-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.patchingSolid3 = {
            name: 'Patching Solid 3 (LEBAR)',
            thickness: thickness,
            width: solid3Width,
            length: width,
            qty: qty * solid3Qty,
            m2: solid3M2,
            price: price,
            total: total,
            material: `SL.${initialFormData.solid_type}`
        };
    }
    
    // 4. Crossgrain (MDF) - ✅ OPTIONAL
    const crossgrainSelect = document.getElementById(`facedrawer-crossgrain-thickness-${index}`);
    const crossgrainQty = parseFloat(document.getElementById(`facedrawer-crossgrain-qty-${index}`).value) || 2;
    
    if (crossgrainSelect && crossgrainSelect.value) {
        const selectedOption = crossgrainSelect.options[crossgrainSelect.selectedIndex];
        const thickness = parseFloat(selectedOption.dataset.thickness);
        const price = parseFloat(selectedOption.dataset.price);
        
        const crossgrainM2 = calculateM2(width, length, qty * crossgrainQty);
        const total = crossgrainM2 * price;
        
        document.getElementById(`facedrawer-crossgrain-m2-${index}`).value = crossgrainM2.toFixed(4);
        document.getElementById(`facedrawer-crossgrain-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`facedrawer-crossgrain-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.crossgrain = {
            name: 'Crossgrain',
            thickness: thickness,
            width: width,
            length: length,
            qty: qty * crossgrainQty,
            m2: crossgrainM2,
            price: price,
            total: total,
            material: 'MDF'
        };
    }
    
    // 5. Veneer Face - ✅ OPTIONAL
    const veneerFacePattern = document.getElementById(`facedrawer-veneer-face-pattern-${index}`).value;
    const veneerFaceSelect = document.getElementById(`facedrawer-veneer-face-type-${index}`);
    const veneerFaceQty = parseFloat(document.getElementById(`facedrawer-veneer-face-qty-${index}`).value) || 1;
    
    if (veneerFaceSelect && veneerFaceSelect.value && veneerFaceQty > 0) {
        const selectedOption = veneerFaceSelect.options[veneerFaceSelect.selectedIndex];
        const veneerType = selectedOption.dataset.type;
        const priceLurus = parseFloat(selectedOption.dataset.priceLurus);
        const priceMotif = parseFloat(selectedOption.dataset.priceMotif);
        const price = veneerFacePattern === 'LURUS' ? priceLurus : priceMotif;
        
        const veneerFaceM2 = calculateM2(width, length, qty * veneerFaceQty);
        const total = veneerFaceM2 * price;
        
        document.getElementById(`facedrawer-veneer-face-m2-${index}`).value = veneerFaceM2.toFixed(4);
        document.getElementById(`facedrawer-veneer-face-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`facedrawer-veneer-face-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.veneerFace = {
            name: 'TV. FACE',
            thickness: 0.5,
            width: width,
            length: length,
            qty: qty * veneerFaceQty,
            m2: veneerFaceM2,
            price: price,
            total: total,
            material: `V.${veneerType}`,
            pattern: veneerFacePattern,
            veneerType: veneerType
        };
    }
    
    // 6. Veneer Back - ✅ OPTIONAL
    const veneerBackPattern = document.getElementById(`facedrawer-veneer-back-pattern-${index}`).value;
    const veneerBackSelect = document.getElementById(`facedrawer-veneer-back-type-${index}`);
    const veneerBackQty = parseFloat(document.getElementById(`facedrawer-veneer-back-qty-${index}`).value) || 1;
    
    if (veneerBackSelect && veneerBackSelect.value && veneerBackQty > 0) {
        const selectedOption = veneerBackSelect.options[veneerBackSelect.selectedIndex];
        const veneerType = selectedOption.dataset.type;
        const priceLurus = parseFloat(selectedOption.dataset.priceLurus);
        const priceMotif = parseFloat(selectedOption.dataset.priceMotif);
        const price = veneerBackPattern === 'LURUS' ? priceLurus : priceMotif;
        
        const veneerBackM2 = calculateM2(width, length, qty * veneerBackQty);
        const total = veneerBackM2 * price;
        
        document.getElementById(`facedrawer-veneer-back-m2-${index}`).value = veneerBackM2.toFixed(4);
        document.getElementById(`facedrawer-veneer-back-price-${index}`).textContent = formatRupiah(price);
        document.getElementById(`facedrawer-veneer-back-total-${index}`).textContent = formatRupiah(total);
        
        componentTotal += total;
        
        subComponents.veneerBack = {
            name: 'TV. BACK',
            thickness: 0.5,
            width: width,
            length: length,
            qty: qty * veneerBackQty,
            m2: veneerBackM2,
            price: price,
            total: total,
            material: `V.${veneerType}`,
            pattern: veneerBackPattern,
            veneerType: veneerType
        };
    }
    
    document.getElementById(`total-facedrawer-${index}`).textContent = formatRupiah(componentTotal);
    
    // ✅ SAVE WITH M³
    calculationData[`facedrawer-${index}`] = {
        componentName: 'FACE DRAWER',
        componentTotal: componentTotal,
        m3: m3,
        details: {
            width: width,
            length: length,
            qty: qty,
            m2: m2,
            thicknessTotal: thicknessTotal,
            coreThickness: coreThickness
        },
        subComponents: subComponents
    };
    
    console.log('✅ Face Drawer Calculation (Flexible):', calculationData[`facedrawer-${index}`]);
    
    updateComponentCard(index, 'filled', componentTotal);
    updateTotalM3AndBiayaProduksi();
    saveComponentValuesToDraft(index, 'Face Drawer');
}
// ============================================
// ✅ CALCULATE DRAWER (FRONT/SIDE/BACK/BASE) - FLEXIBLE INPUT (AUTO CALCULATE) + M³ DISPLAY
// ============================================

function calculateDrawer(index, prefix) {
    console.log(`=== CALCULATE DRAWER (${prefix.toUpperCase()}) - FLEXIBLE ===`);
    
    const dimensionSelect = document.getElementById(`${prefix}-dimension-${index}`);
    
    // ✅ FLEXIBLE: Hanya cek dimension select
    if (!dimensionSelect || !dimensionSelect.value) {
        console.log('❌ Dimension belum dipilih, skip calculation');
        return;
    }
    
    const selectedOption = dimensionSelect.options[dimensionSelect.selectedIndex];
    const drawerType = selectedOption.dataset.drawerType;
    const width = parseFloat(selectedOption.dataset.width);
    const length = parseFloat(selectedOption.dataset.length);
    const price = parseFloat(selectedOption.dataset.price);
    
    // ✅ THICKNESS 11mm UNTUK SEMUA DRAWER
    const thickness = 11;
    
    // Qty mengikuti Face Drawer Qty
    const qty = faceDrawerQty;
    
    document.getElementById(`${prefix}-qty-display-${index}`).value = qty;
    
    const m2 = calculateM2(width, length, qty);
    document.getElementById(`${prefix}-m2-${index}`).value = m2.toFixed(4);
    
    // ✅ CALCULATE M³ (THICKNESS = 11mm)
    const m3 = calculateM3(width, length, thickness, qty);
    
    // ✅ UPDATE M³ DISPLAY IN MODAL
    const m3Display = document.getElementById(`${prefix}-m3-${index}`);
    if (m3Display) {
        m3Display.value = m3.toFixed(6);
    }
    
    const total = m2 * price;
    
    document.getElementById(`${prefix}-price-${index}`).textContent = formatRupiah(price);
    document.getElementById(`total-${prefix}-${index}`).textContent = formatRupiah(total);
    
    // ✅ SAVE WITH M³
    calculationData[`${prefix}-${index}`] = {
        componentName: getComponentName(index),
        componentTotal: total,
        m3: m3,
        details: {
            drawerType: drawerType,
            width: width,
            length: length,
            thickness: thickness,
            qty: qty,
            m2: m2,
            price: price,
            material: initialFormData.box_drawer_type
        }
    };
    
    // ✅ AUTO-SAVE FOR SIDE DRAWER L
    if (prefix === 'sidedrawerl') {
        sideDrawerLData = calculationData[`${prefix}-${index}`];
        sessionStorage.setItem('bom_side_drawer_l_data', JSON.stringify(sideDrawerLData));
        console.log('✅ Side Drawer L Data Auto-Saved:', sideDrawerLData);
    }
    
    console.log(`✅ ${getComponentName(index)} Calculation (Flexible):`, calculationData[`${prefix}-${index}`]);
    
    updateComponentCard(index, 'filled', total);
    updateTotalM3AndBiayaProduksi();
    saveComponentValuesToDraft(index, getComponentName(index));
}

// ============================================
// ✅ CALCULATE SUPPORT DRAWER - FIXED PRICE Rp 2.200 (M³ = 0)
// ============================================

function calculateSupportDrawer(index) {
    console.log('=== CALCULATE SUPPORT DRAWER ===');
    
    const qtyInput = document.getElementById(`support-drawer-qty-${index}`);
    const qty = parseFloat(qtyInput.value) || 0;
    
    if (qty <= 0) {
        console.log('❌ Qty belum diisi, skip calculation');
        return;
    }
    
    const pricePerUnit = 2200; // ✅ FIXED: Rp 2.200
    const total = qty * pricePerUnit;
    
    document.getElementById(`support-drawer-price-${index}`).textContent = formatRupiah(pricePerUnit);
    document.getElementById(`total-support-drawer-${index}`).textContent = formatRupiah(total);
    
    // ✅ SAVE WITH M³ = 0
    calculationData[`support-drawer-${index}`] = {
        componentName: 'SUPPORT DRAWER',
        componentTotal: total,
        m3: 0, // ✅ M³ = 0 untuk Support Drawer
        details: {
            qty: qty,
            pricePerUnit: pricePerUnit,
            material: 'Mix'
        }
    };
    
    console.log('✅ Support Drawer Calculation:', calculationData[`support-drawer-${index}`]);
    console.log(`   Qty: ${qty} pieces`);
    console.log(`   Price per Unit: ${formatRupiah(pricePerUnit)}`);
    console.log(`   Total: ${formatRupiah(total)}`);
    
    updateComponentCard(index, 'filled', total);
    updateTotalM3AndBiayaProduksi();
    saveComponentValuesToDraft(index, 'Support Drawer');
}

// ============================================
// ✅ CALCULATE CLEAT DRAWER - FIXED PRICE Rp 400 (M³ = 0)
// ============================================

function calculateCleatDrawer(index) {
    console.log('=== CALCULATE CLEAT DRAWER ===');
    
    const qtyInput = document.getElementById(`cleat-drawer-qty-${index}`);
    const qty = parseFloat(qtyInput.value) || 0;
    
    if (qty <= 0) {
        console.log('❌ Qty belum diisi, skip calculation');
        return;
    }
    
    const pricePerUnit = 400; // ✅ FIXED: Rp 400
    const total = qty * pricePerUnit;
    
    document.getElementById(`cleat-drawer-price-${index}`).textContent = formatRupiah(pricePerUnit);
    document.getElementById(`total-cleat-drawer-${index}`).textContent = formatRupiah(total);
    
    // ✅ SAVE WITH M³ = 0
    calculationData[`cleat-drawer-${index}`] = {
        componentName: 'CLEAT DRAWER',
        componentTotal: total,
        m3: 0, // ✅ M³ = 0 untuk Cleat Drawer
        details: {
            qty: qty,
            pricePerUnit: pricePerUnit,
            material: 'Mix'
        }
    };
    
    console.log('✅ Cleat Drawer Calculation:', calculationData[`cleat-drawer-${index}`]);
    console.log(`   Qty: ${qty} pieces`);
    console.log(`   Price per Unit: ${formatRupiah(pricePerUnit)}`);
    console.log(`   Total: ${formatRupiah(total)}`);
    
    updateComponentCard(index, 'filled', total);
    updateTotalM3AndBiayaProduksi();
    saveComponentValuesToDraft(index, 'Cleat Drawer');
}

// ============================================
// ✅ BACK PANEL THICKNESS CHANGE HANDLER
// ============================================

function onBackPanelThicknessChange(index) {
    const thickness = parseFloat(document.getElementById(`back-thickness-${index}`).value);
    const veneerSection = document.getElementById(`back-veneer-section-${index}`);
    
    if ([19, 26].includes(thickness)) {
        veneerSection.style.display = 'block';
    } else {
        veneerSection.style.display = 'none';
    }
    
    // Trigger auto-calculate
    setTimeout(() => {
        calculateBackPanel(index);
    }, 100);
}

// ============================================
// AUTO-CALCULATE LISTENERS (ONINPUT)
// ============================================

// Fungsi untuk attach auto-calculate listeners
function attachAutoCalculateListeners(index, componentType) {
    console.log(`Attaching auto-calculate listeners for ${componentType} index ${index}`);
    
    const modalBody = document.getElementById('modalBody');
    if (!modalBody) return;
    
    const inputs = modalBody.querySelectorAll('input[type="number"], input[type="text"], select');
    
    inputs.forEach(input => {
        // Remove old listeners first
        const newInput = input.cloneNode(true);
        input.parentNode.replaceChild(newInput, input);
        
        // Add new listener
        newInput.addEventListener('input', function() {
            console.log(`Input changed: ${this.id}`);
            
            // Small delay untuk user experience
            setTimeout(() => {
                switch(componentType) {
                    case 'Top Panel':
                        calculateTopPanel(index);
                        break;
                    case 'Side Panel L':
                        calculateSidePanel(index, 'sidel');
                        break;
                    case 'Side Panel R':
                        const siderMode = document.getElementById(`sider-mode-${index}`)?.value;
                        if (siderMode === 'manual') {
                            calculateSidePanel(index, 'sider');
                        }
                        break;
                    case 'Bottom Panel':
                        calculateBottomPanel(index);
                        break;
                    case 'Back Panel':
                        calculateBackPanel(index);
                        break;
                    case 'Front Frame':
                        calculateFrame(index, 'front');
                        break;
                    case 'Back Frame':
                        calculateFrame(index, 'back');
                        break;
                    case 'Face Drawer':
                        calculateFaceDrawer(index);
                        break;
                    case 'Front Drawer':
                        calculateDrawer(index, 'frontdrawer');
                        break;
                    case 'Side Drawer L':
                        calculateDrawer(index, 'sidedrawerl');
                        break;
                    case 'Side Drawer R':
                        const sidedrawerrMode = document.getElementById(`sidedrawerr-mode-${index}`)?.value;
                        if (sidedrawerrMode === 'manual') {
                            calculateDrawer(index, 'sidedrawerr');
                        }
                        break;
                    case 'Back Drawer':
                        calculateDrawer(index, 'backdrawer');
                        break;
                    case 'Base Drawer':
                        calculateDrawer(index, 'basedrawer');
                        break;
                    case 'Support Drawer':
                        calculateSupportDrawer(index);
                        break;
                    case 'Cleat Drawer':
                        calculateCleatDrawer(index);
                        break;
                }
            }, 300); // 300ms delay
        });
        
        // Special handling for select dropdowns
        if (newInput.tagName === 'SELECT') {
            newInput.addEventListener('change', function() {
                console.log(`Select changed: ${this.id}`);
                
                setTimeout(() => {
                    switch(componentType) {
                        case 'Top Panel':
                            calculateTopPanel(index);
                            break;
                        case 'Side Panel L':
                            calculateSidePanel(index, 'sidel');
                            break;
                        case 'Side Panel R':
                            const siderMode = document.getElementById(`sider-mode-${index}`)?.value;
                            if (siderMode === 'manual') {
                                calculateSidePanel(index, 'sider');
                            }
                            break;
                        case 'Bottom Panel':
                            calculateBottomPanel(index);
                            break;
                        case 'Back Panel':
                            calculateBackPanel(index);
                            break;
                        case 'Front Frame':
                            calculateFrame(index, 'front');
                            break;
                        case 'Back Frame':
                            calculateFrame(index, 'back');
                            break;
                        case 'Face Drawer':
                            calculateFaceDrawer(index);
                            break;
                        case 'Front Drawer':
                            calculateDrawer(index, 'frontdrawer');
                            break;
                        case 'Side Drawer L':
                            calculateDrawer(index, 'sidedrawerl');
                            break;
                        case 'Side Drawer R':
                            const sidedrawerrMode = document.getElementById(`sidedrawerr-mode-${index}`)?.value;
                            if (sidedrawerrMode === 'manual') {
                                calculateDrawer(index, 'sidedrawerr');
                            }
                            break;
                        case 'Back Drawer':
                            calculateDrawer(index, 'backdrawer');
                            break;
                        case 'Base Drawer':
                            calculateDrawer(index, 'basedrawer');
                            break;
                        case 'Support Drawer':
                            calculateSupportDrawer(index);
                            break;
                        case 'Cleat Drawer':
                            calculateCleatDrawer(index);
                            break;
                    }
                }, 100);
            });
        }
    });
}
// ============================================
// ✅ GET TOP PANEL CONTENT - WITH M³ DISPLAY
// ============================================

function getTopPanelContent(index) {
    const panelType = initialFormData.panel_type;
    const solidType = initialFormData.solid_type;
    
    const mdfPanels = getPanelsByType('MDF');
    const solids = getSolidsByType(solidType);
    const allVeneers = getAllVeneers();
    
    setTimeout(() => {
        attachAutoCalculateListeners(index, 'Top Panel');
    }, 500);
    
    return `
        <div class="alert alert-info alert-sm">
            <i class="fas fa-lock me-2"></i>
            <strong>Material Locked:</strong> Panel = ${panelType} | Solid = SL.${solidType}
        </div>

        <!-- Main Dimensions -->
        <div class="card mb-3 border-primary">
            <div class="card-header bg-primary text-white">
                <strong>Main Dimensions</strong>
            </div>
            <div class="card-body">
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Width (mm) <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="top-width-${index}" placeholder="Width">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Length (mm) <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="top-length-${index}" placeholder="Length">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Thickness Total (mm) <span class="text-danger">*</span></label>
                        <select class="form-select" id="top-thickness-${index}">
                            <option value="">-- Pilih --</option>
                            <option value="18">18mm</option>
                            <option value="21">21mm</option>
                            <option value="28">28mm</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Qty <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="top-qty-${index}" value="1" min="1">
                    </div>
                </div>

                <!-- ✅ M² & M³ DISPLAY ROW -->
                <div class="row g-3 mb-4">
                    <div class="col-md-6">
                        <label class="form-label">M² Total (Auto-calculated)</label>
                        <input type="text" class="form-control bg-light" id="top-m2-${index}" readonly>
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">M³ Total (Auto-calculated)</label>
                        <div class="input-group">
                            <input type="text" class="form-control bg-warning text-dark fw-bold" id="top-m3-${index}" readonly>
                            <span class="input-group-text">m³</span>
                        </div>
                    </div>
                </div>

                <div class="row g-3">
                    <div class="col-md-6">
                        <label class="form-label">Core Thickness</label>
                        <input type="text" class="form-control bg-light" id="top-core-thickness-${index}" readonly>
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">Core Type</label>
                        <input type="text" class="form-control bg-light" id="top-core-type-${index}" readonly>
                    </div>
                </div>
            </div>
        </div>

        <!-- Sub Components -->
        <div class="sub-component-card">
            <h6 class="sub-component-title">1. Patching Panel 1 (Core Panel)</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-4">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="top-patching-panel-qty-${index}" value="1" min="1">
                </div>
                <div class="col-md-4">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="top-patching-panel-m2-${index}" readonly>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="top-patching-panel-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Patching Panel 1:</strong>
                <span class="float-end" id="top-patching-panel-total-${index}">Rp 0</span>
            </div>
        </div>

        <div class="sub-component-card">
            <h6 class="sub-component-title">2. Patching Solid 2 (PANJANG) - Optional</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-3">
                    <label class="form-label">Width (mm)</label>
                    <input type="number" class="form-control" id="top-patching-solid2-width-${index}" placeholder="Width">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Thickness</label>
                    <select class="form-select" id="top-patching-solid2-thickness-${index}">
                        <option value="">-- Pilih --</option>
                        ${solids.map(s => `<option value="${s.id}" data-thickness="${s.thickness}" data-price="${s.price}">${s.thickness}mm - ${formatRupiah(s.price)}/M²</option>`).join('')}
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="top-patching-solid2-qty-${index}" value="2" min="0">
                </div>
                <div class="col-md-2">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="top-patching-solid2-m2-${index}" readonly>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="top-patching-solid2-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Patching Solid 2:</strong>
                <span class="float-end" id="top-patching-solid2-total-${index}">Rp 0</span>
            </div>
        </div>

        <div class="sub-component-card">
            <h6 class="sub-component-title">3. Patching Solid 3 (LEBAR) - Optional</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-3">
                    <label class="form-label">Width (mm)</label>
                    <input type="number" class="form-control" id="top-patching-solid3-width-${index}" placeholder="Width">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Thickness</label>
                    <select class="form-select" id="top-patching-solid3-thickness-${index}">
                        <option value="">-- Pilih --</option>
                        ${solids.map(s => `<option value="${s.id}" data-thickness="${s.thickness}" data-price="${s.price}">${s.thickness}mm - ${formatRupiah(s.price)}/M²</option>`).join('')}
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="top-patching-solid3-qty-${index}" value="2" min="0">
                </div>
                <div class="col-md-2">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="top-patching-solid3-m2-${index}" readonly>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="top-patching-solid3-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Patching Solid 3:</strong>
                <span class="float-end" id="top-patching-solid3-total-${index}">Rp 0</span>
            </div>
        </div>

        <div class="sub-component-card">
            <h6 class="sub-component-title">4. Crossgrain (MDF) - Optional</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-4">
                    <label class="form-label">Thickness</label>
                    <select class="form-select" id="top-crossgrain-thickness-${index}">
                        <option value="">-- Pilih --</option>
                        ${mdfPanels.map(p => `<option value="${p.id}" data-thickness="${p.thickness}" data-price="${p.price}">${p.thickness}mm - ${formatRupiah(p.price)}/M²</option>`).join('')}
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="top-crossgrain-qty-${index}" value="2" min="0">
                </div>
                <div class="col-md-3">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="top-crossgrain-m2-${index}" readonly>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="top-crossgrain-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Crossgrain:</strong>
                <span class="float-end" id="top-crossgrain-total-${index}">Rp 0</span>
            </div>
        </div>

        <div class="sub-component-card">
            <h6 class="sub-component-title">5. Veneer Face - Optional</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-3">
                    <label class="form-label">Pattern</label>
                    <select class="form-select" id="top-veneer-face-pattern-${index}">
                        <option value="LURUS">LURUS</option>
                        <option value="MOTIF">MOTIF</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Veneer Type</label>
                    <select class="form-select" id="top-veneer-face-type-${index}">
                        <option value="">-- Pilih --</option>
                        ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">V.${v.veneer_type}</option>`).join('')}
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="top-veneer-face-qty-${index}" value="1" min="0">
                </div>
                <div class="col-md-2">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="top-veneer-face-m2-${index}" readonly>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="top-veneer-face-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Veneer Face:</strong>
                <span class="float-end" id="top-veneer-face-total-${index}">Rp 0</span>
            </div>
        </div>

        <div class="sub-component-card">
            <h6 class="sub-component-title">6. Veneer Back - Optional</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-3">
                    <label class="form-label">Pattern</label>
                    <select class="form-select" id="top-veneer-back-pattern-${index}">
                        <option value="LURUS">LURUS</option>
                        <option value="MOTIF">MOTIF</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Veneer Type</label>
                    <select class="form-select" id="top-veneer-back-type-${index}">
                        <option value="">-- Pilih --</option>
                        ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">V.${v.veneer_type}</option>`).join('')}
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="top-veneer-back-qty-${index}" value="1" min="0">
                </div>
                <div class="col-md-2">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="top-veneer-back-m2-${index}" readonly>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="top-veneer-back-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Veneer Back:</strong>
                <span class="float-end" id="top-veneer-back-total-${index}">Rp 0</span>
            </div>
        </div>

        <div class="sub-component-card">
            <h6 class="sub-component-title">7. Cleat - Optional</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-4">
                    <label class="form-label">Qty (pieces)</label>
                    <input type="number" class="form-control" id="top-cleat-qty-${index}" value="0" min="0">
                </div>
                <div class="col-md-4">
                    <label class="form-label">Price per Unit</label>
                    <div class="calculated-info">Rp 1.000</div>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Total</label>
                    <div class="calculated-info" id="top-cleat-total-${index}">Rp 0</div>
                </div>
            </div>
        </div>

        <hr>

        <!-- TOTAL COMPONENT -->
        <div class="card border-success">
            <div class="card-body">
                <h4 class="text-success mb-0">
                    TOTAL TOP PANEL: 
                    <span class="float-end" id="total-top-panel-${index}">Rp 0</span>
                </h4>
            </div>
        </div>
    `;
}
// ============================================
// ✅ GET SIDE PANEL L CONTENT - WITH M³ DISPLAY
// ============================================

function getSidePanelContent(index, panelName) {
    const panelType = initialFormData.panel_type;
    const solidType = initialFormData.solid_type;
    
    const mdfPanels = getPanelsByType('MDF');
    const solids = getSolidsByType(solidType);
    const allVeneers = getAllVeneers();
    
    const prefix = 'sidel';
    
    setTimeout(() => {
        attachAutoCalculateListeners(index, panelName);
    }, 500);
    
    return `
        <div class="alert alert-info alert-sm">
            <i class="fas fa-lock me-2"></i>
            <strong>Material Locked:</strong> Panel = ${panelType} | Solid = SL.${solidType}
        </div>

        <!-- Main Dimensions -->
        <div class="card mb-3 border-primary">
            <div class="card-header bg-primary text-white">
                <strong>Main Dimensions</strong>
            </div>
            <div class="card-body">
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Width (mm) <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="${prefix}-width-${index}" placeholder="Width">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Length (mm) <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="${prefix}-length-${index}" placeholder="Length">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Thickness Total (mm) <span class="text-danger">*</span></label>
                        <select class="form-select" id="${prefix}-thickness-${index}">
                            <option value="">-- Pilih --</option>
                            <option value="18">18mm</option>
                            <option value="21">21mm</option>
                            <option value="28">28mm</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Qty <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="${prefix}-qty-${index}" value="1" min="1">
                    </div>
                </div>

                <!-- ✅ M² & M³ DISPLAY ROW -->
                <div class="row g-3 mb-4">
                    <div class="col-md-6">
                        <label class="form-label">M² Total (Auto-calculated)</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-m2-${index}" readonly>
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">M³ Total (Auto-calculated)</label>
                        <div class="input-group">
                            <input type="text" class="form-control bg-warning text-dark fw-bold" id="${prefix}-m3-${index}" readonly>
                            <span class="input-group-text">m³</span>
                        </div>
                    </div>
                </div>

                <div class="row g-3">
                    <div class="col-md-6">
                        <label class="form-label">Core Thickness</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-core-thickness-${index}" readonly>
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">Core Type</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-core-type-${index}" readonly>
                    </div>
                </div>
            </div>
        </div>

        <!-- Sub Components -->
        <div class="sub-component-card">
            <h6 class="sub-component-title">1. Patching Panel 1 (Core Panel)</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-4">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="${prefix}-patching-panel-qty-${index}" value="1" min="1">
                </div>
                <div class="col-md-4">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="${prefix}-patching-panel-m2-${index}" readonly>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="${prefix}-patching-panel-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Patching Panel 1:</strong>
                <span class="float-end" id="${prefix}-patching-panel-total-${index}">Rp 0</span>
            </div>
        </div>

        <div class="sub-component-card">
            <h6 class="sub-component-title">2. Patching Solid 2 (PANJANG) - Optional</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-3">
                    <label class="form-label">Width (mm)</label>
                    <input type="number" class="form-control" id="${prefix}-patching-solid2-width-${index}" placeholder="Width">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Thickness</label>
                    <select class="form-select" id="${prefix}-patching-solid2-thickness-${index}">
                        <option value="">-- Pilih --</option>
                        ${solids.map(s => `<option value="${s.id}" data-thickness="${s.thickness}" data-price="${s.price}">${s.thickness}mm - ${formatRupiah(s.price)}/M²</option>`).join('')}
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="${prefix}-patching-solid2-qty-${index}" value="2" min="0">
                </div>
                <div class="col-md-2">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="${prefix}-patching-solid2-m2-${index}" readonly>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="${prefix}-patching-solid2-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Patching Solid 2:</strong>
                <span class="float-end" id="${prefix}-patching-solid2-total-${index}">Rp 0</span>
            </div>
        </div>

        <div class="sub-component-card">
            <h6 class="sub-component-title">3. Patching Solid 3 (LEBAR) - Optional</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-3">
                    <label class="form-label">Width (mm)</label>
                    <input type="number" class="form-control" id="${prefix}-patching-solid3-width-${index}" placeholder="Width">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Thickness</label>
                    <select class="form-select" id="${prefix}-patching-solid3-thickness-${index}">
                        <option value="">-- Pilih --</option>
                        ${solids.map(s => `<option value="${s.id}" data-thickness="${s.thickness}" data-price="${s.price}">${s.thickness}mm - ${formatRupiah(s.price)}/M²</option>`).join('')}
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="${prefix}-patching-solid3-qty-${index}" value="2" min="0">
                </div>
                <div class="col-md-2">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="${prefix}-patching-solid3-m2-${index}" readonly>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="${prefix}-patching-solid3-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Patching Solid 3:</strong>
                <span class="float-end" id="${prefix}-patching-solid3-total-${index}">Rp 0</span>
            </div>
        </div>

        <div class="sub-component-card">
            <h6 class="sub-component-title">4. Crossgrain (MDF) - Optional</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-4">
                    <label class="form-label">Thickness</label>
                    <select class="form-select" id="${prefix}-crossgrain-thickness-${index}">
                        <option value="">-- Pilih --</option>
                        ${mdfPanels.map(p => `<option value="${p.id}" data-thickness="${p.thickness}" data-price="${p.price}">${p.thickness}mm - ${formatRupiah(p.price)}/M²</option>`).join('')}
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="${prefix}-crossgrain-qty-${index}" value="2" min="0">
                </div>
                <div class="col-md-3">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="${prefix}-crossgrain-m2-${index}" readonly>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="${prefix}-crossgrain-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Crossgrain:</strong>
                <span class="float-end" id="${prefix}-crossgrain-total-${index}">Rp 0</span>
            </div>
        </div>

        <div class="sub-component-card">
            <h6 class="sub-component-title">5. Veneer Face - Optional</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-3">
                    <label class="form-label">Pattern</label>
                    <select class="form-select" id="${prefix}-veneer-face-pattern-${index}">
                        <option value="LURUS">LURUS</option>
                        <option value="MOTIF">MOTIF</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Veneer Type</label>
                    <select class="form-select" id="${prefix}-veneer-face-type-${index}">
                        <option value="">-- Pilih --</option>
                        ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">V.${v.veneer_type}</option>`).join('')}
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="${prefix}-veneer-face-qty-${index}" value="1" min="0">
                </div>
                <div class="col-md-2">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="${prefix}-veneer-face-m2-${index}" readonly>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="${prefix}-veneer-face-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Veneer Face:</strong>
                <span class="float-end" id="${prefix}-veneer-face-total-${index}">Rp 0</span>
            </div>
        </div>

        <div class="sub-component-card">
            <h6 class="sub-component-title">6. Veneer Back - Optional</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-3">
                    <label class="form-label">Pattern</label>
                    <select class="form-select" id="${prefix}-veneer-back-pattern-${index}">
                        <option value="LURUS">LURUS</option>
                        <option value="MOTIF">MOTIF</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Veneer Type</label>
                    <select class="form-select" id="${prefix}-veneer-back-type-${index}">
                        <option value="">-- Pilih --</option>
                        ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">V.${v.veneer_type}</option>`).join('')}
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="${prefix}-veneer-back-qty-${index}" value="1" min="0">
                </div>
                <div class="col-md-2">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="${prefix}-veneer-back-m2-${index}" readonly>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="${prefix}-veneer-back-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Veneer Back:</strong>
                <span class="float-end" id="${prefix}-veneer-back-total-${index}">Rp 0</span>
            </div>
        </div>

        <hr>

        <!-- TOTAL COMPONENT -->
        <div class="card border-success">
            <div class="card-body">
                <h4 class="text-success mb-0">
                    TOTAL SIDE PANEL L: 
                    <span class="float-end" id="total-${prefix}-panel-${index}">Rp 0</span>
                </h4>
            </div>
        </div>
    `;
}
// ============================================
// ✅ GET SIDE PANEL R CONTENT - WITH COPY/MANUAL MODE + M³ DISPLAY
// ============================================

function getSidePanelRContent(index) {
    const panelType = initialFormData.panel_type;
    const solidType = initialFormData.solid_type;
    
    const mdfPanels = getPanelsByType('MDF');
    const solids = getSolidsByType(solidType);
    const allVeneers = getAllVeneers();
    
    const prefix = 'sider';
    
    setTimeout(() => {
        attachAutoCalculateListeners(index, 'Side Panel R');
    }, 500);
    
    return `
        <div class="alert alert-warning alert-sm">
            <i class="fas fa-info-circle me-2"></i>
            <strong>Side Panel R:</strong> Pilih mode Copy dari Side Panel L atau Manual Input
        </div>

        <!-- Mode Selection -->
        <div class="card mb-3 border-warning">
            <div class="card-header bg-warning">
                <strong>Mode Selection</strong>
            </div>
            <div class="card-body">
                <select class="form-select form-select-lg" id="${prefix}-mode-${index}" onchange="onSidePanelRModeChange(${index})">
                    <option value="">-- Pilih Mode --</option>
                    <option value="copy">📋 Copy dari Side Panel L</option>
                    <option value="manual">✏️ Manual Input</option>
                </select>
            </div>
        </div>

        <!-- Manual Input Section (Hidden by default) -->
        <div id="${prefix}-manual-inputs-${index}" style="display: none;">
            <div class="alert alert-info alert-sm">
                <i class="fas fa-lock me-2"></i>
                <strong>Material Locked:</strong> Panel = ${panelType} | Solid = SL.${solidType}
            </div>

            <!-- Main Dimensions -->
            <div class="card mb-3 border-primary">
                <div class="card-header bg-primary text-white">
                    <strong>Main Dimensions</strong>
                </div>
                <div class="card-body">
                    <div class="row g-3">
                        <div class="col-md-3">
                            <label class="form-label">Width (mm) <span class="text-danger">*</span></label>
                            <input type="number" class="form-control" id="${prefix}-width-${index}" placeholder="Width">
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">Length (mm) <span class="text-danger">*</span></label>
                            <input type="number" class="form-control" id="${prefix}-length-${index}" placeholder="Length">
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">Thickness Total (mm) <span class="text-danger">*</span></label>
                            <select class="form-select" id="${prefix}-thickness-${index}">
                                <option value="">-- Pilih --</option>
                                <option value="18">18mm</option>
                                <option value="21">21mm</option>
                                <option value="28">28mm</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <label class="form-label">Qty <span class="text-danger">*</span></label>
                            <input type="number" class="form-control" id="${prefix}-qty-${index}" value="1" min="1">
                        </div>
                    </div>

                    <!-- ✅ M² & M³ DISPLAY ROW -->
                    <div class="row g-3 mb-4">
                        <div class="col-md-6">
                            <label class="form-label">M² Total (Auto-calculated)</label>
                            <input type="text" class="form-control bg-light" id="${prefix}-m2-${index}" readonly>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">M³ Total (Auto-calculated)</label>
                            <div class="input-group">
                                <input type="text" class="form-control bg-warning text-dark fw-bold" id="${prefix}-m3-${index}" readonly>
                                <span class="input-group-text">m³</span>
                            </div>
                        </div>
                    </div>

                    <div class="row g-3">
                        <div class="col-md-6">
                            <label class="form-label">Core Thickness</label>
                            <input type="text" class="form-control bg-light" id="${prefix}-core-thickness-${index}" readonly>
                        </div>
                        <div class="col-md-6">
                            <label class="form-label">Core Type</label>
                            <input type="text" class="form-control bg-light" id="${prefix}-core-type-${index}" readonly>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Sub Components (Same as Side Panel L) -->
            <div class="sub-component-card">
                <h6 class="sub-component-title">1. Patching Panel 1 (Core Panel)</h6>
                <div class="row g-3 mb-3">
                    <div class="col-md-4">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="${prefix}-patching-panel-qty-${index}" value="1" min="1">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-patching-panel-m2-${index}" readonly>
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Price/M²</label>
                        <div class="calculated-info" id="${prefix}-patching-panel-price-${index}">Rp 0</div>
                    </div>
                </div>
                <div class="total-component">
                    <strong>Total Patching Panel 1:</strong>
                    <span class="float-end" id="${prefix}-patching-panel-total-${index}">Rp 0</span>
                </div>
            </div>

            <div class="sub-component-card">
                <h6 class="sub-component-title">2. Patching Solid 2 (PANJANG) - Optional</h6>
                <div class="row g-3 mb-3">
                    <div class="col-md-3">
                        <label class="form-label">Width (mm)</label>
                        <input type="number" class="form-control" id="${prefix}-patching-solid2-width-${index}" placeholder="Width">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Thickness</label>
                        <select class="form-select" id="${prefix}-patching-solid2-thickness-${index}">
                            <option value="">-- Pilih --</option>
                            ${solids.map(s => `<option value="${s.id}" data-thickness="${s.thickness}" data-price="${s.price}">${s.thickness}mm - ${formatRupiah(s.price)}/M²</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="${prefix}-patching-solid2-qty-${index}" value="2" min="0">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-patching-solid2-m2-${index}" readonly>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Price/M²</label>
                        <div class="calculated-info" id="${prefix}-patching-solid2-price-${index}">Rp 0</div>
                    </div>
                </div>
                <div class="total-component">
                    <strong>Total Patching Solid 2:</strong>
                    <span class="float-end" id="${prefix}-patching-solid2-total-${index}">Rp 0</span>
                </div>
            </div>

            <div class="sub-component-card">
                <h6 class="sub-component-title">3. Patching Solid 3 (LEBAR) - Optional</h6>
                <div class="row g-3 mb-3">
                    <div class="col-md-3">
                        <label class="form-label">Width (mm)</label>
                        <input type="number" class="form-control" id="${prefix}-patching-solid3-width-${index}" placeholder="Width">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Thickness</label>
                        <select class="form-select" id="${prefix}-patching-solid3-thickness-${index}">
                            <option value="">-- Pilih --</option>
                            ${solids.map(s => `<option value="${s.id}" data-thickness="${s.thickness}" data-price="${s.price}">${s.thickness}mm - ${formatRupiah(s.price)}/M²</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="${prefix}-patching-solid3-qty-${index}" value="2" min="0">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-patching-solid3-m2-${index}" readonly>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Price/M²</label>
                        <div class="calculated-info" id="${prefix}-patching-solid3-price-${index}">Rp 0</div>
                    </div>
                </div>
                <div class="total-component">
                    <strong>Total Patching Solid 3:</strong>
                    <span class="float-end" id="${prefix}-patching-solid3-total-${index}">Rp 0</span>
                </div>
            </div>

            <div class="sub-component-card">
                <h6 class="sub-component-title">4. Crossgrain (MDF) - Optional</h6>
                <div class="row g-3 mb-3">
                    <div class="col-md-4">
                        <label class="form-label">Thickness</label>
                        <select class="form-select" id="${prefix}-crossgrain-thickness-${index}">
                            <option value="">-- Pilih --</option>
                            ${mdfPanels.map(p => `<option value="${p.id}" data-thickness="${p.thickness}" data-price="${p.price}">${p.thickness}mm - ${formatRupiah(p.price)}/M²</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="${prefix}-crossgrain-qty-${index}" value="2" min="0">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-crossgrain-m2-${index}" readonly>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Price/M²</label>
                        <div class="calculated-info" id="${prefix}-crossgrain-price-${index}">Rp 0</div>
                    </div>
                </div>
                <div class="total-component">
                    <strong>Total Crossgrain:</strong>
                    <span class="float-end" id="${prefix}-crossgrain-total-${index}">Rp 0</span>
                </div>
            </div>

            <div class="sub-component-card">
                <h6 class="sub-component-title">5. Veneer Face - Optional</h6>
                <div class="row g-3 mb-3">
                    <div class="col-md-3">
                        <label class="form-label">Pattern</label>
                        <select class="form-select" id="${prefix}-veneer-face-pattern-${index}">
                            <option value="LURUS">LURUS</option>
                            <option value="MOTIF">MOTIF</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Veneer Type</label>
                        <select class="form-select" id="${prefix}-veneer-face-type-${index}">
                            <option value="">-- Pilih --</option>
                            ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">V.${v.veneer_type}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="${prefix}-veneer-face-qty-${index}" value="1" min="0">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-veneer-face-m2-${index}" readonly>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Price/M²</label>
                        <div class="calculated-info" id="${prefix}-veneer-face-price-${index}">Rp 0</div>
                    </div>
                </div>
                <div class="total-component">
                    <strong>Total Veneer Face:</strong>
                    <span class="float-end" id="${prefix}-veneer-face-total-${index}">Rp 0</span>
                </div>
            </div>

            <div class="sub-component-card">
                <h6 class="sub-component-title">6. Veneer Back - Optional</h6>
                <div class="row g-3 mb-3">
                    <div class="col-md-3">
                        <label class="form-label">Pattern</label>
                        <select class="form-select" id="${prefix}-veneer-back-pattern-${index}">
                            <option value="LURUS">LURUS</option>
                            <option value="MOTIF">MOTIF</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Veneer Type</label>
                        <select class="form-select" id="${prefix}-veneer-back-type-${index}">
                            <option value="">-- Pilih --</option>
                            ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">V.${v.veneer_type}</option>`).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="${prefix}-veneer-back-qty-${index}" value="1" min="0">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-veneer-back-m2-${index}" readonly>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Price/M²</label>
                        <div class="calculated-info" id="${prefix}-veneer-back-price-${index}">Rp 0</div>
                    </div>
                </div>
                <div class="total-component">
                    <strong>Total Veneer Back:</strong>
                    <span class="float-end" id="${prefix}-veneer-back-total-${index}">Rp 0</span>
                </div>
            </div>
        </div>

        <hr>

        <!-- TOTAL COMPONENT -->
        <div class="card border-success">
            <div class="card-body">
                <h4 class="text-success mb-0">
                    TOTAL SIDE PANEL R: 
                    <span class="float-end" id="total-${prefix}-panel-${index}">Rp 0</span>
                </h4>
            </div>
        </div>
    `;
}

// ============================================
// ✅ GET BOTTOM PANEL CONTENT - DEFAULT QTY = 1 FOR ALL
// ============================================

function getBottomPanelContent(index) {
    const panelType = initialFormData.panel_type;
    const allVeneers = getAllVeneers();
    
    setTimeout(() => {
        attachAutoCalculateListeners(index, 'Bottom Panel');
    }, 500);
    
    return `
        <div class="alert alert-info alert-sm">
            <i class="fas fa-lock me-2"></i>
            <strong>Material Locked:</strong> Panel = ${panelType}
        </div>

        <div class="alert alert-warning alert-sm">
            <i class="fas fa-info-circle me-2"></i>
            <strong>Bottom Panel Components:</strong> Core Panel, Veneer Face, Veneer Back, E.Band (Default Qty = 1 untuk semua)
        </div>

        <!-- Main Dimensions -->
        <div class="card mb-3 border-primary">
            <div class="card-header bg-primary text-white">
                <strong>Dimensi Panel</strong>
            </div>
            <div class="card-body">
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Width (mm) <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="bottom-width-${index}" placeholder="Width">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Length (mm) <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="bottom-length-${index}" placeholder="Length">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Thickness Total (mm) <span class="text-danger">*</span></label>
                        <select class="form-select" id="bottom-thickness-${index}">
                            <option value="">-- Pilih --</option>
                            <option value="16">16mm</option>
                            <option value="19">19mm</option>
                            <option value="26">26mm</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Qty <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="bottom-qty-${index}" value="1" min="1">
                    </div>
                </div>

                <!-- M² & M³ DISPLAY -->
                <div class="row g-3 mt-3">
                    <div class="col-md-6">
                        <label class="form-label">M² Total (Auto-calculated)</label>
                        <input type="text" class="form-control bg-light fw-bold" id="bottom-m2-${index}" readonly>
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">M³ Total (Auto-calculated)</label>
                        <div class="input-group">
                            <input type="text" class="form-control bg-warning text-dark fw-bold" id="bottom-m3-${index}" readonly>
                            <span class="input-group-text">m³</span>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- 1. CORE PANEL -->
        <div class="sub-component-card">
            <h6 class="sub-component-title">1. Core Panel</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-3">
                    <label class="form-label">Core Type (Locked)</label>
                    <input type="text" class="form-control bg-light" id="bottom-core-type-${index}" value="${panelType}" readonly>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Core Thickness</label>
                    <input type="text" class="form-control bg-light" id="bottom-core-thickness-${index}" readonly>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="bottom-core-qty-${index}" value="1" min="1">
                </div>
                <div class="col-md-2">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="bottom-core-m2-${index}" readonly>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="bottom-core-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Core Panel:</strong>
                <span class="float-end" id="bottom-core-total-${index}">Rp 0</span>
            </div>
        </div>

        <!-- 2. VENEER FACE -->
        <div class="sub-component-card">
            <h6 class="sub-component-title">2. Veneer Face</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-3">
                    <label class="form-label">Pattern</label>
                    <select class="form-select" id="bottom-veneer-face-pattern-${index}">
                        <option value="LURUS">LURUS</option>
                        <option value="MOTIF">MOTIF</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Veneer Type</label>
                    <select class="form-select" id="bottom-veneer-face-type-${index}">
                        <option value="">-- Pilih --</option>
                        ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">V.${v.veneer_type}</option>`).join('')}
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="bottom-veneer-face-qty-${index}" value="1" min="0">
                </div>
                <div class="col-md-2">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="bottom-veneer-face-m2-${index}" readonly>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="bottom-veneer-face-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Veneer Face:</strong>
                <span class="float-end" id="bottom-veneer-face-total-${index}">Rp 0</span>
            </div>
        </div>

        <!-- 3. VENEER BACK -->
        <div class="sub-component-card">
            <h6 class="sub-component-title">3. Veneer Back</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-3">
                    <label class="form-label">Pattern</label>
                    <select class="form-select" id="bottom-veneer-back-pattern-${index}">
                        <option value="LURUS">LURUS</option>
                        <option value="MOTIF">MOTIF</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Veneer Type</label>
                    <select class="form-select" id="bottom-veneer-back-type-${index}">
                        <option value="">-- Pilih --</option>
                        ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">V.${v.veneer_type}</option>`).join('')}
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="bottom-veneer-back-qty-${index}" value="1" min="0">
                </div>
                <div class="col-md-2">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="bottom-veneer-back-m2-${index}" readonly>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="bottom-veneer-back-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Veneer Back:</strong>
                <span class="float-end" id="bottom-veneer-back-total-${index}">Rp 0</span>
            </div>
        </div>

        <!-- 4. E.BAND -->
        <div class="sub-component-card">
            <h6 class="sub-component-title">4. E.Band</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-4">
                    <label class="form-label">Qty (pieces)</label>
                    <input type="number" class="form-control" id="bottom-eband-qty-${index}" value="1" min="0">
                </div>
                <div class="col-md-4">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="bottom-eband-m2-${index}" readonly>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="bottom-eband-price-${index}">Rp 3.000</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total E.Band:</strong>
                <span class="float-end" id="bottom-eband-total-${index}">Rp 0</span>
            </div>
        </div>

        <hr>

        <!-- TOTAL COMPONENT -->
        <div class="card border-success">
            <div class="card-body">
                <h4 class="text-success mb-0">
                    TOTAL BOTTOM PANEL: 
                    <span class="float-end" id="total-bottom-panel-${index}">Rp 0</span>
                </h4>
            </div>
        </div>
    `;
}
// ============================================
// ✅ GET BACK PANEL CONTENT - ONLY 3 SUB-COMPONENTS
// ============================================

function getBackPanelContent(index) {
    const allVeneers = getAllVeneers();
    
    setTimeout(() => {
        attachAutoCalculateListeners(index, 'Back Panel');
    }, 500);
    
    return `
        <div class="alert alert-warning alert-sm">
            <i class="fas fa-exclamation-triangle me-2"></i>
            <strong>Back Panel Special:</strong> Pilih MDF atau Plywood (bukan locked material)
        </div>

        <div class="alert alert-info alert-sm">
            <i class="fas fa-info-circle me-2"></i>
            <strong>Back Panel Components:</strong> Core Panel (MDF/Plywood), Veneer Face, Veneer Back (Default Qty = 1 untuk semua)
        </div>

        <!-- Main Dimensions -->
        <div class="card mb-3 border-primary">
            <div class="card-header bg-primary text-white">
                <strong>Dimensi Panel</strong>
            </div>
            <div class="card-body">
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Width (mm) <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="back-width-${index}" placeholder="Width">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Length (mm) <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="back-length-${index}" placeholder="Length">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Thickness Total (mm) <span class="text-danger">*</span></label>
                        <select class="form-select" id="back-thickness-${index}" onchange="onBackPanelThicknessChange(${index})">
                            <option value="">-- Pilih --</option>
                            <option value="6">6mm</option>
                            <option value="19">19mm</option>
                            <option value="26">26mm</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Qty <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="back-qty-${index}" value="1" min="1">
                    </div>
                </div>

                <!-- M² & M³ DISPLAY -->
                <div class="row g-3 mt-3">
                    <div class="col-md-6">
                        <label class="form-label">M² Total (Auto-calculated)</label>
                        <input type="text" class="form-control bg-light fw-bold" id="back-m2-${index}" readonly>
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">M³ Total (Auto-calculated)</label>
                        <div class="input-group">
                            <input type="text" class="form-control bg-warning text-dark fw-bold" id="back-m3-${index}" readonly>
                            <span class="input-group-text">m³</span>
                        </div>
                    </div>
                </div>

                <!-- Panel Type & Core Info -->
                <div class="row g-3 mt-3">
                    <div class="col-md-4">
                        <label class="form-label">Panel Type <span class="text-danger">*</span></label>
                        <select class="form-select" id="back-panel-type-${index}">
                            <option value="">-- Pilih --</option>
                            <option value="MDF">MDF</option>
                            <option value="Plywood">Plywood</option>
                        </select>
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Core Thickness</label>
                        <input type="text" class="form-control bg-light" id="back-core-thickness-${index}" readonly>
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Core Type</label>
                        <input type="text" class="form-control bg-light" id="back-core-type-${index}" readonly>
                    </div>
                </div>

                <!-- Thickness Breakdown Info -->
                <div class="alert alert-secondary mt-3 mb-0">
                    <small>
                        <i class="fas fa-info-circle me-2"></i>
                        <strong>Thickness Breakdown:</strong>
                        <ul class="mb-0 mt-2" style="font-size: 12px;">
                            <li>6mm = Core 5mm + Veneer Face 0.5mm + Veneer Back 0.5mm</li>
                            <li>19mm = Core 18mm + Veneer Face 0.5mm + Veneer Back 0.5mm</li>
                            <li>26mm = Core 25mm + Veneer Face 0.5mm + Veneer Back 0.5mm</li>
                        </ul>
                    </small>
                </div>
            </div>
        </div>

        <!-- 1. CORE PANEL (MDF/Plywood) -->
        <div class="sub-component-card">
            <h6 class="sub-component-title">1. Core Panel (MDF/Plywood)</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-3">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="back-core-qty-${index}" value="1" min="1">
                </div>
                <div class="col-md-3">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="back-core-m2-${index}" readonly>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="back-core-price-${index}">Rp 0</div>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Total</label>
                    <div class="calculated-info" id="back-core-total-${index}">Rp 0</div>
                </div>
            </div>
        </div>

        <!-- 2. VENEER FACE -->
        <div class="sub-component-card">
            <h6 class="sub-component-title">2. Veneer Face</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-3">
                    <label class="form-label">Pattern</label>
                    <select class="form-select" id="back-veneer-face-pattern-${index}">
                        <option value="LURUS">LURUS</option>
                        <option value="MOTIF">MOTIF</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Veneer Type</label>
                    <select class="form-select" id="back-veneer-face-type-${index}">
                        <option value="">-- Pilih --</option>
                        ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">V.${v.veneer_type}</option>`).join('')}
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="back-veneer-face-qty-${index}" value="1" min="0">
                </div>
                <div class="col-md-2">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="back-veneer-face-m2-${index}" readonly>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="back-veneer-face-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Veneer Face:</strong>
                <span class="float-end" id="back-veneer-face-total-${index}">Rp 0</span>
            </div>
        </div>

        <!-- 3. VENEER BACK -->
        <div class="sub-component-card">
            <h6 class="sub-component-title">3. Veneer Back</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-3">
                    <label class="form-label">Pattern</label>
                    <select class="form-select" id="back-veneer-back-pattern-${index}">
                        <option value="LURUS">LURUS</option>
                        <option value="MOTIF">MOTIF</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Veneer Type</label>
                    <select class="form-select" id="back-veneer-back-type-${index}">
                        <option value="">-- Pilih --</option>
                        ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">V.${v.veneer_type}</option>`).join('')}
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="back-veneer-back-qty-${index}" value="1" min="0">
                </div>
                <div class="col-md-2">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="back-veneer-back-m2-${index}" readonly>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="back-veneer-back-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Veneer Back:</strong>
                <span class="float-end" id="back-veneer-back-total-${index}">Rp 0</span>
            </div>
        </div>

        <hr>

        <!-- TOTAL COMPONENT -->
        <div class="card border-success">
            <div class="card-body">
                <h4 class="text-success mb-0">
                    TOTAL BACK PANEL: 
                    <span class="float-end" id="total-back-panel-${index}">Rp 0</span>
                </h4>
            </div>
        </div>
    `;
}

// ============================================
// ✅ GET FRAME CONTENT (FRONT/BACK) - ONLY SOLID
// ============================================

function getFrameContent(index, frameName) {
    const solidType = initialFormData.solid_type;
    const solids = getSolidsByType(solidType);
    
    const prefix = frameName === 'Front Frame' ? 'front' : 'back';
    
    setTimeout(() => {
        attachAutoCalculateListeners(index, frameName);
    }, 500);
    
    return `
        <div class="alert alert-info alert-sm">
            <i class="fas fa-lock me-2"></i>
            <strong>Material Locked:</strong> Solid = SL.${solidType}
        </div>

        <div class="alert alert-warning alert-sm">
            <i class="fas fa-info-circle me-2"></i>
            <strong>${frameName} Component:</strong> Core Solid only (pilih thickness & qty)
        </div>

        <!-- Dimensi Frame -->
        <div class="card mb-3 border-primary">
            <div class="card-header bg-primary text-white">
                <strong>Dimensi Frame</strong>
            </div>
            <div class="card-body">
                <div class="row g-3">
                    <div class="col-md-6">
                        <label class="form-label">Lebar (mm) <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="${prefix}-width-${index}" placeholder="Lebar">
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">Panjang (mm) <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="${prefix}-length-${index}" placeholder="Panjang">
                    </div>
                </div>

                <!-- M² Display -->
                <div class="row g-3 mt-3">
                    <div class="col-md-12">
                        <label class="form-label">M² Total (Auto-calculated)</label>
                        <input type="text" class="form-control bg-light fw-bold" id="${prefix}-m2-${index}" readonly>
                    </div>
                </div>
            </div>
        </div>

        <!-- Core Solid -->
        <div class="card mb-3 border-success">
            <div class="card-header bg-success text-white">
                <strong>Core Solid (Locked: SL.${solidType})</strong>
            </div>
            <div class="card-body">
                <div class="row g-3">
                    <div class="col-md-4">
                        <label class="form-label">Pilih Tebal Solid <span class="text-danger">*</span></label>
                        <select class="form-select" id="${prefix}-solid-thickness-${index}">
                            <option value="">-- Pilih Tebal --</option>
                            ${solids.map(s => `
                                <option value="${s.id}" 
                                        data-thickness="${s.thickness}" 
                                        data-price="${s.price}">
                                    ${s.thickness}mm - ${formatRupiah(s.price)}/M²
                                </option>
                            `).join('')}
                        </select>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Qty</label>
                        <input type="number" class="form-control" id="${prefix}-solid-qty-${index}" value="1" min="1">
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-solid-m2-${index}" readonly>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Price/M²</label>
                        <div class="calculated-info" id="${prefix}-solid-price-${index}">Rp 0</div>
                    </div>
                    <div class="col-md-2">
                        <label class="form-label">Total</label>
                        <div class="calculated-info fw-bold text-success" id="${prefix}-solid-total-${index}">Rp 0</div>
                    </div>
                </div>

                <!-- M³ Display -->
                <div class="row g-3 mt-3">
                    <div class="col-md-12">
                        <label class="form-label">M³ Total (Auto-calculated)</label>
                        <div class="input-group">
                            <input type="text" class="form-control bg-warning text-dark fw-bold" id="${prefix}-m3-${index}" readonly>
                            <span class="input-group-text">m³</span>
                        </div>
                        <small class="text-muted">Formula: Lebar × Panjang × Thickness × Qty</small>
                    </div>
                </div>
            </div>
        </div>

        <hr>

        <!-- TOTAL COMPONENT -->
        <div class="card border-success">
            <div class="card-body">
                <h4 class="text-success mb-0">
                    TOTAL ${frameName.toUpperCase()}: 
                    <span class="float-end" id="total-${prefix}-frame-${index}">Rp 0</span>
                </h4>
            </div>
        </div>
    `;
}
// ============================================
// ✅ GET FACE DRAWER CONTENT - WITH M³ DISPLAY
// ============================================

function getFaceDrawerContent(index) {
    const panelType = initialFormData.panel_type;
    const solidType = initialFormData.solid_type;
    
    const mdfPanels = getPanelsByType('MDF');
    const solids = getSolidsByType(solidType);
    const allVeneers = getAllVeneers();
    
    setTimeout(() => {
        attachAutoCalculateListeners(index, 'Face Drawer');
    }, 500);
    
    return `
        <div class="alert alert-info alert-sm">
            <i class="fas fa-lock me-2"></i>
            <strong>Material Locked:</strong> Panel = ${panelType} | Solid = SL.${solidType}
        </div>

        <div class="alert alert-warning alert-sm">
            <i class="fas fa-exclamation-triangle me-2"></i>
            <strong>Important:</strong> Qty Face Drawer akan digunakan sebagai qty default untuk semua drawer lainnya!
        </div>

        <!-- Main Dimensions -->
        <div class="card mb-3 border-primary">
            <div class="card-header bg-primary text-white">
                <strong>Main Dimensions</strong>
            </div>
            <div class="card-body">
                <div class="row g-3">
                    <div class="col-md-3">
                        <label class="form-label">Width (mm) <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="facedrawer-width-${index}" placeholder="Width">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Length (mm) <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="facedrawer-length-${index}" placeholder="Length">
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Thickness Total (mm) <span class="text-danger">*</span></label>
                        <select class="form-select" id="facedrawer-thickness-${index}">
                            <option value="">-- Pilih --</option>
                            <option value="18">18mm</option>
                            <option value="21">21mm</option>
                            <option value="28">28mm</option>
                        </select>
                    </div>
                    <div class="col-md-3">
                        <label class="form-label">Qty <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="facedrawer-qty-${index}" value="1" min="1" onchange="onFaceDrawerQtyChange(${index})">
                    </div>
                </div>

                <!-- ✅ M² & M³ DISPLAY ROW -->
                <div class="row g-3 mb-4">
                    <div class="col-md-6">
                        <label class="form-label">M² Total (Auto-calculated)</label>
                        <input type="text" class="form-control bg-light" id="facedrawer-m2-${index}" readonly>
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">M³ Total (Auto-calculated)</label>
                        <div class="input-group">
                            <input type="text" class="form-control bg-warning text-dark fw-bold" id="facedrawer-m3-${index}" readonly>
                            <span class="input-group-text">m³</span>
                        </div>
                    </div>
                </div>

                <div class="row g-3">
                    <div class="col-md-6">
                        <label class="form-label">Core Thickness</label>
                        <input type="text" class="form-control bg-light" id="facedrawer-core-thickness-${index}" readonly>
                    </div>
                    <div class="col-md-6">
                        <label class="form-label">Core Type</label>
                        <input type="text" class="form-control bg-light" id="facedrawer-core-type-${index}" readonly>
                    </div>
                </div>
            </div>
        </div>

        <!-- Sub Components -->
        <div class="sub-component-card">
            <h6 class="sub-component-title">1. Patching Panel 1 (Core Panel)</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-4">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="facedrawer-patching-panel-qty-${index}" value="1" min="1">
                </div>
                <div class="col-md-4">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="facedrawer-patching-panel-m2-${index}" readonly>
                </div>
                <div class="col-md-4">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="facedrawer-patching-panel-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Patching Panel 1:</strong>
                <span class="float-end" id="facedrawer-patching-panel-total-${index}">Rp 0</span>
            </div>
        </div>

        <div class="sub-component-card">
            <h6 class="sub-component-title">2. Patching Solid 2 (PANJANG) - Optional</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-3">
                    <label class="form-label">Width (mm)</label>
                    <input type="number" class="form-control" id="facedrawer-patching-solid2-width-${index}" placeholder="Width">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Thickness</label>
                    <select class="form-select" id="facedrawer-patching-solid2-thickness-${index}">
                        <option value="">-- Pilih --</option>
                        ${solids.map(s => `<option value="${s.id}" data-thickness="${s.thickness}" data-price="${s.price}">${s.thickness}mm - ${formatRupiah(s.price)}/M²</option>`).join('')}
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="facedrawer-patching-solid2-qty-${index}" value="2" min="0">
                </div>
                <div class="col-md-2">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="facedrawer-patching-solid2-m2-${index}" readonly>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="facedrawer-patching-solid2-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Patching Solid 2:</strong>
                <span class="float-end" id="facedrawer-patching-solid2-total-${index}">Rp 0</span>
            </div>
        </div>

        <div class="sub-component-card">
            <h6 class="sub-component-title">3. Patching Solid 3 (LEBAR) - Optional</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-3">
                    <label class="form-label">Width (mm)</label>
                    <input type="number" class="form-control" id="facedrawer-patching-solid3-width-${index}" placeholder="Width">
                </div>
                <div class="col-md-3">
                    <label class="form-label">Thickness</label>
                    <select class="form-select" id="facedrawer-patching-solid3-thickness-${index}">
                        <option value="">-- Pilih --</option>
                        ${solids.map(s => `<option value="${s.id}" data-thickness="${s.thickness}" data-price="${s.price}">${s.thickness}mm - ${formatRupiah(s.price)}/M²</option>`).join('')}
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="facedrawer-patching-solid3-qty-${index}" value="2" min="0">
                </div>
                <div class="col-md-2">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="facedrawer-patching-solid3-m2-${index}" readonly>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="facedrawer-patching-solid3-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Patching Solid 3:</strong>
                <span class="float-end" id="facedrawer-patching-solid3-total-${index}">Rp 0</span>
            </div>
        </div>

        <div class="sub-component-card">
            <h6 class="sub-component-title">4. Crossgrain (MDF) - Optional</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-4">
                    <label class="form-label">Thickness</label>
                    <select class="form-select" id="facedrawer-crossgrain-thickness-${index}">
                        <option value="">-- Pilih --</option>
                        ${mdfPanels.map(p => `<option value="${p.id}" data-thickness="${p.thickness}" data-price="${p.price}">${p.thickness}mm - ${formatRupiah(p.price)}/M²</option>`).join('')}
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="facedrawer-crossgrain-qty-${index}" value="2" min="0">
                </div>
                <div class="col-md-3">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="facedrawer-crossgrain-m2-${index}" readonly>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="facedrawer-crossgrain-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Crossgrain:</strong>
                <span class="float-end" id="facedrawer-crossgrain-total-${index}">Rp 0</span>
            </div>
        </div>

        <div class="sub-component-card">
            <h6 class="sub-component-title">5. Veneer Face - Optional</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-3">
                    <label class="form-label">Pattern</label>
                    <select class="form-select" id="facedrawer-veneer-face-pattern-${index}">
                        <option value="LURUS">LURUS</option>
                        <option value="MOTIF">MOTIF</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Veneer Type</label>
                    <select class="form-select" id="facedrawer-veneer-face-type-${index}">
                        <option value="">-- Pilih --</option>
                        ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">V.${v.veneer_type}</option>`).join('')}
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="facedrawer-veneer-face-qty-${index}" value="1" min="0">
                </div>
                <div class="col-md-2">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="facedrawer-veneer-face-m2-${index}" readonly>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="facedrawer-veneer-face-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Veneer Face:</strong>
                <span class="float-end" id="facedrawer-veneer-face-total-${index}">Rp 0</span>
            </div>
        </div>

        <div class="sub-component-card">
            <h6 class="sub-component-title">6. Veneer Back - Optional</h6>
            <div class="row g-3 mb-3">
                <div class="col-md-3">
                    <label class="form-label">Pattern</label>
                    <select class="form-select" id="facedrawer-veneer-back-pattern-${index}">
                        <option value="LURUS">LURUS</option>
                        <option value="MOTIF">MOTIF</option>
                    </select>
                </div>
                <div class="col-md-3">
                    <label class="form-label">Veneer Type</label>
                    <select class="form-select" id="facedrawer-veneer-back-type-${index}">
                        <option value="">-- Pilih --</option>
                        ${allVeneers.map(v => `<option value="${v.id}" data-type="${v.veneer_type}" data-price-lurus="${v.price_lurus}" data-price-motif="${v.price_motif}">V.${v.veneer_type}</option>`).join('')}
                    </select>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Qty</label>
                    <input type="number" class="form-control" id="facedrawer-veneer-back-qty-${index}" value="1" min="0">
                </div>
                <div class="col-md-2">
                    <label class="form-label">M²</label>
                    <input type="text" class="form-control bg-light" id="facedrawer-veneer-back-m2-${index}" readonly>
                </div>
                <div class="col-md-2">
                    <label class="form-label">Price/M²</label>
                    <div class="calculated-info" id="facedrawer-veneer-back-price-${index}">Rp 0</div>
                </div>
            </div>
            <div class="total-component">
                <strong>Total Veneer Back:</strong>
                <span class="float-end" id="facedrawer-veneer-back-total-${index}">Rp 0</span>
            </div>
        </div>

        <hr>

        <!-- TOTAL COMPONENT -->
        <div class="card border-success">
            <div class="card-body">
                <h4 class="text-success mb-0">
                    TOTAL FACE DRAWER: 
                    <span class="float-end" id="total-facedrawer-${index}">Rp 0</span>
                </h4>
            </div>
        </div>
    `;
}
// ============================================
// ✅ GET DRAWER CONTENT (FRONT/SIDE/BACK/BASE) - WITH M³ DISPLAY
// ============================================

function getDrawerContent(index, drawerName) {
    const boxDrawerType = initialFormData.box_drawer_type;
    
    let drawers = [];
    let prefix = '';
    
    if (drawerName === 'Front Drawer') {
        drawers = getFrontBackDrawers(boxDrawerType);
        prefix = 'frontdrawer';
    } else if (drawerName === 'Side Drawer L') {
        drawers = getSideDrawers(boxDrawerType);
        prefix = 'sidedrawerl';
    } else if (drawerName === 'Back Drawer') {
        drawers = getFrontBackDrawers(boxDrawerType);
        prefix = 'backdrawer';
    } else if (drawerName === 'Base Drawer') {
        drawers = getBaseDrawers(boxDrawerType);
        prefix = 'basedrawer';
    }
    
    setTimeout(() => {
        attachAutoCalculateListeners(index, drawerName);
    }, 500);
    
    return `
        <div class="alert alert-info alert-sm">
            <i class="fas fa-lock me-2"></i>
            <strong>Material Locked:</strong> Box Drawer = ${boxDrawerType}
        </div>

        <div class="alert alert-warning alert-sm">
            <i class="fas fa-info-circle me-2"></i>
            <strong>Qty Auto:</strong> Qty mengikuti Face Drawer (saat ini: ${faceDrawerQty})
        </div>

        <!-- Drawer Selection -->
        <div class="card mb-3 border-primary">
            <div class="card-header bg-primary text-white">
                <strong>Drawer Selection</strong>
            </div>
            <div class="card-body">
                <div class="row g-3">
                    <div class="col-md-12">
                        <label class="form-label">Pilih Dimensi Drawer <span class="text-danger">*</span></label>
                        <select class="form-select form-select-lg" id="${prefix}-dimension-${index}" onchange="calculateDrawer(${index}, '${prefix}')">
                            <option value="">-- Pilih Dimensi --</option>
                            ${drawers.map(d => `
                                <option value="${d.id}" 
                                        data-drawer-type="${d.drawer_type}"
                                        data-width="${d.width}" 
                                        data-length="${d.length}" 
                                        data-price="${d.price}">
                                    ${d.width} × ${d.length} mm - ${formatRupiah(d.price)}/M²
                                </option>
                            `).join('')}
                        </select>
                    </div>
                </div>

                <div class="row g-3 mt-3">
                    <div class="col-md-4">
                        <label class="form-label">Qty (Auto dari Face Drawer)</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-qty-display-${index}" value="${faceDrawerQty}" readonly>
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">M²</label>
                        <input type="text" class="form-control bg-light" id="${prefix}-m2-${index}" readonly>
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Price/M²</label>
                        <div class="calculated-info" id="${prefix}-price-${index}">Rp 0</div>
                    </div>
                </div>

                <!-- ✅ M³ DISPLAY ROW -->
                <div class="row g-3 mt-3">
                    <div class="col-md-12">
                        <label class="form-label">M³ Total (Auto-calculated - Thickness 11mm)</label>
                        <div class="input-group">
                            <input type="text" class="form-control bg-warning text-dark fw-bold" id="${prefix}-m3-${index}" readonly>
                            <span class="input-group-text">m³</span>
                        </div>
                        <small class="text-muted">Formula: Width × Length × 11mm × Qty</small>
                    </div>
                </div>
            </div>
        </div>

        <hr>

        <!-- TOTAL COMPONENT -->
        <div class="card border-success">
            <div class="card-body">
                <h4 class="text-success mb-0">
                    TOTAL ${drawerName.toUpperCase()}: 
                    <span class="float-end" id="total-${prefix}-${index}">Rp 0</span>
                </h4>
            </div>
        </div>
    `;
}

// ============================================
// ✅ GET SIDE DRAWER R CONTENT - WITH COPY/MANUAL MODE + M³ DISPLAY
// ============================================

function getSideDrawerRContent(index) {
    const boxDrawerType = initialFormData.box_drawer_type;
    const sideDrawers = getSideDrawers(boxDrawerType);
    const prefix = 'sidedrawerr';
    
    setTimeout(() => {
        attachAutoCalculateListeners(index, 'Side Drawer R');
    }, 500);
    
    return `
        <div class="alert alert-warning alert-sm">
            <i class="fas fa-info-circle me-2"></i>
            <strong>Side Drawer R:</strong> Pilih mode Copy dari Side Drawer L atau Manual Input
        </div>

        <!-- Mode Selection -->
        <div class="card mb-3 border-warning">
            <div class="card-header bg-warning">
                <strong>Mode Selection</strong>
            </div>
            <div class="card-body">
                <select class="form-select form-select-lg" id="${prefix}-mode-${index}" onchange="onSideDrawerRModeChange(${index})">
                    <option value="">-- Pilih Mode --</option>
                    <option value="copy">📋 Copy dari Side Drawer L</option>
                    <option value="manual">✏️ Manual Input</option>
                </select>
            </div>
        </div>

        <!-- Manual Input Section (Hidden by default) -->
        <div id="${prefix}-manual-inputs-${index}" style="display: none;">
            <div class="alert alert-info alert-sm">
                <i class="fas fa-lock me-2"></i>
                <strong>Material Locked:</strong> Box Drawer = ${boxDrawerType}
            </div>

            <div class="alert alert-warning alert-sm">
                <i class="fas fa-info-circle me-2"></i>
                <strong>Qty Auto:</strong> Qty mengikuti Face Drawer (saat ini: ${faceDrawerQty})
            </div>

            <!-- Drawer Selection -->
            <div class="card mb-3 border-primary">
                <div class="card-header bg-primary text-white">
                    <strong>Drawer Selection</strong>
                </div>
                <div class="card-body">
                    <div class="row g-3">
                        <div class="col-md-12">
                            <label class="form-label">Pilih Dimensi Drawer <span class="text-danger">*</span></label>
                            <select class="form-select form-select-lg" id="${prefix}-dimension-${index}" onchange="calculateDrawer(${index}, '${prefix}')">
                                <option value="">-- Pilih Dimensi --</option>
                                ${sideDrawers.map(d => `
                                    <option value="${d.id}" 
                                            data-drawer-type="${d.drawer_type}"
                                            data-width="${d.width}" 
                                            data-length="${d.length}" 
                                            data-price="${d.price}">
                                        ${d.width} × ${d.length} mm - ${formatRupiah(d.price)}/M²
                                    </option>
                                `).join('')}
                            </select>
                        </div>
                    </div>

                    <div class="row g-3 mt-3">
                        <div class="col-md-4">
                            <label class="form-label">Qty (Auto dari Face Drawer)</label>
                            <input type="text" class="form-control bg-light" id="${prefix}-qty-display-${index}" value="${faceDrawerQty}" readonly>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">M²</label>
                            <input type="text" class="form-control bg-light" id="${prefix}-m2-${index}" readonly>
                        </div>
                        <div class="col-md-4">
                            <label class="form-label">Price/M²</label>
                            <div class="calculated-info" id="${prefix}-price-${index}">Rp 0</div>
                        </div>
                    </div>

                    <!-- ✅ M³ DISPLAY ROW -->
                    <div class="row g-3 mt-3">
                        <div class="col-md-12">
                            <label class="form-label">M³ Total (Auto-calculated - Thickness 11mm)</label>
                            <div class="input-group">
                                <input type="text" class="form-control bg-warning text-dark fw-bold" id="${prefix}-m3-${index}" readonly>
                                <span class="input-group-text">m³</span>
                            </div>
                            <small class="text-muted">Formula: Width × Length × 11mm × Qty</small>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <hr>

        <!-- TOTAL COMPONENT -->
        <div class="card border-success">
            <div class="card-body">
                <h4 class="text-success mb-0">
                    TOTAL SIDE DRAWER R: 
                    <span class="float-end" id="total-${prefix}-${index}">Rp 0</span>
                </h4>
            </div>
        </div>
    `;
}

// ============================================
// ✅ GET SUPPORT DRAWER CONTENT - FIXED PRICE Rp 2.200
// ============================================

function getSupportDrawerContent(index) {
    setTimeout(() => {
        attachAutoCalculateListeners(index, 'Support Drawer');
    }, 500);
    
    return `
        <div class="alert alert-info alert-sm">
            <i class="fas fa-info-circle me-2"></i>
            <strong>Support Drawer:</strong> Fixed price per unit - Rp 2.200
        </div>

        <div class="card mb-3 border-primary">
            <div class="card-header bg-primary text-white">
                <strong>Support Drawer Details</strong>
            </div>
            <div class="card-body">
                <div class="row g-3">
                    <div class="col-md-4">
                        <label class="form-label">Qty (pieces) <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="support-drawer-qty-${index}" value="0" min="0">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Price per Unit</label>
                        <div class="calculated-info" id="support-drawer-price-${index}">Rp 2.200</div>
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Material</label>
                        <div class="calculated-info">Mix</div>
                    </div>
                </div>

                <div class="alert alert-secondary mt-3">
                    <i class="fas fa-cube me-2"></i>
                    <strong>M³:</strong> Support Drawer tidak memiliki M³ (M³ = 0)
                </div>
            </div>
        </div>

        <hr>

        <!-- TOTAL COMPONENT -->
        <div class="card border-success">
            <div class="card-body">
                <h4 class="text-success mb-0">
                    TOTAL SUPPORT DRAWER: 
                    <span class="float-end" id="total-support-drawer-${index}">Rp 0</span>
                </h4>
            </div>
        </div>
    `;
}

// ============================================
// ✅ GET CLEAT DRAWER CONTENT - FIXED PRICE Rp 400
// ============================================

function getCleatDrawerContent(index) {
    setTimeout(() => {
        attachAutoCalculateListeners(index, 'Cleat Drawer');
    }, 500);
    
    return `
        <div class="alert alert-info alert-sm">
            <i class="fas fa-info-circle me-2"></i>
            <strong>Cleat Drawer:</strong> Fixed price per unit - Rp 400
        </div>

        <div class="card mb-3 border-primary">
            <div class="card-header bg-primary text-white">
                <strong>Cleat Drawer Details</strong>
            </div>
            <div class="card-body">
                <div class="row g-3">
                    <div class="col-md-4">
                        <label class="form-label">Qty (pieces) <span class="text-danger">*</span></label>
                        <input type="number" class="form-control" id="cleat-drawer-qty-${index}" value="0" min="0">
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Price per Unit</label>
                        <div class="calculated-info" id="cleat-drawer-price-${index}">Rp 400</div>
                    </div>
                    <div class="col-md-4">
                        <label class="form-label">Material</label>
                        <div class="calculated-info">Mix</div>
                    </div>
                </div>

                <div class="row g-3 mt-3">
                    <div class="col-md-12">
                        <label class="form-label">Dimensions (Standard)</label>
                        <div class="calculated-info">11mm × 11mm × 60mm</div>
                    </div>
                </div>

                <div class="alert alert-secondary mt-3">
                    <i class="fas fa-cube me-2"></i>
                    <strong>M³:</strong> Cleat Drawer tidak memiliki M³ (M³ = 0)
                </div>
            </div>
        </div>

        <hr>

        <!-- TOTAL COMPONENT -->
        <div class="card border-success">
            <div class="card-body">
                <h4 class="text-success mb-0">
                    TOTAL CLEAT DRAWER: 
                    <span class="float-end" id="total-cleat-drawer-${index}">Rp 0</span>
                </h4>
            </div>
        </div>
    `;
}
</script>
@endpush
