<?php

namespace App\Http\Controllers\Invoice;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\InvoiceStone;
use App\Models\InvoiceBom;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class InvoiceStoneController extends Controller
{
    /**
     * Display a listing of invoice stones
     */
    public function index()
    {
        $invoiceStones = InvoiceStone::with('invoiceBom')
            ->latest()
            ->paginate(10);
        
        return view('invoice.stone.index', compact('invoiceStones'));
    }

    /**
     * Show the form for creating a new invoice stone
     */
    public function create()
    {
        // Get all fixed invoice BOMs that don't have stone yet
        $invoiceBoms = InvoiceBom::where('is_fixed', true)
            ->whereDoesntHave('invoiceStone')
            ->latest()
            ->get();
        
        return view('invoice.stone.create', compact('invoiceBoms'));
    }

    /**
     * Store a newly created invoice stone
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'invoice_bom_id' => 'required|exists:invoice_boms,id',
            'stone_data' => 'required|json',
            'grand_total' => 'required|numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();
            
            // Check if invoice already has stone
            $existing = InvoiceStone::where('invoice_bom_id', $request->invoice_bom_id)->first();
            if ($existing) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invoice BOM ini sudah memiliki data stone!'
                ], 422);
            }

            // Create invoice stone
            $invoiceStone = InvoiceStone::create([
                'invoice_bom_id' => $request->invoice_bom_id,
                'stone_data' => $request->stone_data,
                'grand_total' => $request->grand_total
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Invoice Stone berhasil dibuat!',
                'invoiceStone' => $invoiceStone
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Invoice Stone creation error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified invoice stone
     */
    public function show($id)
    {
        $invoiceStone = InvoiceStone::with('invoiceBom')->findOrFail($id);
        return view('invoice.stone.show', compact('invoiceStone'));
    }

    /**
     * Show the form for editing the specified invoice stone
     */
    public function edit($id)
    {
        $invoiceStone = InvoiceStone::with('invoiceBom')->findOrFail($id);
        
        return view('invoice.stone.edit', compact('invoiceStone'));
    }

    /**
     * Update the specified invoice stone
     */
    public function update(Request $request, $id)
    {
        $invoiceStone = InvoiceStone::findOrFail($id);
        
        $validator = Validator::make($request->all(), [
            'stone_data' => 'required|json',
            'grand_total' => 'required|numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();
            
            $invoiceStone->update([
                'stone_data' => $request->stone_data,
                'grand_total' => $request->grand_total
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Invoice Stone berhasil diupdate!',
                'invoiceStone' => $invoiceStone
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Invoice Stone update error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified invoice stone
     */
    public function destroy($id)
    {
        $invoiceStone = InvoiceStone::findOrFail($id);
        $invoiceStone->delete();

        return redirect()->route('invoice.stone.index')
            ->with('success', 'Invoice Stone berhasil dihapus!');
    }
}