<?php

namespace App\Http\Controllers\Invoice;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\InvoiceHandle;
use App\Models\InvoiceBom;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class InvoiceHandleController extends Controller
{
    /**
     * Display a listing of invoice handles
     */
    public function index()
    {
        $invoiceHandles = InvoiceHandle::with('invoiceBom')
            ->latest()
            ->paginate(10);
        
        return view('invoice.handle.index', compact('invoiceHandles'));
    }

    /**
     * Show the form for creating a new invoice handle
     */
    public function create()
    {
        // Get all fixed invoice BOMs that don't have handle yet
        $invoiceBoms = InvoiceBom::where('is_fixed', true)
            ->whereDoesntHave('invoiceHandle')
            ->latest()
            ->get();
        
        return view('invoice.handle.create', compact('invoiceBoms'));
    }

    /**
     * Store a newly created invoice handle
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'invoice_bom_id' => 'required|exists:invoice_boms,id',
            'handle_data' => 'required|json',
            'grand_total' => 'required|numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();
            
            // Check if invoice already has handle
            $existing = InvoiceHandle::where('invoice_bom_id', $request->invoice_bom_id)->first();
            if ($existing) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invoice BOM ini sudah memiliki data handle!'
                ], 422);
            }

            // Create invoice handle
            $invoiceHandle = InvoiceHandle::create([
                'invoice_bom_id' => $request->invoice_bom_id,
                'handle_data' => $request->handle_data,
                'grand_total' => $request->grand_total
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Invoice Handle berhasil dibuat!',
                'invoiceHandle' => $invoiceHandle
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Invoice Handle creation error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified invoice handle
     */
    public function show($id)
    {
        $invoiceHandle = InvoiceHandle::with('invoiceBom')->findOrFail($id);
        return view('invoice.handle.show', compact('invoiceHandle'));
    }

    /**
     * Show the form for editing the specified invoice handle
     */
    public function edit($id)
    {
        $invoiceHandle = InvoiceHandle::with('invoiceBom')->findOrFail($id);
        
        return view('invoice.handle.edit', compact('invoiceHandle'));
    }

    /**
     * Update the specified invoice handle
     */
    public function update(Request $request, $id)
    {
        $invoiceHandle = InvoiceHandle::findOrFail($id);
        
        $validator = Validator::make($request->all(), [
            'handle_data' => 'required|json',
            'grand_total' => 'required|numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();
            
            $invoiceHandle->update([
                'handle_data' => $request->handle_data,
                'grand_total' => $request->grand_total
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Invoice Handle berhasil diupdate!',
                'invoiceHandle' => $invoiceHandle
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Invoice Handle update error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified invoice handle
     */
    public function destroy($id)
    {
        $invoiceHandle = InvoiceHandle::findOrFail($id);
        $invoiceHandle->delete();

        return redirect()->route('invoice.handle.index')
            ->with('success', 'Invoice Handle berhasil dihapus!');
    }
}