<?php

namespace App\Http\Controllers\Invoice;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\InvoiceFitting;
use App\Models\InvoiceBom;
use App\Models\Fitting;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class InvoiceFittingController extends Controller
{
    /**
     * Display a listing of invoice fitting
     */
    public function index()
    {
        $invoiceFittings = InvoiceFitting::with('invoiceBom')
            ->latest()
            ->paginate(10);
        
        return view('invoice.fitting.index', compact('invoiceFittings'));
    }

    /**
     * Show the form for creating a new invoice fitting
     */
    public function create()
    {
        // Get all fixed invoice BOMs (yang sudah fix)
        $invoiceBoms = InvoiceBom::where('is_fixed', true)
            ->whereDoesntHave('invoiceFitting')
            ->latest()
            ->get();
        
        // Get all fitting grouped by type
        $fittingTypes = Fitting::getTypes();
        $fittingData = [];
        
        foreach ($fittingTypes as $type) {
            $fittingData[$type] = Fitting::byType($type)
                ->orderBy('material_description')
                ->get();
        }
        
        return view('invoice.fitting.create', compact('invoiceBoms', 'fittingTypes', 'fittingData'));
    }

    /**
     * Store a newly created invoice fitting
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'invoice_bom_id' => 'required|exists:invoice_boms,id',
            'fitting_data' => 'required|json',
            'grand_total' => 'required|numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();
            
            // Check if invoice already has fitting
            $existing = InvoiceFitting::where('invoice_bom_id', $request->invoice_bom_id)->first();
            if ($existing) {
                return response()->json([
                    'success' => false,
                    'message' => 'Invoice BOM ini sudah memiliki data fitting!'
                ], 422);
            }

            // Create invoice fitting
            $invoiceFitting = InvoiceFitting::create([
                'invoice_bom_id' => $request->invoice_bom_id,
                'fitting_data' => $request->fitting_data,
                'grand_total' => $request->grand_total
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Invoice Fitting berhasil dibuat!',
                'invoiceFitting' => $invoiceFitting
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Invoice Fitting creation error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified invoice fitting
     */
    public function show($id)
    {
        $invoiceFitting = InvoiceFitting::with('invoiceBom')->findOrFail($id);
        return view('invoice.fitting.show', compact('invoiceFitting'));
    }

    /**
     * Show the form for editing the specified invoice fitting
     */
    public function edit($id)
    {
        $invoiceFitting = InvoiceFitting::with('invoiceBom')->findOrFail($id);
        
        // Get all fitting grouped by type
        $fittingTypes = Fitting::getTypes();
        $fittingData = [];
        
        foreach ($fittingTypes as $type) {
            $fittingData[$type] = Fitting::byType($type)
                ->orderBy('material_description')
                ->get();
        }
        
        return view('invoice.fitting.edit', compact('invoiceFitting', 'fittingTypes', 'fittingData'));
    }

    /**
     * Update the specified invoice fitting
     */
    public function update(Request $request, $id)
    {
        $invoiceFitting = InvoiceFitting::findOrFail($id);
        
        $validator = Validator::make($request->all(), [
            'fitting_data' => 'required|json',
            'grand_total' => 'required|numeric|min:0'
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();
            
            $invoiceFitting->update([
                'fitting_data' => $request->fitting_data,
                'grand_total' => $request->grand_total
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Invoice Fitting berhasil diupdate!',
                'invoiceFitting' => $invoiceFitting
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Invoice Fitting update error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified invoice fitting
     */
    public function destroy($id)
    {
        $invoiceFitting = InvoiceFitting::findOrFail($id);
        $invoiceFitting->delete();

        return redirect()->route('invoice.fitting.index')
            ->with('success', 'Invoice Fitting berhasil dihapus!');
    }

    /**
     * Get fitting materials by type (AJAX)
     */
    public function getFittingByType(Request $request)
    {
        $type = $request->input('type');
        
        $fitting = Fitting::byType($type)
            ->orderBy('material_description')
            ->get();
        
        return response()->json([
            'success' => true,
            'fitting' => $fitting
        ]);
    }
}