<?php

namespace App\Http\Controllers\Invoice;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\InvoiceBom;
use App\Models\ProductPanel;
use App\Models\ProductSolid;
use App\Models\ProductVeneer;
use App\Models\ProductDrawer;
use App\Models\MaterialSolid;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class InvoiceBomController extends Controller
{
    /**
     * Display a listing of invoices
     */
    public function index()
    {
        $invoices = InvoiceBom::latest()->paginate(10);
        return view('invoice.bom.index', compact('invoices'));
    }

    /**
     * Show the form for creating a new invoice
     */
    public function create()
    {
        // Get all product data for dropdowns
        $panels = ProductPanel::orderBy('panel_type')->orderBy('thickness')->get();
        $solids = ProductSolid::orderBy('solid_type')->orderBy('thickness')->get();
        $veneers = ProductVeneer::orderBy('veneer_type')->get();
        $drawers = ProductDrawer::orderBy('material_type')->orderBy('drawer_type')->get();
        $materials = MaterialSolid::orderBy('material_name')->get();

        // Group data for easier access
        $panelTypes = ProductPanel::select('panel_type')->distinct()->pluck('panel_type');
        $solidTypes = ProductSolid::select('solid_type')->distinct()->pluck('solid_type');
        $veneerTypes = ProductVeneer::select('veneer_type')->distinct()->pluck('veneer_type');

        return view('invoice.bom.create', compact(
            'panels',
            'solids',
            'veneers',
            'drawers',
            'materials',
            'panelTypes',
            'solidTypes',
            'veneerTypes'
        ));
    }

    /**
     * Store a newly created invoice
     */
    public function store(Request $request)  // ✅ TAMBAHKAN INI!
    {
        $validator = Validator::make($request->all(), [
        'product_name' => 'required|string',
        'product_category' => 'required|string',
        'product_type' => 'required|string',
        'product_code' => 'required|string',
        'size' => 'required|string',
        'invoice_date' => 'required|date',
        'order_qty' => 'required|integer|min:1',
        'panel_type' => 'nullable|string',
        'solid_type' => 'nullable|string',
        'veneer_face_type' => 'nullable|string',
        'veneer_back_type' => 'nullable|string',
        'box_drawer_type' => 'nullable|string',
        'calculation_data' => 'required',
        'grand_total' => 'required|numeric|min:0',
        'total_m3' => 'nullable|numeric|min:0',        // ✅ BARU
        'harga_per_m3' => 'nullable|numeric|min:0',    // ✅ BARU
        'biaya_produksi' => 'nullable|numeric|min:0',  // ✅ BARU
        'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:11264',
    ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation error',
                'errors' => $validator->errors()
            ], 422);
        }

        try {
            DB::beginTransaction();
            
            // ✅ Handle image upload
            $imagePath = null;
            if ($request->hasFile('image')) {
                $image = $request->file('image');
                $imageName = time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
                
                // Create directory if not exists
                $uploadPath = public_path('uploads/invoice-bom');
                if (!file_exists($uploadPath)) {
                    mkdir($uploadPath, 0775, true);
                }
                
                // Move image to public/uploads/invoice-bom
                $image->move($uploadPath, $imageName);
                $imagePath = 'uploads/invoice-bom/' . $imageName;
            }

            // Parse calculation_data from JSON string
            $calculationData = json_decode($request->calculation_data, true);
            
            // Log for debugging
            Log::info('Calculation Data Received:', $calculationData);

            // Create invoice
$invoice = InvoiceBom::create([
    'product_name' => $request->product_name,
    'product_category' => $request->product_category,
    'product_type' => $request->product_type,
    'product_code' => $request->product_code,
    'size' => $request->size,
    'invoice_date' => $request->invoice_date,
    'order_qty' => $request->order_qty,
    'panel_type' => $request->panel_type,
    'solid_type' => $request->solid_type,
    'veneer_face_type' => $request->veneer_face_type,
    'veneer_back_type' => $request->veneer_back_type,
    'box_drawer_type' => $request->box_drawer_type,
    'image_path' => $imagePath,
    'calculation_data' => $request->calculation_data,
    'grand_total' => $request->grand_total,
    'total_m3' => $request->total_m3,           // ✅ BARU
    'harga_per_m3' => $request->harga_per_m3,   // ✅ BARU
    'biaya_produksi' => $request->biaya_produksi, // ✅ BARU
    'is_fixed' => false,
]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Invoice berhasil dibuat!',
                'invoice' => $invoice
            ]);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Invoice creation error: ' . $e->getMessage());
            Log::error('Stack trace: ' . $e->getTraceAsString());
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified invoice
     */
    public function show($id)
    {
        $invoice = InvoiceBom::findOrFail($id);
        return view('invoice.bom.show', compact('invoice'));
    }

  /**
 * Show the form for editing the specified invoice BOM
 */
public function edit($id)
{
    $invoice = InvoiceBom::findOrFail($id);
    
    // Check if invoice is already fixed
    if ($invoice->is_fixed) {
        return redirect()->route('invoice.bom.show', $id)
            ->with('error', 'Invoice yang sudah di-fix tidak bisa diedit!');
    }
    
    // Get all products for dropdowns
    $panels = ProductPanel::orderBy('panel_type')->orderBy('thickness')->get();
    $solids = ProductSolid::orderBy('solid_type')->orderBy('thickness')->get();
    $veneers = ProductVeneer::orderBy('veneer_type')->get();
    $drawers = ProductDrawer::orderBy('material_type')->orderBy('drawer_type')->get();
    $materials = MaterialSolid::orderBy('material_name')->get();

    // Group data for easier access (DITAMBAHKAN INI)
    $panelTypes = ProductPanel::select('panel_type')->distinct()->pluck('panel_type');
    $solidTypes = ProductSolid::select('solid_type')->distinct()->pluck('solid_type');
    $veneerTypes = ProductVeneer::select('veneer_type')->distinct()->pluck('veneer_type');
    
    // Parse calculation_data JSON
    $calculationData = [];
    if ($invoice->calculation_data) {
        $calculationData = is_string($invoice->calculation_data) 
            ? json_decode($invoice->calculation_data, true)
            : $invoice->calculation_data;
    }
    
    // Parse component_data JSON (if exists)
    $componentData = [];
    if ($invoice->component_data) {
        $componentData = is_string($invoice->component_data)
            ? json_decode($invoice->component_data, true)
            : $invoice->component_data;
    }
    
    return view('invoice.bom.edit', compact(
        'invoice',
        'panels',
        'solids',
        'veneers',
        'drawers',
        'materials',
        'panelTypes',
        'solidTypes',
        'veneerTypes',
        'calculationData',
        'componentData'
    ));
}

/**
 * Update the specified invoice BOM
 */
public function update(Request $request, $id)
{
    $invoice = InvoiceBom::findOrFail($id);
    
    // Check if invoice is already fixed
    if ($invoice->is_fixed) {
        return response()->json([
            'success' => false,
            'message' => 'Invoice yang sudah di-fix tidak bisa diedit!'
        ], 422);
    }
    
    // ✅ VALIDATION
    $validator = Validator::make($request->all(), [
        'product_name' => 'required|string',
        'product_category' => 'required|string',
        'product_type' => 'required|string',
        'product_code' => 'required|string',
        'size' => 'required|string',
        'invoice_date' => 'required|date',
        'order_qty' => 'required|integer|min:1',
        'panel_type' => 'nullable|string',
        'solid_type' => 'nullable|string',
        'veneer_face_type' => 'nullable|string',
        'veneer_back_type' => 'nullable|string',
        'box_drawer_type' => 'nullable|string',
        'calculation_data' => 'required',
        'grand_total' => 'required|numeric|min:0',
        'total_m3' => 'nullable|numeric|min:0',        // ✅ BARU
        'harga_per_m3' => 'nullable|numeric|min:0',    // ✅ BARU
        'biaya_produksi' => 'nullable|numeric|min:0',  // ✅ BARU
        'image' => 'nullable|image|mimes:jpeg,png,jpg,gif,webp|max:11264',
    ]);

    if ($validator->fails()) {
        return response()->json([
            'success' => false,
            'message' => 'Validation error',
            'errors' => $validator->errors()
        ], 422);
    }
    
    try {
        DB::beginTransaction();
        
        $updateData = [
            'product_name' => $request->product_name,
            'product_category' => $request->product_category,
            'product_type' => $request->product_type,
            'product_code' => $request->product_code,
            'size' => $request->size,
            'invoice_date' => $request->invoice_date,
            'order_qty' => $request->order_qty,
            'panel_type' => $request->panel_type,
            'solid_type' => $request->solid_type,
            'veneer_face_type' => $request->veneer_face_type,
            'veneer_back_type' => $request->veneer_back_type,
            'box_drawer_type' => $request->box_drawer_type,
            'calculation_data' => $request->calculation_data,
            'grand_total' => $request->grand_total,
            'total_m3' => $request->total_m3,           // ✅ BARU
            'harga_per_m3' => $request->harga_per_m3,   // ✅ BARU
            'biaya_produksi' => $request->biaya_produksi, // ✅ BARU
        ];
        
        // ✅ HANDLE IMAGE UPLOAD (NAMA FIELD: 'image')
        if ($request->hasFile('image')) {
            // Delete old image if exists
            if ($invoice->image_path && file_exists(public_path($invoice->image_path))) {
                unlink(public_path($invoice->image_path));
            }
            
            $image = $request->file('image');
            $imageName = time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
            
            // Create directory if not exists
            $uploadPath = public_path('uploads/invoice-bom');
            if (!file_exists($uploadPath)) {
                mkdir($uploadPath, 0775, true);
            }
            
            // Move image to public/uploads/invoice-bom
            $image->move($uploadPath, $imageName);
            $updateData['image_path'] = 'uploads/invoice-bom/' . $imageName;
        }
        
        $invoice->update($updateData);
        
        DB::commit();
        
        Log::info('Invoice BOM updated successfully', [
            'invoice_id' => $invoice->id,
            'grand_total' => $invoice->grand_total,
            'has_new_image' => $request->hasFile('image')
        ]);
        
        return response()->json([
            'success' => true,
            'message' => 'Invoice BOM berhasil diupdate!',
            'invoice' => $invoice
        ]);
        
    } catch (\Exception $e) {
        DB::rollBack();
        Log::error('Invoice BOM update error: ' . $e->getMessage());
        Log::error('Stack trace: ' . $e->getTraceAsString());
        
        return response()->json([
            'success' => false,
            'message' => 'Terjadi kesalahan: ' . $e->getMessage()
        ], 500);
    }
}

    /**
     * Fix invoice (lock editing)
     */
    public function fix($id)
    {
        $invoice = InvoiceBom::findOrFail($id);
        $invoice->update(['is_fixed' => true]);

        return redirect()->route('invoice.bom.show', $invoice->id)
            ->with('success', 'Invoice BOM berhasil di-fix!');
    }

    /**
     * Unfix invoice (unlock editing)
     */
    public function unfix($id)
    {
        $invoice = InvoiceBom::findOrFail($id);
        $invoice->update(['is_fixed' => false]);

        return redirect()->route('invoice.bom.show', $invoice->id)
            ->with('success', 'Invoice BOM berhasil di-unfix!');
    }

    /**
     * Delete invoice
     */
    public function destroy($id)
    {
        $invoice = InvoiceBom::findOrFail($id);

        // Delete image if exists
        if ($invoice->image_path && file_exists(public_path($invoice->image_path))) {
            unlink(public_path($invoice->image_path));
        }

        $invoice->delete();

        return redirect()->route('invoice.bom.index')
            ->with('success', 'Invoice BOM berhasil dihapus!');
    }

    /**
     * ✅ Print BOM Invoice
     */
    public function print($id)
    {
        $invoice = InvoiceBom::findOrFail($id);
        $calculationData = json_decode($invoice->calculation_data, true);
        
        // Format data untuk print dengan THICKNESS
        $printData = [];
        $grandTotal = 0;
        
        if ($calculationData) {
            foreach ($calculationData as $componentKey => $componentData) {
                if (isset($componentData['subComponents']) && is_array($componentData['subComponents'])) {
                    foreach ($componentData['subComponents'] as $subKey => $subData) {
                        $printData[] = [
                            'component_name' => $componentData['componentName'] ?? '-',
                            'part_name' => $subData['name'] ?? '-',
                            'thickness' => $subData['thickness'] ?? '-',
                            'width' => $subData['width'] ?? '-',
                            'length' => $subData['length'] ?? '-',
                            'qty' => $subData['qty'] ?? 1,
                            'm2' => $subData['m2'] ?? 0,
                            'material' => $subData['material'] ?? '-',
                            'pattern' => $subData['pattern'] ?? null,
                            'note' => $subData['note'] ?? null,
                            'price' => $subData['price'] ?? $subData['pricePerUnit'] ?? 0,
                            'total' => $subData['total'] ?? 0,
                        ];
                    }
                }
                
                $grandTotal += $componentData['componentTotal'] ?? 0;
            }
        }
        
        return view('invoice.bom.print', compact('invoice', 'printData', 'grandTotal'));
    }

   /**
 * ✅ UPDATED: Print HPP Invoice (WITH BOX ADDED)
 */
public function printHpp($id)
{
    $invoice = InvoiceBom::with([
        'invoiceHardware', 
        'invoiceFitting',
        'invoiceHandle',
        'invoiceStone',
        'invoiceMetal',
        'invoiceUkir',
        'invoiceBox'
    ])->findOrFail($id);
    
    $calculationData = json_decode($invoice->calculation_data, true);
    
    // Calculate BOM Grand Total
    $grandTotal = 0;
    
    if ($calculationData) {
        foreach ($calculationData as $componentData) {
            $grandTotal += $componentData['componentTotal'] ?? 0;
        }
    }
    
    // ✅ Get Hardware Total
    $hardwareTotal = 0;
    if ($invoice->invoiceHardware) {
        $hardwareTotal = $invoice->invoiceHardware->grand_total;
    }
    
    // ✅ Get Fitting Total
    $fittingTotal = 0;
    if ($invoice->invoiceFitting) {
        $fittingTotal = $invoice->invoiceFitting->grand_total;
    }
    
    // ✅ Get Handle Total
    $handleTotal = 0;
    if ($invoice->invoiceHandle) {
        $handleTotal = $invoice->invoiceHandle->grand_total;
    }
    
    // ✅ Get Stone Total
    $stoneTotal = 0;
    if ($invoice->invoiceStone) {
        $stoneTotal = $invoice->invoiceStone->grand_total;
    }
    
    // ✅ Get Metal Total
    $metalTotal = 0;
    if ($invoice->invoiceMetal) {
        $metalTotal = $invoice->invoiceMetal->grand_total;
    }
    
    // ✅ Get Ukir/Bubut Total
    $ukirTotal = 0;
    if ($invoice->invoiceUkir) {
        $ukirTotal = $invoice->invoiceUkir->grand_total;
    }
    
    // ✅ NEW: Get Box Total
    $boxTotal = 0;
    if ($invoice->invoiceBox) {
        $boxTotal = $invoice->invoiceBox->grand_total;
    }
    
    // ✅ Get Painting Totals
    $materialPainting = $invoice->getTotalMaterialPainting();
    $biayaProduksiPainting = $invoice->getTotalBiayaProduksiPainting();
    
    // ✅ Biaya Produksi Packing (hardcoded untuk sementara, bisa diubah jadi dynamic)
    $biayaProduksiPacking = 0; // Bisa diisi manual atau dari database
    
    // ✅ HPP Data Structure (UPDATED STRUCTURE WITH BOX)
    $hppData = [
        'material_wood_panel' => $grandTotal,
        'hardware' => $hardwareTotal,
        'fitting' => $fittingTotal,
        'handle' => $handleTotal,
        'stone' => $stoneTotal,
        'metal' => $metalTotal,
        'ukir_bubut' => $ukirTotal,
        'box' => $boxTotal,
        'biaya_produksi_painting' => $biayaProduksiPainting,
        'material_painting' => $materialPainting,
        'biaya_produksi_packing' => $biayaProduksiPacking,
    ];
    
    // ✅ UPDATED CALCULATIONS
    // Total HPP Unfinished = Material Wood & Panel + Hardware + Fitting + Handle + Stone + Metal + Ukir/Bubut + Box + Biaya Produksi Painting
    $hppData['total_hpp_unfinished'] = $hppData['material_wood_panel'] 
        + $hppData['hardware'] 
        + $hppData['fitting'] 
        + $hppData['handle'] 
        + $hppData['stone']
        + $hppData['metal']
        + $hppData['ukir_bubut']
        + $hppData['box']
        + $hppData['biaya_produksi_painting'];
    
    // Total HPP Include Finishing = Total HPP Unfinished + Material Painting + Biaya Produksi Packing
    $hppData['total_hpp_finished'] = $hppData['total_hpp_unfinished'] 
        + $hppData['material_painting']
        + $hppData['biaya_produksi_packing'];
    
    return view('invoice.bom.print-hpp', compact('invoice', 'hppData'));
}

    /**
     * Save draft
     */
    public function saveDraft(Request $request)
{
    try {
        $draftData = $request->all();
        
        // Add timestamp
        $draftData['saved_at'] = now()->toDateTimeString();
        
        // Save to session
        session(['invoice_bom_draft' => $draftData]);
        
        Log::info('Draft saved successfully', [
            'draft_size' => strlen(json_encode($draftData)),
            'grand_total' => $draftData['grand_total'] ?? 0
        ]);
        
        return response()->json([
            'success' => true, 
            'message' => 'Draft tersimpan!',
            'saved_at' => $draftData['saved_at']
        ]);
        
    } catch (\Exception $e) {
        Log::error('Draft save error: ' . $e->getMessage());
        
        return response()->json([
            'success' => false, 
            'message' => 'Gagal menyimpan draft: ' . $e->getMessage()
        ], 500);
    }
}

   /**
 * ✅ Save painting data (ALLOW EMPTY)
 */
public function savePainting(Request $request, $id)
{
    try {
        Log::info('Save painting request received', [
            'invoice_id' => $id,
            'request_data' => $request->all()
        ]);
        
        $invoice = InvoiceBom::findOrFail($id);
        
        // ✅ UPDATED: Allow empty array (user can clear painting data)
        $paintingData = $request->input('painting_data', []);
        
        // Validate that it's an array
        if (!is_array($paintingData)) {
            return response()->json([
                'success' => false,
                'message' => 'Painting data must be an array'
            ], 422);
        }
        
        // Save painting data (can be empty)
        $invoice->update([
            'painting_data' => $paintingData
        ]);
        
        Log::info('Painting data saved successfully', [
            'invoice_id' => $id,
            'painting_data' => $paintingData,
            'count' => count($paintingData)
        ]);
        
        return response()->json([
            'success' => true,
            'message' => count($paintingData) > 0 
                ? 'Painting data berhasil disimpan!' 
                : 'Painting data berhasil dihapus!',
            'painting_data' => $invoice->painting_data,
            'total_material_painting' => $invoice->getTotalMaterialPainting(),
            'total_biaya_produksi_painting' => $invoice->getTotalBiayaProduksiPainting()
        ]);
        
    } catch (\Illuminate\Database\Eloquent\ModelNotFoundException $e) {
        Log::error('Invoice not found', [
            'invoice_id' => $id
        ]);
        
        return response()->json([
            'success' => false,
            'message' => 'Invoice tidak ditemukan!'
        ], 404);
        
    } catch (\Exception $e) {
        Log::error('Save painting data error', [
            'invoice_id' => $id,
            'error' => $e->getMessage(),
            'trace' => $e->getTraceAsString()
        ]);
        
        return response()->json([
            'success' => false,
            'message' => 'Terjadi kesalahan: ' . $e->getMessage()
        ], 500);
    }
}

    /**
     * ✅ NEW: Get painting data
     */
    public function getPainting($id)
    {
        try {
            $invoice = InvoiceBom::findOrFail($id);
            
            return response()->json([
                'success' => true,
                'painting_data' => $invoice->painting_data ?? [],
                'total_material_painting' => $invoice->getTotalMaterialPainting(),
                'total_biaya_produksi_painting' => $invoice->getTotalBiayaProduksiPainting()
            ]);
            
        } catch (\Exception $e) {
            Log::error('Get painting data error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get draft
     */
    public function getDraft()
{
    try {
        $draft = session('invoice_bom_draft', null);
        
        if ($draft) {
            Log::info('Draft loaded successfully', [
                'saved_at' => $draft['saved_at'] ?? 'unknown'
            ]);
        }
        
        return response()->json([
            'success' => true, 
            'draft' => $draft
        ]);
        
    } catch (\Exception $e) {
        Log::error('Draft load error: ' . $e->getMessage());
        
        return response()->json([
            'success' => false, 
            'message' => 'Gagal memuat draft: ' . $e->getMessage()
        ], 500);
    }
}

    /**
     * Clear draft
     */
    public function clearDraft()
{
    try {
        session()->forget('invoice_bom_draft');
        
        Log::info('Draft cleared successfully');
        
        return response()->json([
            'success' => true, 
            'message' => 'Draft dihapus!'
        ]);
        
    } catch (\Exception $e) {
        Log::error('Draft clear error: ' . $e->getMessage());
        
        return response()->json([
            'success' => false, 
            'message' => 'Gagal menghapus draft: ' . $e->getMessage()
        ], 500);
    }
}
}