<?php
// Start session
session_start();

// Include database connection
require_once "../config/connection.php";

// Set timezone to Indonesia (WIB/WITA/WIT - adjust according to your location)
date_default_timezone_set('Asia/Jakarta');

// Function to detect mobile device with enhanced detection
function isMobileDevice() {
    if (!isset($_SERVER['HTTP_USER_AGENT'])) return false;
    
    $userAgent = strtolower($_SERVER['HTTP_USER_AGENT']);
    $mobileKeywords = [
        'mobile', 'android', 'iphone', 'ipad', 'ipod', 'blackberry', 
        'windows phone', 'opera mini', 'iemobile', 'webos', 'nokia',
        'samsung', 'htc', 'lg', 'motorola', 'sony', 'xiaomi'
    ];
    
    foreach ($mobileKeywords as $keyword) {
        if (strpos($userAgent, $keyword) !== false) {
            return true;
        }
    }
    
    // Check for mobile-specific patterns
    if (preg_match('/(up\.browser|up\.link|mmp|symbian|smartphone|midp|wap|phone|windows ce|pda|mobile|mini|palm|netfront)/i', $userAgent)) {
        return true;
    }
    
    // Check screen width if available (modern approach)
    if (isset($_SERVER['HTTP_X_MOBILE']) || isset($_SERVER['HTTP_X_WAP_PROFILE'])) {
        return true;
    }
    
    return false;
}

// Function to send JSON response with enhanced mobile-optimized formatting
function sendJSONResponse($success, $message, $item = null) {
    header('Content-Type: application/json; charset=utf-8');
    header('Cache-Control: no-cache, must-revalidate');
    header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');
    
    // Detect mobile device
    $isMobile = isMobileDevice();
    
    // Optimize message length and formatting for mobile displays
    if ($isMobile) {
        if (!$success && strlen($message) > 120) {
            // Truncate long error messages for mobile
            $message = substr($message, 0, 117) . '...';
        }
        
        // Add appropriate emojis for mobile users
        if ($success) {
            $message = "✅ " . $message;
        } else {
            $message = "❌ " . $message;
        }
    }
    
    $response = [
        'success' => $success,
        'message' => $message,
        'item' => $item,
        'mobile' => $isMobile,
        'timestamp' => date('Y-m-d H:i:s'),
        'timezone' => 'Asia/Jakarta'
    ];
    
    echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    exit();
}

// Redirect if not borrower user
if (!is_logged_in() || !is_user_type('borrower')) {
    sendJSONResponse(false, 'Akses tidak diizinkan. Silakan login kembali.');
}

// Enhanced parameter validation
if (!isset($_GET['kode_barang']) || empty($_GET['kode_barang'])) {
    sendJSONResponse(false, 'Kode barang harus diisi');
}

$kode_barang = trim($_GET['kode_barang']);

// Enhanced QR data extraction with better JSON handling
try {
    // Try to parse as JSON first (in case QR code contains JSON data)
    $qr_data = json_decode($kode_barang, true);
    if (json_last_error() === JSON_ERROR_NONE && is_array($qr_data)) {
        if (isset($qr_data['kode_barang'])) {
            $kode_barang = trim($qr_data['kode_barang']);
        } elseif (isset($qr_data['code'])) {
            $kode_barang = trim($qr_data['code']);
        } elseif (isset($qr_data['item_code'])) {
            $kode_barang = trim($qr_data['item_code']);
        }
    }
} catch (Exception $e) {
    // If JSON parsing fails, use the original value
    error_log("QR JSON parsing failed: " . $e->getMessage());
}

// Enhanced kode_barang format validation
if (!preg_match('/^CP\.\d{4}$/', $kode_barang)) {
    $isMobile = isMobileDevice();
    if ($isMobile) {
        sendJSONResponse(false, 'Format QR Code salah. Harusnya: CP.#### (contoh: CP.0001)');
    } else {
        sendJSONResponse(false, 'Format QR Code tidak valid. Expected: CP.#### (contoh: CP.0001)');
    }
}

try {
    // Enhanced item search with comprehensive data retrieval
    $stmt = $conn->prepare("
        SELECT 
            csi.id,
            csi.kode_barang,
            csi.status,
            csi.lokasi,
            csi.rack_no,
            csi.tanggal_pembuatan,
            csi.tanggal_kedaluwarsa,
            csi.keterangan,
            csi.color_id,
            csi.component_id,
            csi.stock_number,
            cc.component_name,
            cc.description as component_description,
            mcp.code_color,
            mcp.deskripsi as color_description,
            mcp.status as color_panel_status,
            b.nama_buyer,
            b.kode_buyer,
            col.nama_collection,
            cpc.quantity as total_quantity,
            cpc.available_quantity
        FROM color_stock_items csi
        JOIN color_components cc ON csi.component_id = cc.id
        JOIN master_color_panel mcp ON csi.color_id = mcp.id
        JOIN buyers b ON mcp.buyer_id = b.id
        JOIN collections col ON mcp.collection_id = col.id
        LEFT JOIN color_panel_components cpc ON (csi.color_id = cpc.color_id AND csi.component_id = cpc.component_id)
        WHERE csi.kode_barang = ?
        LIMIT 1
    ");
    
    $stmt->bind_param("s", $kode_barang);
    $stmt->execute();
    $result = $stmt->get_result();
    
    if ($result->num_rows === 0) {
        $stmt->close();
        
        // Enhanced "not found" logging for debugging
        log_activity($conn, $_SESSION['user_id'], 'borrower', 'QR Code Not Found', 
            "borrower attempted to scan non-existent QR code: {$kode_barang}");
        
        sendJSONResponse(false, 'QR Code tidak ditemukan di database color panel');
    }
    
    $item = $result->fetch_assoc();
    $stmt->close();
    
    // Enhanced expiry date checking with automatic status update
    if (!empty($item['tanggal_kedaluwarsa'])) {
        $timezone = new DateTimeZone('Asia/Jakarta');
        $current_date = new DateTime('now', $timezone);
        $expiry_date = new DateTime($item['tanggal_kedaluwarsa'], $timezone);
        
        if ($current_date > $expiry_date) {
            // Auto-update expired items
            $update_expired_stmt = $conn->prepare("UPDATE color_stock_items SET status = 'Expired' WHERE id = ?");
            $update_expired_stmt->bind_param("i", $item['id']);
            $update_expired_stmt->execute();
            $update_expired_stmt->close();
            
            // Log the expiry update
            log_activity($conn, $_SESSION['user_id'], 'borrower', 'Auto Expire Update', 
                "Item {$kode_barang} automatically updated to Expired status");
            
            sendJSONResponse(false, "QR Code {$kode_barang} kedaluwarsa pada " . $expiry_date->format('d-m-Y'));
        }
    }
    
    // Enhanced borrowing eligibility check
    $restricted_statuses = ['Expired', 'Hilang', 'Dipinjam'];
    if (in_array($item['status'], $restricted_statuses)) {
        if ($item['status'] === 'Dipinjam') {
            // Enhanced borrower info retrieval with more details
            $borrower_stmt = $conn->prepare("
                SELECT 
                    bu.nama_borrower,
                    bu.lokasi_posisi,
                    bu.nik,
                    st.tanggal_transaksi,
                    st.tanggal_pengembalian,
                    st.reservation_number,
                    TIMESTAMPDIFF(HOUR, NOW(), st.tanggal_pengembalian) as hours_remaining
                FROM transaction_items ti
                JOIN stock_transactions st ON ti.transaction_id = st.id
                JOIN borrower_users bu ON st.borrower_id = bu.id
                WHERE ti.stock_item_id = ? 
                AND st.status = 'Disetujui'
                AND st.status_peminjaman = 'Dipinjam'
                AND ti.dikembalikan = 0
                ORDER BY st.tanggal_transaksi DESC
                LIMIT 1
            ");
            
            $borrower_stmt->bind_param("i", $item['id']);
            $borrower_stmt->execute();
            $borrower_result = $borrower_stmt->get_result();
            
            if ($borrower_result->num_rows > 0) {
                $borrower_data = $borrower_result->fetch_assoc();
                
                try {
                    $pengembalian_date = new DateTime($borrower_data['tanggal_pengembalian'], new DateTimeZone('Asia/Jakarta'));
                    $return_date_formatted = $pengembalian_date->format('d-m-Y H:i');
                    
                    $hours_remaining = $borrower_data['hours_remaining'];
                    $isMobile = isMobileDevice();
                    
                    if ($isMobile) {
                        // Compact message for mobile
                        $error_message = "Sedang dipinjam oleh {$borrower_data['nama_borrower']} ({$borrower_data['lokasi_posisi']}). Kembali: {$return_date_formatted}";
                        if ($hours_remaining > 0) {
                            $error_message .= " (~{$hours_remaining}h lagi)";
                        }
                    } else {
                        // Detailed message for desktop
                        $error_message = "QR Code {$kode_barang} sedang dipinjam oleh {$borrower_data['nama_borrower']} dari {$borrower_data['lokasi_posisi']}. Jadwal pengembalian: {$return_date_formatted}";
                        if ($hours_remaining > 0) {
                            $error_message .= " (sekitar {$hours_remaining} jam lagi)";
                        }
                        $error_message .= ". Reservasi: {$borrower_data['reservation_number']}";
                    }
                    
                    sendJSONResponse(false, $error_message);
                } catch (Exception $e) {
                    sendJSONResponse(false, "QR Code {$kode_barang} sedang dipinjam oleh borrower lain");
                }
            } else {
                sendJSONResponse(false, "QR Code {$kode_barang} sedang dipinjam dan tidak dapat dipinjam kembali");
            }
            $borrower_stmt->close();
        } else {
            $isMobile = isMobileDevice();
            if ($isMobile) {
                sendJSONResponse(false, "QR Code tidak dapat dipinjam (Status: {$item['status']})");
            } else {
                sendJSONResponse(false, "QR Code {$kode_barang} tidak dapat dipinjam (Status: {$item['status']})");
            }
        }
    }
    
    // Enhanced expiry warning calculation
    $expiry_warning = '';
    $expiry_status = 'none';
    
    if (!empty($item['tanggal_kedaluwarsa'])) {
        $timezone = new DateTimeZone('Asia/Jakarta');
        $current_date = new DateTime('now', $timezone);
        $expiry_date = new DateTime($item['tanggal_kedaluwarsa'], $timezone);
        $days_until_expiry = $current_date->diff($expiry_date)->days;
        
        if ($expiry_date > $current_date) {
            if ($days_until_expiry <= 7) {
                $expiry_status = 'critical'; // Less than a week
                $expiry_warning = " (⚠️ Kedaluwarsa dalam {$days_until_expiry} hari)";
            } elseif ($days_until_expiry <= 30) {
                $expiry_status = 'warning'; // Less than a month
                $expiry_warning = " (Kedaluwarsa dalam {$days_until_expiry} hari)";
            } else {
                $expiry_status = 'safe';
            }
        }
    }
    
    // Enhanced date formatting with error handling
    $tanggal_pembuatan_formatted = '';
    $tanggal_kedaluwarsa_formatted = '';
    
    if (!empty($item['tanggal_pembuatan'])) {
        try {
            $pembuatan_date = new DateTime($item['tanggal_pembuatan']);
            $tanggal_pembuatan_formatted = $pembuatan_date->format('d-m-Y');
        } catch (Exception $e) {
            $tanggal_pembuatan_formatted = $item['tanggal_pembuatan'];
        }
    }
    
    if (!empty($item['tanggal_kedaluwarsa'])) {
        try {
            $kedaluwarsa_date = new DateTime($item['tanggal_kedaluwarsa']);
            $tanggal_kedaluwarsa_formatted = $kedaluwarsa_date->format('d-m-Y');
        } catch (Exception $e) {
            $tanggal_kedaluwarsa_formatted = $item['tanggal_kedaluwarsa'];
        }
    }
    
    // Prepare comprehensive item data for response
    $item_data = [
        'id' => (int)$item['id'],
        'kode_barang' => $item['kode_barang'],
        'status' => $item['status'],
        'lokasi' => $item['lokasi'],
        'rack_no' => $item['rack_no'] ?: '-',
        'stock_number' => (int)$item['stock_number'],
        'component_name' => $item['component_name'],
        'component_description' => $item['component_description'] ?: '',
        'color_code' => $item['code_color'],
        'color_description' => $item['color_description'] ?: '',
        'color_panel_status' => $item['color_panel_status'],
        'buyer_name' => $item['nama_buyer'],
        'buyer_code' => $item['kode_buyer'],
        'collection_name' => $item['nama_collection'],
        'tanggal_pembuatan' => $tanggal_pembuatan_formatted,
        'tanggal_kedaluwarsa' => $tanggal_kedaluwarsa_formatted,
        'keterangan' => $item['keterangan'] ?: '-',
        'color_id' => (int)$item['color_id'],
        'component_id' => (int)$item['component_id'],
        'total_quantity' => (int)$item['total_quantity'],
        'available_quantity' => (int)$item['available_quantity'],
        'expiry_status' => $expiry_status,
        'expiry_warning' => $expiry_warning
    ];
    
    // Enhanced mobile detection and optimization
    $isMobile = isMobileDevice();
    
    // Generate mobile-optimized success message with enhanced formatting
    if ($isMobile) {
        // Ultra-compact message for mobile with key info
        $success_message = "{$item['nama_buyer']} - {$item['code_color']} ({$item['component_name']})";
        
        // Add compact status indicators
        if ($item['status'] === 'Rusak') {
            $success_message .= " [RUSAK]";
        } elseif ($item['status'] === 'Renew') {
            $success_message .= " [RENEW]";
        }
        
        // Add rack info if available
        if (!empty($item['rack_no'])) {
            $success_message .= " - Rack: {$item['rack_no']}";
        }
        
        // Add critical expiry warning for mobile
        if ($expiry_status === 'critical') {
            $success_message .= $expiry_warning;
        }
        
        // Add mobile-specific enhancements
        $item_data['mobile_optimized'] = true;
        $item_data['mobile_display'] = [
            'short_name' => $item['kode_buyer'] . " " . $item['code_color'],
            'component_short' => substr($item['component_name'], 0, 15) . (strlen($item['component_name']) > 15 ? '...' : ''),
            'status_emoji' => $item['status'] === 'Tersedia' ? '✅' : ($item['status'] === 'Rusak' ? '⚠️' : '❌'),
            'priority_info' => [
                'buyer' => $item['nama_buyer'],
                'color' => $item['code_color'],
                'component' => $item['component_name'],
                'rack' => $item['rack_no'] ?: 'N/A'
            ]
        ];
        
    } else {
        // Comprehensive message for desktop
        $success_message = "QR Code berhasil discan! {$item['kode_barang']}: {$item['nama_buyer']} - {$item['code_color']} ({$item['component_name']})";
        
        // Add detailed status information
        if ($item['status'] === 'Rusak') {
            $success_message .= " - Status: RUSAK (kualitas warna mungkin kurang bagus)";
        } elseif ($item['status'] === 'Renew') {
            $success_message .= " - Status: RENEW (perlu diperbaharui)";
        }
        
        // Add comprehensive location info
        if (!empty($item['rack_no'])) {
            $success_message .= " - Lokasi: {$item['lokasi']}, Rack: {$item['rack_no']}";
        } else {
            $success_message .= " - Lokasi: {$item['lokasi']}";
        }
        
        // Add stock info
        if ($item['available_quantity'] > 0) {
            $success_message .= " - Tersedia: {$item['available_quantity']} dari {$item['total_quantity']}";
        }
        
        // Add expiry warning
        $success_message .= $expiry_warning;
    }
    
    // Enhanced logging with device info and performance metrics
    $device_info = $isMobile ? '[Mobile QR Scan]' : '[Desktop QR Scan]';
    $scan_details = "scanned QR code: {$kode_barang} - Found: {$item['nama_buyer']} {$item['code_color']} ({$item['component_name']}) {$device_info}";
    
    // Add performance metrics to logging
    $execution_time = microtime(true) - $_SERVER['REQUEST_TIME_FLOAT'];
    $scan_details .= " [Execution: " . round($execution_time * 1000, 2) . "ms]";
    
    log_activity($conn, $_SESSION['user_id'], 'borrower', 'QR Code Scan', $scan_details);
    
    sendJSONResponse(true, $success_message, $item_data);
    
} catch (mysqli_sql_exception $e) {
    error_log("Database error in validate_code_barang.php: " . $e->getMessage());
    
    $isMobile = isMobileDevice();
    $error_message = $isMobile 
        ? 'Error database. Coba scan ulang.' 
        : 'Terjadi kesalahan database saat memproses QR code. Silakan scan ulang.';
    
    sendJSONResponse(false, $error_message);
    
} catch (Exception $e) {
    error_log("General error in validate_code_barang.php (QR Scanner): " . $e->getMessage());
    
    // Enhanced error logging with context
    $error_context = [
        'kode_barang' => $kode_barang ?? 'unknown',
        'user_id' => $_SESSION['user_id'] ?? 'unknown',
        'user_agent' => substr($_SERVER['HTTP_USER_AGENT'] ?? 'unknown', 0, 100),
        'ip_address' => $_SERVER['REMOTE_ADDR'] ?? 'unknown'
    ];
    
    error_log("QR validation context: " . json_encode($error_context));
    
    // Mobile-friendly error message with actionable advice
    $isMobile = isMobileDevice();
    if ($isMobile) {
        $error_message = 'Error sistem. Pastikan koneksi stabil dan scan ulang.';
    } else {
        $error_message = 'Terjadi kesalahan sistem saat memproses QR code. Pastikan koneksi internet stabil dan silakan scan ulang.';
    }
    
    sendJSONResponse(false, $error_message);
}
?>